(*  Title:      Provers/simplifier.ML
    ID:         $Id: simplifier.ML,v 1.47 1998/06/25 13:33:30 wenzelm Exp $
    Author:     Tobias Nipkow and Markus Wenzel, TU Muenchen

Generic simplifier, suitable for most logics.  See Pure/thm.ML for the
actual meta-level rewriting engine.
*)

infix 4
  setsubgoaler setloop addloop delloop setSSolver addSSolver setSolver
  addSolver addsimps delsimps addeqcongs deleqcongs
  setmksimps setmkeqTrue setmksym settermless addsimprocs delsimprocs;

signature BASIC_SIMPLIFIER =
sig
  type simproc
  val mk_simproc: string -> cterm list
    -> (Sign.sg -> thm list -> term -> thm option) -> simproc
  type simpset
  val empty_ss: simpset
  val rep_ss: simpset ->
   {mss: meta_simpset,
    subgoal_tac:        simpset -> int -> tactic,
    loop_tacs:          (string * (int -> tactic))list,
           finish_tac: thm list -> int -> tactic,
    unsafe_finish_tac: thm list -> int -> tactic};
  val print_ss: simpset -> unit
  val print_simpset: theory -> unit
  val setsubgoaler: simpset *  (simpset -> int -> tactic) -> simpset
  val setloop:      simpset *             (int -> tactic) -> simpset
  val addloop:      simpset *  (string * (int -> tactic)) -> simpset
  val delloop:      simpset *   string                    -> simpset
  val setSSolver:   simpset * (thm list -> int -> tactic) -> simpset
  val addSSolver:   simpset * (thm list -> int -> tactic) -> simpset
  val setSolver:    simpset * (thm list -> int -> tactic) -> simpset
  val addSolver:    simpset * (thm list -> int -> tactic) -> simpset
  val setmksimps:   simpset * (thm -> thm list) -> simpset
  val setmkeqTrue:  simpset * (thm -> thm option) -> simpset
  val setmksym:     simpset * (thm -> thm option) -> simpset
  val settermless:  simpset * (term * term -> bool) -> simpset
  val addsimps:     simpset * thm list -> simpset
  val delsimps:     simpset * thm list -> simpset
  val addeqcongs:   simpset * thm list -> simpset
  val deleqcongs:   simpset * thm list -> simpset
  val addsimprocs:  simpset * simproc list -> simpset
  val delsimprocs:  simpset * simproc list -> simpset
  val merge_ss:     simpset * simpset -> simpset
  val prems_of_ss:  simpset -> thm list
  val simpset_ref_of_sg: Sign.sg -> simpset ref
  val simpset_ref_of: theory -> simpset ref
  val simpset_of_sg: Sign.sg -> simpset
  val simpset_of: theory -> simpset
  val SIMPSET: (simpset -> tactic) -> tactic
  val SIMPSET': (simpset -> 'a -> tactic) -> 'a -> tactic
  val simpset: unit -> simpset
  val simpset_ref: unit -> simpset ref
  val Addsimps: thm list -> unit
  val Delsimps: thm list -> unit
  val Addsimprocs: simproc list -> unit
  val Delsimprocs: simproc list -> unit
  val               simp_tac: simpset -> int -> tactic
  val           asm_simp_tac: simpset -> int -> tactic
  val          full_simp_tac: simpset -> int -> tactic
  val        asm_lr_simp_tac: simpset -> int -> tactic
  val      asm_full_simp_tac: simpset -> int -> tactic
  val safe_asm_full_simp_tac: simpset -> int -> tactic
  val               Simp_tac:            int -> tactic
  val           Asm_simp_tac:            int -> tactic
  val          Full_simp_tac:            int -> tactic
  val        Asm_lr_simp_tac:            int -> tactic
  val      Asm_full_simp_tac:            int -> tactic
  val          simplify: simpset -> thm -> thm
  val      asm_simplify: simpset -> thm -> thm
  val     full_simplify: simpset -> thm -> thm
  val asm_full_simplify: simpset -> thm -> thm
end;

signature SIMPLIFIER =
sig
  include BASIC_SIMPLIFIER
  val          rewrite: simpset -> cterm -> thm
  val      asm_rewrite: simpset -> cterm -> thm
  val     full_rewrite: simpset -> cterm -> thm
  val asm_full_rewrite: simpset -> cterm -> thm
  val setup: (theory -> theory) list
  val get_local_simpset: local_theory -> simpset
  val put_local_simpset: simpset -> local_theory -> local_theory
  val simp_add: tag
  val simp_del: tag
  val simp_add_global: theory attribute
  val simp_del_global: theory attribute
  val simp_add_local: local_theory attribute
  val simp_del_local: local_theory attribute
end;

structure Simplifier: SIMPLIFIER =
struct


(** simplification procedures **)

(* datatype simproc *)

datatype simproc =
  Simproc of string * cterm list * (Sign.sg -> thm list -> term -> thm option) * stamp;

fun mk_simproc name lhss proc =
  Simproc (name, map (Thm.cterm_fun Logic.varify) lhss, proc, stamp ());

fun rep_simproc (Simproc args) = args;



(** simplification sets **)

(* type simpset *)

datatype simpset =
  Simpset of {
    mss: meta_simpset,
    subgoal_tac:        simpset -> int -> tactic,
    loop_tacs:          (string * (int -> tactic))list,
           finish_tac: thm list -> int -> tactic,
    unsafe_finish_tac: thm list -> int -> tactic};

fun make_ss (mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac) =
  Simpset {mss = mss, subgoal_tac = subgoal_tac, loop_tacs = loop_tacs,
    finish_tac = finish_tac, unsafe_finish_tac = unsafe_finish_tac};

val empty_ss =
  let val mss = Thm.set_mk_sym(Thm.empty_mss, Some o symmetric_fun)
  in make_ss (mss, K (K no_tac), [], K (K no_tac), K (K no_tac)) end;

fun rep_ss (Simpset args) = args;
fun prems_of_ss (Simpset {mss, ...}) = Thm.prems_of_mss mss;


(* print simpsets *)

fun print_ss ss =
  let
    val Simpset {mss, ...} = ss;
    val {simps, procs, congs} = Thm.dest_mss mss;

    val pretty_thms = map Display.pretty_thm;
    fun pretty_proc (name, lhss) =
      Pretty.big_list (name ^ ":") (map Display.pretty_cterm lhss);
  in
    Pretty.writeln (Pretty.big_list "simplification rules:" (pretty_thms simps));
    Pretty.writeln (Pretty.big_list "simplification procedures:" (map pretty_proc procs));
    Pretty.writeln (Pretty.big_list "congruences:" (pretty_thms congs))
  end;


(* extend simpsets *)

fun (Simpset {mss, subgoal_tac = _, loop_tacs, finish_tac, unsafe_finish_tac})
    setsubgoaler subgoal_tac =
  make_ss (mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs = _, finish_tac, unsafe_finish_tac})
    setloop tac =
  make_ss (mss, subgoal_tac, [("",tac)], finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    addloop tac = make_ss (mss, subgoal_tac, 
	(case assoc_string (loop_tacs,(fst tac)) of None => () | Some x => 
	 warning ("overwriting looper "^fst tac); overwrite(loop_tacs,tac)),
           finish_tac, unsafe_finish_tac);

fun (ss as Simpset {mss,subgoal_tac,loop_tacs,finish_tac,unsafe_finish_tac})
    delloop name =
  let val (del,rest) = partition (fn (n,_) => n=name) loop_tacs
  in if null del then (warning ("No such looper in simpset: " ^ name); ss)
     else make_ss (mss, subgoal_tac, rest, finish_tac, unsafe_finish_tac)
  end;

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac = _, unsafe_finish_tac})
    setSSolver finish_tac =
  make_ss (mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    addSSolver tac =
  make_ss (mss, subgoal_tac, loop_tacs, fn hyps => finish_tac hyps ORELSE' tac hyps,
    unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac = _})
    setSolver unsafe_finish_tac =
  make_ss (mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    addSolver tac =
  make_ss (mss, subgoal_tac, loop_tacs, finish_tac,
    fn hyps => unsafe_finish_tac hyps ORELSE' tac hyps);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    setmksimps mk_simps =
  make_ss (Thm.set_mk_rews (mss, map (Thm.strip_shyps o Drule.zero_var_indexes) o mk_simps),
    subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    setmkeqTrue mk_eq_True =
  make_ss (Thm.set_mk_eq_True (mss, mk_eq_True),
    subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    setmksym mksym =
  make_ss (Thm.set_mk_sym (mss, mksym),
    subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs,  finish_tac, unsafe_finish_tac})
    settermless termless =
  make_ss (Thm.set_termless (mss, termless), subgoal_tac, loop_tacs,
    finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    addsimps rews =
  make_ss (Thm.add_simps (mss, rews), subgoal_tac, loop_tacs,
           finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    delsimps rews =
  make_ss (Thm.del_simps (mss, rews), subgoal_tac, loop_tacs,
           finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    addeqcongs newcongs =
  make_ss (Thm.add_congs (mss, newcongs), subgoal_tac, loop_tacs,
    finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    deleqcongs oldcongs =
  make_ss (Thm.del_congs (mss, oldcongs), subgoal_tac, loop_tacs,
    finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    addsimprocs simprocs =
  make_ss
    (Thm.add_simprocs (mss, map rep_simproc simprocs),
      subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);

fun (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac})
    delsimprocs simprocs =
  make_ss
    (Thm.del_simprocs (mss, map rep_simproc simprocs),
      subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);


(* merge simpsets *)	(*NOTE: ignores tactics of 2nd simpset (except loopers)*)

fun merge_ss
   (Simpset {mss = mss1, loop_tacs = loop_tacs1, subgoal_tac, finish_tac, unsafe_finish_tac},
    Simpset {mss = mss2, loop_tacs = loop_tacs2, ...}) =
  make_ss (Thm.merge_mss (mss1, mss2), subgoal_tac,
    merge_alists loop_tacs1 loop_tacs2, finish_tac, unsafe_finish_tac);



(** global and local simpset data **)

val simpsetN = "Provers/simpset";


(* data kind 'Provers/simpset' *)

structure SimpsetDataArgs =
struct
  val name = simpsetN;
  type T = simpset ref;

  val empty = ref empty_ss;
  fun prep_ext (ref ss) = (ref ss): T;		  (*create new reference!*)
  fun merge (ref ss1, ref ss2) = ref (merge_ss (ss1, ss2));
  fun print _ (ref ss) = print_ss ss;
end;

structure SimpsetData = TheoryDataFun(SimpsetDataArgs);
val print_simpset = SimpsetData.print;
val simpset_ref_of_sg = SimpsetData.get_sg;
val simpset_ref_of = SimpsetData.get;


(* access global simpset *)

val simpset_of_sg = ! o simpset_ref_of_sg;
val simpset_of = simpset_of_sg o sign_of;

fun SIMPSET tacf state = tacf (simpset_of_sg (sign_of_thm state)) state;
fun SIMPSET' tacf i state = tacf (simpset_of_sg (sign_of_thm state)) i state;

val simpset = simpset_of o Context.the_context;
val simpset_ref = simpset_ref_of_sg o sign_of o Context.the_context;


(* change global simpset *)

fun change_simpset f x = simpset_ref () := (f (simpset (), x));

val Addsimps = change_simpset (op addsimps);
val Delsimps = change_simpset (op delsimps);
val Addsimprocs = change_simpset (op addsimprocs);
val Delsimprocs = change_simpset (op delsimprocs);


(* local simpset *)

exception SimpsetLocal of simpset;
val simpset_localK = Object.kind simpsetN;

fun get_local_simpset (thy, data) =
  (case Symtab.lookup (data, simpsetN) of
    Some (SimpsetLocal ss) => ss
  | None => simpset_of thy
  | _ => Object.kind_error simpset_localK);

fun put_local_simpset ss (thy, data) =
  (thy, Symtab.update ((simpsetN, SimpsetLocal ss), data));



(** simplifier attributes **)

(* tags *)

val simpN = "simp";
val simp_addN = "add";
val simp_delN = "del";

val simp_tag = (simpN, []);
val simp_add = (simpN, [simp_addN]);
val simp_del = (simpN, [simp_delN]);


(* attributes *)

local
  fun simp_attr change args (x, tth) =
    if null args orelse args = [simp_addN] then change (op addsimps) (x, tth)
    else if args = [simp_delN] then change (op delsimps) (x, tth)
    else Attribute.fail simpN ("bad argument(s) " ^ commas_quote args);

  fun change_global_ss f (thy, tth) =
    let val r = simpset_ref_of thy
    in r := f (! r, [Attribute.thm_of tth]); (thy, tth) end;

  fun change_local_ss f (lthy, tth) =
    let val ss = f (get_local_simpset lthy, [Attribute.thm_of tth])
    in (put_local_simpset ss lthy, tth) end;

  val simp_attr_global = simp_attr change_global_ss;
  val simp_attr_local = simp_attr change_local_ss;
in
(* FIXME
  val setup_attrs = IsarThy.add_attributes
    [(simpN, simp_attr_global, simp_attr_local, "simplifier")];
*)
  val setup_attrs = I;

  val simp_add_global = simp_attr_global [simp_addN];
  val simp_del_global = simp_attr_global [simp_delN];
  val simp_add_local = simp_attr_local [simp_addN];
  val simp_del_local = simp_attr_local [simp_delN];
end;



(** simplification tactics **)

fun solve_all_tac (subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac) mss =
  let
    val ss =
      make_ss (mss, subgoal_tac, loop_tacs, unsafe_finish_tac, unsafe_finish_tac);
    val solve1_tac = (subgoal_tac ss THEN_ALL_NEW (K no_tac)) 1
  in DEPTH_SOLVE solve1_tac end;

fun loop_tac loop_tacs = FIRST'(map snd loop_tacs);

(*not totally safe: may instantiate unknowns that appear also in other subgoals*)
fun basic_gen_simp_tac mode =
  fn (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac}) =>
    let
      fun simp_loop_tac i =
        asm_rewrite_goal_tac mode
          (solve_all_tac (subgoal_tac,loop_tacs,finish_tac,unsafe_finish_tac))
          mss i
        THEN (finish_tac (prems_of_mss mss) i ORELSE
              TRY ((loop_tac loop_tacs THEN_ALL_NEW simp_loop_tac) i))
    in simp_loop_tac end;

fun gen_simp_tac mode (ss as Simpset {unsafe_finish_tac, ...}) =
  basic_gen_simp_tac mode (ss setSSolver unsafe_finish_tac);

val          simp_tac = gen_simp_tac (false, false, false);
val      asm_simp_tac = gen_simp_tac (false, true, false);
val     full_simp_tac = gen_simp_tac (true,  false, false);
val   asm_lr_simp_tac = gen_simp_tac (true,  true, false);
val asm_full_simp_tac = gen_simp_tac (true,  true, true);

(*not totally safe: may instantiate unknowns that appear also in other subgoals*)
val safe_asm_full_simp_tac = basic_gen_simp_tac (true, true, false);

(*the abstraction over the proof state delays the dereferencing*)
fun          Simp_tac i st =          simp_tac (simpset ()) i st;
fun      Asm_simp_tac i st =      asm_simp_tac (simpset ()) i st;
fun     Full_simp_tac i st =     full_simp_tac (simpset ()) i st;
fun   Asm_lr_simp_tac i st =   asm_lr_simp_tac (simpset ()) i st;
fun Asm_full_simp_tac i st = asm_full_simp_tac (simpset ()) i st;



(** simplification rules and conversions **)

fun simp rew mode (Simpset {mss, subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac}) thm =
  let
    val tacf = solve_all_tac (subgoal_tac, loop_tacs, finish_tac, unsafe_finish_tac);
    fun prover m th = apsome fst (Seq.pull (tacf m th));
  in rew mode prover mss thm end;

val simp_thm = simp Drule.rewrite_thm;
val simp_cterm = simp Drule.rewrite_cterm;

val          simplify = simp_thm (false, false, false);
val      asm_simplify = simp_thm (false, true, false);
val     full_simplify = simp_thm (true, false, false);
val asm_full_simplify = simp_thm (true, true, false);

val          rewrite = simp_cterm (false, false, false);
val      asm_rewrite = simp_cterm (false, true, false);
val     full_rewrite = simp_cterm (true, false, false);
val asm_full_rewrite = simp_cterm (true, true, false);



(** theory setup **)

val setup = [SimpsetData.init, setup_attrs];


end;


structure BasicSimplifier: BASIC_SIMPLIFIER = Simplifier;
open BasicSimplifier;
