(*  Title: 	HOL/List
    ID:         List.ML,v 1.11 1994/08/25 09:05:37 lcp Exp
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1993  University of Cambridge

Definition of type 'a list by a least fixed point
*)

open List;

val list_con_defs = [NIL_def, CONS_def];

goal List.thy "list(A) = {Numb(0)} <+> (A <*> list(A))";
let val rew = rewrite_rule list_con_defs in  
by (fast_tac (univ_cs addSIs (equalityI :: map rew list.intrs)
                      addEs [rew list.elim]) 1)
end;
val list_unfold = result();

(*This justifies using list in other recursive type definitions*)
goalw List.thy list.defs "!!A B. A<=B ==> list(A) <= list(B)";
by (rtac lfp_mono 1);
by (REPEAT (ares_tac basic_monos 1));
val list_mono = result();

(*Type checking -- list creates well-founded sets*)
goalw List.thy (list_con_defs @ list.defs) "list(sexp) <= sexp";
by (rtac lfp_lowerbound 1);
by (fast_tac (univ_cs addIs sexp.intrs@[sexp_In0I,sexp_In1I]) 1);
val list_sexp = result();

(* A <= sexp ==> list(A) <= sexp *)
val list_subset_sexp = standard ([list_mono, list_sexp] MRS subset_trans);

(*Induction for the type 'a list *)
val prems = goalw List.thy [Nil_def,Cons_def]
    "[| P(Nil);   \
\       !!x xs. P(xs) ==> P(x # xs) |]  ==> P(l)";
by (rtac (Rep_list_inverse RS subst) 1);   (*types force good instantiation*)
by (rtac (Rep_list RS list.induct) 1);
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [rangeE, ssubst, Abs_list_inverse RS subst] 1));
val list_induct = result();

(*Perform induction on xs. *)
fun list_ind_tac a M = 
    EVERY [res_inst_tac [("l",a)] list_induct M,
	   rename_last_tac a ["1"] (M+1)];

(*** Isomorphisms ***)

goal List.thy "inj(Rep_list)";
by (rtac inj_inverseI 1);
by (rtac Rep_list_inverse 1);
val inj_Rep_list = result();

goal List.thy "inj_onto(Abs_list,list(range(Leaf)))";
by (rtac inj_onto_inverseI 1);
by (etac Abs_list_inverse 1);
val inj_onto_Abs_list = result();

(** Distinctness of constructors **)

goalw List.thy list_con_defs "CONS(M,N) ~= NIL";
by (rtac In1_not_In0 1);
val CONS_not_NIL = result();
val NIL_not_CONS = standard (CONS_not_NIL RS not_sym);

val CONS_neq_NIL = standard (CONS_not_NIL RS notE);
val NIL_neq_CONS = sym RS CONS_neq_NIL;

goalw List.thy [Nil_def,Cons_def] "x # xs ~= Nil";
by (rtac (CONS_not_NIL RS (inj_onto_Abs_list RS inj_onto_contraD)) 1);
by (REPEAT (resolve_tac (list.intrs @ [rangeI, Rep_list]) 1));
val Cons_not_Nil = result();

val Nil_not_Cons = standard (Cons_not_Nil RS not_sym);

val Cons_neq_Nil = standard (Cons_not_Nil RS notE);
val Nil_neq_Cons = sym RS Cons_neq_Nil;

(** Injectiveness of CONS and Cons **)

goalw List.thy [CONS_def] "(CONS(K,M)=CONS(L,N)) = (K=L & M=N)";
by (fast_tac (HOL_cs addSEs [Scons_inject, make_elim In1_inject]) 1);
val CONS_CONS_eq = result();

val CONS_inject = standard (CONS_CONS_eq RS iffD1 RS conjE);

(*For reasoning about abstract list constructors*)
val list_cs = set_cs addIs [Rep_list] @ list.intrs
	             addSEs [CONS_neq_NIL,NIL_neq_CONS,CONS_inject]
		     addSDs [inj_onto_Abs_list RS inj_ontoD,
			     inj_Rep_list RS injD, Leaf_inject];

goalw List.thy [Cons_def] "(x#xs=y#ys) = (x=y & xs=ys)";
by (fast_tac list_cs 1);
val Cons_Cons_eq = result();
val Cons_inject = standard (Cons_Cons_eq RS iffD1 RS conjE);

val [major] = goal List.thy "CONS(M,N): list(A) ==> M: A & N: list(A)";
by (rtac (major RS setup_induction) 1);
by (etac list.induct 1);
by (ALLGOALS (fast_tac list_cs));
val CONS_D = result();

val prems = goalw List.thy [CONS_def,In1_def]
    "CONS(M,N): sexp ==> M: sexp & N: sexp";
by (cut_facts_tac prems 1);
by (fast_tac (set_cs addSDs [Scons_D]) 1);
val sexp_CONS_D = result();


(*Basic ss with constructors and their freeness*)
val list_free_simps = [Cons_not_Nil, Nil_not_Cons, Cons_Cons_eq,
		       CONS_not_NIL, NIL_not_CONS, CONS_CONS_eq]
                      @ list.intrs;
val list_free_ss = HOL_ss  addsimps  list_free_simps;

goal List.thy "!!N. N: list(A) ==> !M. N ~= CONS(M,N)";
by (etac list.induct 1);
by (ALLGOALS (asm_simp_tac list_free_ss));
val not_CONS_self = result();

goal List.thy "!x. l ~= x#l";
by (list_ind_tac "l" 1);
by (ALLGOALS (asm_simp_tac list_free_ss));
val not_Cons_self = result();


goal List.thy "(xs ~= []) = (? y ys. xs = y#ys)";
by(list_ind_tac "xs" 1);
by(simp_tac list_free_ss 1);
by(asm_simp_tac list_free_ss 1);
by(REPEAT(resolve_tac [exI,refl,conjI] 1));
val neq_Nil_conv = result();

(** Conversion rules for List_case: case analysis operator **)

goalw List.thy [List_case_def,NIL_def] "List_case(c, h, NIL) = c";
by (rtac Case_In0 1);
val List_case_NIL = result();

goalw List.thy [List_case_def,CONS_def]  "List_case(c, h, CONS(M,N)) = h(M,N)";
by (simp_tac (HOL_ss addsimps [Split,Case_In1]) 1);
val List_case_CONS = result();

(*** List_rec -- by wf recursion on pred_sexp ***)

(* The trancl(pred_sexp) is essential because pred_sexp_CONS_I1,2 would not
   hold if pred_sexp^+ were changed to pred_sexp. *)

val List_rec_unfold = [List_rec_def, wf_pred_sexp RS wf_trancl] MRS def_wfrec
                      |> standard;

(** pred_sexp lemmas **)

goalw List.thy [CONS_def,In1_def]
    "!!M. [| M: sexp;  N: sexp |] ==> <M, CONS(M,N)> : pred_sexp^+";
by (asm_simp_tac pred_sexp_ss 1);
val pred_sexp_CONS_I1 = result();

goalw List.thy [CONS_def,In1_def]
    "!!M. [| M: sexp;  N: sexp |] ==> <N, CONS(M,N)> : pred_sexp^+";
by (asm_simp_tac pred_sexp_ss 1);
val pred_sexp_CONS_I2 = result();

val [prem] = goal List.thy
    "<CONS(M1,M2), N> : pred_sexp^+ ==> \
\    <M1,N> : pred_sexp^+ & <M2,N> : pred_sexp^+";
by (rtac (prem RS (pred_sexp_subset_Sigma RS trancl_subset_Sigma RS 
		   subsetD RS SigmaE2)) 1);
by (etac (sexp_CONS_D RS conjE) 1);
by (REPEAT (ares_tac [conjI, pred_sexp_CONS_I1, pred_sexp_CONS_I2,
		      prem RSN (2, trans_trancl RS transD)] 1));
val pred_sexp_CONS_D = result();

(** Conversion rules for List_rec **)

goal List.thy "List_rec(NIL,c,h) = c";
by (rtac (List_rec_unfold RS trans) 1);
by (simp_tac (HOL_ss addsimps [List_case_NIL]) 1);
val List_rec_NIL = result();

goal List.thy "!!M. [| M: sexp;  N: sexp |] ==> \
\    List_rec(CONS(M,N), c, h) = h(M, N, List_rec(N,c,h))";
by (rtac (List_rec_unfold RS trans) 1);
by (asm_simp_tac
    (HOL_ss addsimps [List_case_CONS, list.CONS_I, pred_sexp_CONS_I2, 
		      cut_apply])1);
val List_rec_CONS = result();

(*** list_rec -- by List_rec ***)

val Rep_list_in_sexp =
    [range_Leaf_subset_sexp RS list_subset_sexp, Rep_list] MRS subsetD;

local
  val list_rec_simps = list_free_simps @
	          [List_rec_NIL, List_rec_CONS, 
		   Abs_list_inverse, Rep_list_inverse,
		   Rep_list, rangeI, inj_Leaf, Inv_f_f,
		   sexp.LeafI, Rep_list_in_sexp]
in
  val list_rec_Nil = prove_goalw List.thy [list_rec_def, Nil_def]
      "list_rec(Nil,c,h) = c"
   (fn _=> [simp_tac (HOL_ss addsimps list_rec_simps) 1]);

  val list_rec_Cons = prove_goalw List.thy [list_rec_def, Cons_def]
      "list_rec(a#l, c, h) = h(a, l, list_rec(l,c,h))"
   (fn _=> [simp_tac (HOL_ss addsimps list_rec_simps) 1]);
end;

val list_simps = [List_rec_NIL, List_rec_CONS,
		  list_rec_Nil, list_rec_Cons];
val list_ss = list_free_ss addsimps list_simps;


(*Type checking.  Useful?*)
val major::A_subset_sexp::prems = goal List.thy
    "[| M: list(A);    	\
\       A<=sexp;      	\
\       c: C(NIL);      \
\       !!x y r. [| x: A;  y: list(A);  r: C(y) |] ==> h(x,y,r): C(CONS(x,y)) \
\    |] ==> List_rec(M,c,h) : C(M :: 'a item)";
val sexp_ListA_I = A_subset_sexp RS list_subset_sexp RS subsetD;
val sexp_A_I = A_subset_sexp RS subsetD;
by (rtac (major RS list.induct) 1);
by (ALLGOALS(asm_simp_tac (list_ss addsimps ([sexp_A_I,sexp_ListA_I]@prems))));
val List_rec_type = result();

(** Generalized map functionals **)

goalw List.thy [Rep_map_def] "Rep_map(f,Nil) = NIL";
by (rtac list_rec_Nil 1);
val Rep_map_Nil = result();

goalw List.thy [Rep_map_def]
    "Rep_map(f, x#xs) = CONS(f(x), Rep_map(f,xs))";
by (rtac list_rec_Cons 1);
val Rep_map_Cons = result();

goalw List.thy [Rep_map_def] "!!f. (!!x. f(x): A) ==> Rep_map(f,xs): list(A)";
by (rtac list_induct 1);
by(ALLGOALS(asm_simp_tac list_ss));
val Rep_map_type = result();

goalw List.thy [Abs_map_def] "Abs_map(g,NIL) = Nil";
by (rtac List_rec_NIL 1);
val Abs_map_NIL = result();

val prems = goalw List.thy [Abs_map_def]
    "[| M: sexp;  N: sexp |] ==> \
\    Abs_map(g, CONS(M,N)) = g(M) # Abs_map(g,N)";
by (REPEAT (resolve_tac (List_rec_CONS::prems) 1));
val Abs_map_CONS = result();

(*These 2 rules ease the use of primitive recursion.  NOTE USE OF == *)
val [rew] = goal List.thy
    "[| !!xs. f(xs) == list_rec(xs,c,h) |] ==> f([]) = c";
by (rewtac rew);
by (rtac list_rec_Nil 1);
val def_list_rec_Nil = result();

val [rew] = goal List.thy
    "[| !!xs. f(xs) == list_rec(xs,c,h) |] ==> f(x#xs) = h(x,xs,f(xs))";
by (rewtac rew);
by (rtac list_rec_Cons 1);
val def_list_rec_Cons = result();

fun list_recs def =
      [standard (def RS def_list_rec_Nil),
       standard (def RS def_list_rec_Cons)];

(*** Unfolding the basic combinators ***)

val [null_Nil,null_Cons] = list_recs null_def;
val [_,hd_Cons] = list_recs hd_def;
val [_,tl_Cons] = list_recs tl_def;
val [ttl_Nil,ttl_Cons] = list_recs ttl_def;
val [append_Nil,append_Cons] = list_recs append_def;
val [mem_Nil, mem_Cons] = list_recs mem_def;
val [map_Nil,map_Cons] = list_recs map_def;
val [list_case_Nil,list_case_Cons] = list_recs list_case_def;
val [filter_Nil,filter_Cons] = list_recs filter_def;
val [list_all_Nil,list_all_Cons] = list_recs list_all_def;

val list_ss = arith_ss addsimps
  [Cons_not_Nil, Nil_not_Cons, Cons_Cons_eq,
   list_rec_Nil, list_rec_Cons,
   null_Nil, null_Cons, hd_Cons, tl_Cons, ttl_Nil, ttl_Cons,
   mem_Nil, mem_Cons,
   list_case_Nil, list_case_Cons,
   append_Nil, append_Cons,
   map_Nil, map_Cons,
   list_all_Nil, list_all_Cons,
   filter_Nil, filter_Cons];


(** @ - append **)

goal List.thy "(xs@ys)@zs = xs@(ys@zs)";
by(list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac list_ss));
val append_assoc = result();

goal List.thy "xs @ [] = xs";
by(list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac list_ss));
val append_Nil2 = result();

(** mem **)

goal List.thy "x mem (xs@ys) = (x mem xs | x mem ys)";
by(list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac (list_ss setloop (split_tac [expand_if]))));
val mem_append = result();

goal List.thy "x mem [x:xs.P(x)] = (x mem xs & P(x))";
by(list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac (list_ss setloop (split_tac [expand_if]))));
val mem_filter = result();

(** list_all **)

goal List.thy "(Alls x:xs.True) = True";
by(list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac list_ss));
val list_all_True = result();

goal List.thy "list_all(p,xs@ys) = (list_all(p,xs) & list_all(p,ys))";
by(list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac list_ss));
val list_all_conj = result();

goal List.thy "(Alls x:xs.P(x)) = (!x. x mem xs --> P(x))";
by(list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac (list_ss setloop (split_tac [expand_if]))));
by(fast_tac HOL_cs 1);
val list_all_mem_conv = result();


(** The functional "map" **)

val map_simps = [Abs_map_NIL, Abs_map_CONS, 
		 Rep_map_Nil, Rep_map_Cons, 
		 map_Nil, map_Cons];
val map_ss = list_free_ss addsimps map_simps;

val [major,A_subset_sexp,minor] = goal List.thy 
    "[| M: list(A);  A<=sexp;  !!z. z: A ==> f(g(z)) = z |] \
\    ==> Rep_map(f, Abs_map(g,M)) = M";
by (rtac (major RS list.induct) 1);
by (ALLGOALS (asm_simp_tac(map_ss addsimps [sexp_A_I,sexp_ListA_I,minor])));
val Abs_map_inverse = result();

(*Rep_map_inverse is obtained via Abs_Rep_map and map_ident*)

(** list_case **)

goal List.thy
 "P(list_case(a,f,xs)) = ((xs=[] --> P(a)) & \
\                         (!y ys. xs=y#ys --> P(f(y,ys))))";
by(list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac list_ss));
by(fast_tac HOL_cs 1);
val expand_list_case = result();


(** Additional mapping lemmas **)

goal List.thy "map(%x.x, xs) = xs";
by (list_ind_tac "xs" 1);
by (ALLGOALS (asm_simp_tac map_ss));
val map_ident = result();

goal List.thy "map(f, xs@ys) = map(f,xs) @ map(f,ys)";
by (list_ind_tac "xs" 1);
by (ALLGOALS (asm_simp_tac (map_ss addsimps [append_Nil,append_Cons])));
val map_append = result();

goalw List.thy [o_def] "map(f o g, xs) = map(f, map(g, xs))";
by (list_ind_tac "xs" 1);
by (ALLGOALS (asm_simp_tac map_ss));
val map_compose = result();

goal List.thy "!!f. (!!x. f(x): sexp) ==> \
\	Abs_map(g, Rep_map(f,xs)) = map(%t. g(f(t)), xs)";
by (list_ind_tac "xs" 1);
by(ALLGOALS(asm_simp_tac(map_ss addsimps
       [Rep_map_type,list_sexp RS subsetD])));
val Abs_Rep_map = result();

val list_ss = list_ss addsimps
  [mem_append, mem_filter, append_assoc, append_Nil2, map_ident,
   list_all_True, list_all_conj];

