(*  Title: 	ZF/upair
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

UNORDERED pairs in Zermelo-Fraenkel Set Theory 

Observe the order of dependence:
    Upair is defined in terms of Replace
    Un is defined in terms of Upair and Union (similarly for Int)
    cons is defined in terms of Upair and Un
    Ordered pairs and descriptions are defined using cons ("set notation")
*)

(* A few lemmas about power sets before derivation of pairing *)

val Pow_bottom = prove_goal ZF_Rule.thy "0 : Pow(A)"
 (fn _ =>
  [ (REPEAT (ares_tac [PowI,subsetI,emptyE] 1)) ]);

val Pow_top = prove_goal ZF_Rule.thy "A : Pow(A)"
 (fn _ =>
  [ (REPEAT (ares_tac [PowI,subsetI] 1)) ]);

val Pow_neq_0 = prove_goal ZF_Rule.thy "Pow(A) = 0 ==> P"
 (fn prems=>
  [ (rtac equals0D 1),
    (REPEAT (resolve_tac (prems@[Pow_top]) 1)) ]);


(* The pairing axiom is derivable from axioms for replacement and power sets *)

val pairing = prove_goalw ZF_Rule.thy [Upair_def]
    "a:Upair(b,c) <-> (a=b | a=c)"
 (fn _ =>
  [ (*could also use      (fast_tac (lemmas_cs addSEs [equalityE]) 1), *)
    (fast_tac (FOL_cs addIs [ReplaceI, Pow_bottom, Pow_top] 
		     addEs [ReplaceE, sym RS Pow_neq_0, Pow_neq_0]) 1) ]);


(*** Unordered pairs - Upair ***)

val UpairI1 = prove_goal ZF_Rule.thy "a : Upair(a,b)"
 (fn _ =>
  [ (resolve_tac [pairing RS iffD2] 1),
    (resolve_tac [refl RS disjI1] 1) ]);

val UpairI2 = prove_goal ZF_Rule.thy "b : Upair(a,b)"
 (fn _ =>
  [ (resolve_tac [pairing RS iffD2] 1),
    (resolve_tac [refl RS disjI2] 1) ]);

val UpairE = prove_goal ZF_Rule.thy
    "[| a : Upair(b,c);  a = b ==> P;  a = c ==> P |] ==> P"
 (fn major::prems=>
  [ (resolve_tac [major RS (pairing RS iffD1 RS disjE)] 1),
    (REPEAT (eresolve_tac prems 1)) ]);


(*** Rules for binary union -- Un -- defined via Upair ***)

val UnI1 = prove_goalw ZF_Rule.thy [Un_def] "c : A ==> c : A Un B"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems @ [UnionI,UpairI1]) 1)) ]);

val UnI2 = prove_goalw ZF_Rule.thy [Un_def] "c : B ==> c : A Un B"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems @ [UnionI,UpairI2]) 1)) ]);

(*Classical introduction rule: no commitment to A vs B*)
val UnCI = prove_goal ZF_Rule.thy 
   "(~c : B ==> c : A) ==> c : A Un B"
 (fn prems=>
  [ (rtac classical 1),
    (REPEAT (ares_tac (prems@[UnI1,notI]) 1)),
    (REPEAT (ares_tac (prems@[UnI2,notE]) 1)) ]);

val UnE = prove_goalw ZF_Rule.thy [Un_def] 
    "[| c : A Un B;  c:A ==> P;  c:B ==> P |] ==> P"
 (fn major::prems=>
  [ (rtac (major RS UnionE) 1),
    (REPEAT (etac UpairE 1
     ORELSE (EVERY1 [resolve_tac prems, etac subst, assume_tac]))) ]);


(*** Rules for small intersection -- Int -- defined via Upair ***)

val IntI = prove_goalw ZF_Rule.thy [Int_def]
    "[| c : A;  c : B |] ==> c : A Int B"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems @ [UpairI1,InterI]) 1
     ORELSE eresolve_tac [UpairE, ssubst] 1)) ]);

val IntD1 = prove_goalw ZF_Rule.thy [Int_def] "c : A Int B ==> c : A"
 (fn [major]=>
  [ (rtac (UpairI1 RS (major RS InterD)) 1) ]);

val IntD2 = prove_goalw ZF_Rule.thy [Int_def] "c : A Int B ==> c : B"
 (fn [major]=>
  [ (rtac (UpairI2 RS (major RS InterD)) 1) ]);

val IntE = prove_goal ZF_Rule.thy
    "[| c : A Int B;  [| c:A; c:B |] ==> P |] ==> P"
 (fn prems=>
  [ (resolve_tac prems 1),
    (REPEAT (resolve_tac [IntD1,IntD2] 1 THEN resolve_tac prems 1)) ]);


(*** Rules for set difference -- defined via Upair ***)

val DiffI = prove_goalw ZF_Rule.thy [Diff_def] "[| c : A;  ~ c : B |] ==> c : A - B"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems @ [CollectI]) 1)) ]);

val DiffD1 = prove_goalw ZF_Rule.thy [Diff_def] "c : A - B ==> c : A"
 (fn major::prems=>
  [ (rtac (major RS CollectD1) 1) ]);

val DiffD2 = prove_goalw ZF_Rule.thy [Diff_def] "[| c : A - B;  c : B |] ==> P"
 (fn major::prems=>
  [ (rtac (major RS CollectD2 RS notE) 1),
    (resolve_tac prems 1) ]);

val DiffE = prove_goal ZF_Rule.thy
    "[| c : A - B;  [| c:A; ~ c:B |] ==> P |] ==> P"
 (fn prems=>
  [ (resolve_tac prems 1),
    (REPEAT (ares_tac [notI] 1 
     ORELSE (resolve_tac [DiffD1,DiffD2] 1 THEN resolve_tac prems 1))) ]);


(*** Rules for cons -- defined via Un and Upair ***)

val consI1 = prove_goalw ZF_Rule.thy [cons_def] "a : cons(a,B)"
 (fn prems=>
  [ (REPEAT (resolve_tac [UnI1,UpairI1] 1)) ]);

val consI2 = prove_goalw ZF_Rule.thy [cons_def] "a : B ==> a : cons(b,B)"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems @ [UnI2]) 1)) ]);

(*Classical introduction rule*)
val consCI = prove_goalw ZF_Rule.thy [cons_def]
   "(~ a:B ==> a=b) ==> a: cons(b,B)"
 (fn prems=>
  [ (rtac UnCI 1),
    (eresolve_tac (prems RL [ssubst]) 1),
    (rtac UpairI1 1) ]);

val consE = prove_goalw ZF_Rule.thy [cons_def]
    "[| a : cons(b,A);  a=b ==> P;  a:A ==> P |] ==> P"
 (fn major::prems=>
  [ (rtac (major RS UnE) 1),
    (REPEAT (eresolve_tac (prems @ [UpairE]) 1)) ]);

(*** Singletons - using cons ***)

val singletonI = prove_goal ZF_Rule.thy "a : {a}"
 (fn prems=>
  [ (rtac consI1 1) ]);

val singletonE = prove_goal ZF_Rule.thy "[| a : {b}; a=b ==> P |] ==> P"
 (fn major::prems=>
  [ (rtac (major RS consE) 1),
    (REPEAT (eresolve_tac (prems @ [emptyE]) 1)) ]);


(*** Rules for Descriptions ***)

val the_equality = prove_goalw ZF_Rule.thy [the_def]
    "[| P(a);  !!x. P(x) ==> x=a |] ==> (THE x. P(x)) = a"
 (fn prems=>
  [ (fast_tac (lemmas_cs addIs ([equalityI,singletonI]@prems) 
	               addEs (prems RL [subst])) 1) ]);

(* Only use this if you already know EX!x. P(x) *)
val the_equality2 = prove_goal ZF_Rule.thy
    "[| EX! x. P(x);  P(a) |] ==> (THE x. P(x)) = a"
 (fn major::prems=>
  [ (rtac (major RS ex1E) 1),
    (REPEAT (ares_tac ([the_equality]@prems) 1)),
    (rtac trans 1),
    (rtac sym 2),
    (REPEAT (ares_tac prems 1 ORELSE eresolve_tac [mp,allE] 1)) ]);

val theI = prove_goal ZF_Rule.thy "EX! x. P(x) ==> P(THE x. P(x))"
 (fn major::prems=>
  [ (rtac (major RS ex1E) 1),
    (resolve_tac [major RS the_equality2 RS ssubst] 1),
    (REPEAT (assume_tac 1)) ]);

val the_cong = prove_goalw ZF_Rule.thy [the_def]
    "[| !!y. P(y) <-> Q(y) |] ==> (THE x. P(x)) = (THE x. Q(x))"
 (fn prems=>
  [ (prove_cong_tac (prems@[Replace_cong]) 1) ]);


(*** Foundation lemmas ***)

val mem_anti_sym = prove_goal ZF_Rule.thy "[| a:b;  b:a |] ==> P"
 (fn prems=>
  [ (rtac disjE 1),
    (res_inst_tac [("A","{a,b}")] foundation 1),
    (etac equals0D 1),
    (rtac consI1 1),
    (fast_tac (lemmas_cs addIs (prems@[consI1,consI2]) 
		        addSEs [consE,equalityE]) 1) ]);

val mem_anti_refl = prove_goal ZF_Rule.thy "a:a ==> P"
 (fn prems=>
  [ (rtac mem_anti_sym 1),
    (resolve_tac prems 1),
    (resolve_tac prems 1) ]);


(*** Rules for succ ***)

val succI1 = prove_goalw ZF_Rule.thy [succ_def] "i : succ(i)"
 (fn prems=>
  [ (rtac consI1 1) ]);

val succI2 = prove_goalw ZF_Rule.thy [succ_def]
    "i : j ==> i : succ(j)"
 (fn prems=>
  [ (REPEAT (resolve_tac (prems @ [consI2]) 1)) ]);

(*Classical introduction rule*)
val succCI = prove_goalw ZF_Rule.thy [succ_def]
   "(~ i:j ==> i=j) ==> i: succ(j)"
 (fn prems=> [ (rtac consCI 1), (eresolve_tac prems 1) ]);

val succE = prove_goalw ZF_Rule.thy [succ_def]
    "[| i : succ(j);  i=j ==> P;  i:j ==> P |] ==> P"
 (fn major::prems=>
  [ (rtac (major RS consE) 1),
    (REPEAT (eresolve_tac prems 1)) ]);

val succ_neq_0 = prove_goal ZF_Rule.thy "[| succ(n)=0 |] ==> P"
 (fn [major]=>
  [ (rtac (major RS equalityD1 RS subsetD RS emptyE) 1),
    (rtac succI1 1) ]);

val succ_less = prove_goalw ZF_Rule.thy [succ_def] "succ(i)<=j ==> i:j"
 (fn [prem]=>
  [ (rtac (consI1 RS (prem RS subsetD)) 1) ]);

val succ_inject = prove_goal ZF_Rule.thy "succ(m) = succ(n) ==> m=n"
 (fn prems=>
  [ (resolve_tac (prems RL [equalityE]) 1),
    (REPEAT (eresolve_tac [asm_rl, sym, succE, make_elim succ_less,
			  mem_anti_sym] 1)) ]);

(*UpairI1/2 should become UpairCI;  mem_anti_refl as a hazE? *)
val upair_cs = lemmas_cs
  addSIs [singletonI, DiffI, IntI, UnCI, consCI, succCI, UpairI1,UpairI2]
  addSEs [singletonE, DiffE, IntE, UnE, consE, succE, UpairE];

