(*  Title: 	ZF/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Syntax definitions for Zermelo-Fraenkel Set Theory
*)

signature ZF_SYNTAX =
  sig
  val const_decs: (string list * string) list
  val sext: sext
  end;

functor ZF_SyntaxFun () : ZF_SYNTAX = 
struct

(** Ordered tuples -- change the type of Pair from tuple to i 
    @PairA is for the outermost tuple and ensures that <...> are printed
    @PairB is for inner tuples nested to the right. **)

(*Both @PairA and @PairB translate to Pair*)
fun pairA_tr[x, y] = Const("Pair", dummyT) $ x $ y;
fun pairB_tr[x, y] = Const("Pair", dummyT) $ x $ y;

(*Pair translates to either @PairA or @PairB*)
fun pairB_tr' (Const("Pair",_) $ x $ y) = 
	Const("@PairB", dummyT) $ x $ pairB_tr' y
  | pairB_tr' x = x;

fun pair_tr'[x,y] = Const("@PairA", dummyT) $ x $ pairB_tr' y;

(*** finite sets ***)

val mtset = Const("0", dummyT);
val cons = Const("cons", dummyT);

(* enumeration of finite set elements *)
fun make_finset (Const("@Sing",_)$e) = cons $ e $ mtset
  | make_finset (Const("@Ulist",_)$e$l) = cons $ e $ make_finset l;

fun enum_tr[elts] = make_finset elts;

(*Return the elements of a finite set, raise exception if ill-formed.*)
fun dest_finset (Const("cons",_) $ e $ Const("0",_)) = 
	Const("@Sing",dummyT) $ e
  | dest_finset (Const("cons",_) $ e $ l) = 
	Const("@Ulist", dummyT) $ e $ dest_finset l
  | dest_finset (Const("0",_)) = mtset
  | dest_finset _ = raise Match;

fun enum_tr'[x,y] = Const("@Enum", dummyT) $ dest_finset(cons$x$y);

(** Replacement  { y . x:A, P[x,y] } = Replace(A, %x y.P[x,y]) **)
fun rep_tr[Free(y,_), Free(x,T), A, Pxy] =
    Const("Replace", dummyT) $ A $ absfree(x, T, absfree(y, T, Pxy));

fun rep_tr'[A, Abs(x,Tx, Abs(y,Ty,P))] =
    let val (y',Py)  = variant_abs(y,Ty,P)
        val (x',Pxy) = variant_abs(x,Tx,Py)
    in Const("@Rep",dummyT) $ Free(y',Ty) $ Free(x',Tx) $ A $ Pxy end;

(** Union/Inter of a family of sets -- involves RepFun **)
fun repfun_tr q [Free(id,T),A,B] = 
    Const(q,dummyT) $ (Const("RepFun",dummyT) $ A $ absfree(id,T,B));

fun repfun_tr' q [Const("RepFun",_) $ A $ Abs(id,T,B)] =
    let val (id',B') = variant_abs(id,T,B)
    in Const(q,dummyT) $ Free(id',T) $ A $ B' end;

(** For quantifications of the form %x:A.P(x) **)
fun qnt_tr q [Free(id,T), A, P] = Const(q,dummyT)$ A $absfree(id,T,P);

fun qnt_tr' q [A, Abs(id,T,P)] =
    let val (id',P') = variant_abs(id,T,P)
    in Const(q,dummyT) $ Free(id',T) $ A $ P' end;

(** 'Dependent' type operators **)
fun ndependent_tr q [A,B] = 
    Const(q,dummyT) $ A $ Abs("x", dummyT, incr_boundvars 1 B);

(*If there is a dependence then use quantifier q; else use r. *)
fun dependent_tr' (q,r) [A, Abs(id,T,B)] =
    if  0 mem (loose_bnos B) then 
      let val (id',B') = variant_abs(id,T,B)
      in Const(q,dummyT) $ Free(id',T) $ A $ B' end
    else Const(r,dummyT) $ A $ B;

val mixfix =
 [
  Delimfix("0",	"i", "0"),                   	(*the empty set*)
  Infixl("``",		"[i,i]=>i", 90),     	(*image*)
  Infixl("-``",		"[i,i]=>i", 90),     	(*inverse image*)
  Infixl("`",		"[i,i]=>i", 90),     	(*function application*)
  Mixfix("(_ */ _)", 	"[i,i]=>i", " *", [78,77], 80),    (*cart product*)
  Infixl("Int", 	"[i,i]=>i", 70),     	(*intersection*)
  Infixl("Un",		"[i,i]=>i", 65),     	(*union*)
  Infixl("-",		"[i,i]=>i", 65),     	(*set difference*)  
  Mixfix("(_ ->/ _)", 	"[i,i]=>i", " ->", [56,55], 55),   (*function space*)
  Infixl("<=",		"[i,i]=>o", 50),     	(*subset relation*)
  Infixl(":",		"[i,i]=>o", 50),     	(*membership relation*)
    (*ordered pairing and n-tuples*)
  Delimfix("(1<_,/_>)", "[i,tuple]=>i", "@PairA"),
  Delimfix("_",		"i=>tuple", ""),
  Delimfix("_,/_", 	"[i,tuple]=>tuple", "@PairB"),
    (*finite sets*)
  Delimfix("{_}", 	"ulist=>i", "@Enum"),	
  Delimfix("_", 	"i=>ulist", "@Sing"),
  Delimfix("_,/_", 	"[i,ulist]=>ulist", "@Ulist"),
    (*binding operators*)
  Delimfix("(1{_: _ ./ _})",	"[ID,i,o]=>i", "@Coll"),   (*collection*)
  Delimfix("(1{_ ./ _: _, _})",	"[ID,ID,i,o]=>i", "@Rep"), (*replacement*)
  Mixfix("(3INT _:_./ _)", 	"[ID,i,i]=>i", "@INTER", [], 10),
  Mixfix("(3UN _:_./ _)", 	"[ID,i,i]=>i", "@UNION",  [], 10),
  Mixfix("(3PROD _:_./ _)", 	"[ID,i,i]=>i", "@PROD", [], 10),
  Mixfix("(3SUM _:_./ _)", 	"[ID,i,i]=>i", "@SUM",  [], 10),
  Mixfix("(3THE _./ _)", 	"[ID,o]=>i",   "@THE",  [], 10),
  Mixfix("(3lam _:_./ _)", 	"[ID,i,i]=>i", "@lam",  [], 10),
  Mixfix("(3ALL _:_./ _)", 	"[ID,i,o]=>o", "@Ball", [], 10),
  Mixfix("(3EX _:_./ _)", 	"[ID,i,o]=>o", "@Bex",  [], 10)
 ];

val sext = Sext{
  mixfix=mixfix,
  parse_translation=
    [("@PairA",		pairA_tr),
     ("@PairB",		pairB_tr),
     ("@Enum",		enum_tr),
     ("@Rep",		rep_tr),
     mk_binder_tr("@THE","The"),
     ("@INTER",		repfun_tr "Inter"),
     ("@UNION",		repfun_tr "Union"),
     (" ->",		ndependent_tr "Pi"),
     (" *",		ndependent_tr "Sigma"),
     ("@PROD",		qnt_tr "Pi"),
     ("@SUM",		qnt_tr "Sigma"),
     ("@Coll",		qnt_tr "Collect"),
     ("@Ball",		qnt_tr "Ball"),
     ("@Bex",		qnt_tr "Bex"),
     ("@lam",		qnt_tr "Lambda")],
  print_translation=
    [("Pair",		pair_tr'),
     ("cons",		enum_tr'),
     ("Replace",	rep_tr'),
     mk_binder_tr'("The","@THE"),
     ("Inter",		repfun_tr' "@INTER"),
     ("Union",		repfun_tr' "@UNION"),
     ("Pi",		dependent_tr' ("@PROD"," ->")),
     ("Sigma",		dependent_tr' ("@SUM"," *")),
     ("Collect",	qnt_tr' "@Coll"),
     ("Ball",		qnt_tr' "@Ball"),
     ("Bex",		qnt_tr' "@Bex"),
     ("Lambda",		qnt_tr' "@lam")
    ]};


(*Constants and their types*)
val const_decs =
 [
  (["Pow","Union","Inter","fst","snd",
    "converse","succ"], 	"i=>i"),
  (["Inf"],			"i"),
  (["split"],			"[i, [i,i]=>i] => i"),
  (["Pair","Upair","cons"],  	"[i,i]=>i"),
  (["Collect"],			"[i, i=>o] => i"),
  (["PrimReplace", "Replace"],	"[i, [i,i]=>o] => i"),
  (["RepFun"],			"[i, i=>i] => i"),
  (["Pi","Sigma"],		"[i,i=>i]=>i"),
  (["domain","range","field"],	"i=>i"),
  (["Ball"], 			"[i, i=>o]=>o"),     (*bounded quantifiers*)
  (["Bex"], 			"[i, i=>o]=>o"),
  (["Lambda"], 			"[i, i=>i]=>i"),
  (["restrict"],		"[i, i] =>i"),
  (["The"], 			"[i=>o]=>i")
 ];

end;
