(*  Title: 	Syntax
    Author: 	Tobias Nipkow
*)

signature SYNTAX =
sig
  include LEXICON0
  include EXTENSION0
  include TYPE_EXT0
  include SEXTENSION1
  include PRINTER0
  structure Pretty:PRETTY
  type syntax
  val type_syn: syntax
  val extend: syntax * (indexname -> sort) -> string list * sext -> syntax
  val merge: syntax * syntax -> syntax
  val read: syntax -> typ -> string -> term
  val pretty_term: syntax -> term -> Pretty.T
  val print_syntax: syntax -> unit
  val string_of_typ: syntax -> typ -> string
end;


functor SyntaxFun(structure TypeExt:TYPE_EXT and Parser:PARSER
	and SExtension:SEXTENSION and Printer:PRINTER
        sharing TypeExt.Extension.XGram = Parser.XGram = Printer.XGram
	and TypeExt.Extension = SExtension.Extension)
	: SYNTAX =
struct

structure Extension = TypeExt.Extension;
structure XGram = Extension.XGram;
structure Symtab = XGram.Symtab;
structure Lexicon = Parser.ParseTree.Lexicon;

open Lexicon Extension TypeExt SExtension Printer

local open XGram in

datatype Tables = Tab of {gram: Parser.Gram, lexicon: Lexicon,
			  trtab: (term list -> term) Symtab.table,
			  prtab: Printer.tab,
			  tmixfixs: string list};

fun all_strings(opl:string Prod list) : string list =
    flat(map (fn Prod(_,syn,_,_) => terminals syn) opl);

fun str_to_tok(opl:string Prod list, lex:Lexicon) : Token Prod list =
      map (fn Prod(t,syn,s,pa) =>
              Prod(t,translate (hd o tokenize lex) syn,s,pa))
          opl;

fun xgram_to_tables(XGram{Roots,Prods,TrTab1,TrTab2,TMixFixs}) =
    let val lex = mk_lexicon(all_strings Prods);
        val gram = Parser.compile_xgram(Roots, str_to_tok(Prods,lex),
                                        token_to_string lex);
        val prtab = Printer.mk_print_tab(Prods,TrTab2);
    in Tab{gram=gram,trtab=TrTab1,lexicon=lex,prtab=prtab,tmixfixs=TMixFixs}
    end;


datatype GramGraph = emptyGG | extGG of GramGraphR * Extension.ext |
                     mergeGG of GramGraphR * GramGraphR
withtype GramGraphR = GramGraph ref;

datatype syntax = Syntax of GramGraphR * Tables;

fun flatGG ggr (xg,v) = if ggr mem v then (xg,v) else flatGG' ggr (xg,ggr::v)

and flatGG'(ref emptyGG) xgv = xgv |
    flatGG'(ref(extGG(ggr,ext))) xgv =
      let val (xg',v') = flatGG ggr xgv
      in (Extension.extend xg' ext, v') end |
    flatGG'(ref(mergeGG(ggr1,ggr2))) xgv = flatGG ggr1 (flatGG ggr2 xgv)

fun flattenGG ggr = fst(flatGG ggr (Extension.empty,[]));

fun mk_tables ggr = Syntax(ggr,xgram_to_tables(flattenGG ggr));

fun read (Syntax(_,Tab{gram,lexicon,trtab,...})) ty s =
let val tokens = tokenize lexicon s;
    val nt0 = Extension.typ_to_nt ty;
    val nt = if Parser.parsable(gram,nt0) then nt0 else Extension.ANY;
in Parser.ParseTree.pt_to_term (apl(trtab,Symtab.lookup))
		(Parser.parse(gram, nt, tokens))
   handle Parser.SYNTAX_ERR tl => error("Syntax error at\n" ^
           space_implode " " (map (token_to_string lexicon) tl))
end;

fun pretty_term(Syntax(_,Tab{prtab,tmixfixs,...})) =
	Printer.pretty_term(prtab,tmixfixs);

fun string_of_typ(Syntax(_,Tab{prtab,tmixfixs,...})) =
	Printer.string_of_typ(prtab,tmixfixs);

fun print_syntax(Syntax(_,Tab{gram,lexicon,...})) =
	Parser.print_gram(gram,lexicon);


val type_syn = mk_tables(ref(extGG(ref emptyGG,TypeExt.type_ext)));

fun extend (syn as Syntax(ggr,_), defS) (roots,sext) =
let fun readT s = TypeExt.typ_of_term defS (read syn Extension.typeT s)
    val ext = SExtension.ext_of_sext (roots,sext,readT)
in mk_tables(ref(extGG(ggr,ext))) end;

fun merge(Syntax(ggr1,_),Syntax(ggr2,_)) = mk_tables(ref(mergeGG(ggr1,ggr2)));

end;

end;
