(*  Title: 	LK/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

Sequent Calculus for classical first-order logic.
    Adapted from Philippe de Groote's work.
*)

signature LK_SYNTAX =
  sig
  val const_decs: (string list * string) list
  val seq_tr1   : term -> term
  val seq_tr1'  : term -> term
  val sext: sext
  end;

functor LK_SyntaxFun () : LK_SYNTAX = 
struct

(*Abstract over "sobj" -- representation of a sequence of formulae *)
fun abs_sobj t = Abs("sobj", Type("sobj",[]), t);

(*Representation of empty sequence*)
val Sempty =  abs_sobj (Bound 0);

fun seq_obj_tr(Const("@SeqId",_)$id) = id |
    seq_obj_tr(Const("@SeqVar",_)$id) = id |
    seq_obj_tr(fm) = Const("Seqof",dummyT)$fm;

fun seq_tr(_$obj$seq) = seq_obj_tr(obj)$seq_tr(seq) |
    seq_tr(_) = Bound 0;

fun seq_tr1(Const("@MtSeq",_)) = Sempty |
    seq_tr1(seq) = abs_sobj(seq_tr seq);

fun true_tr[s1,s2] = Const("Trueprop",dummyT)$seq_tr1 s1$seq_tr1 s2;

fun seq_obj_tr'(Const("Seqof",_)$fm) = fm |
    seq_obj_tr'(id) = Const("@SeqId",dummyT)$id;

fun seq_tr'(obj$sq,C) =
      let val sq' = case sq of
            Bound 0 => Const("@MtSeqCont",dummyT) |
            _ => seq_tr'(sq,Const("@SeqCont",dummyT))
      in C $ seq_obj_tr' obj $ sq' end;

fun seq_tr1'(Bound 0) = Const("@MtSeq",dummyT) |
    seq_tr1' s = seq_tr'(s,Const("@NmtSeq",dummyT));

fun true_tr'[Abs(_,_,s1),Abs(_,_,s2)] =
      Const("@Trueprop",dummyT)$seq_tr1' s1$seq_tr1' s2;


val mixfix =
 [ (*Representation of sequents*)
  Mixfix("((_)/ |- (_))", "[sequence,sequence] => prop", "@Trueprop", 
	 [6,6], 5),
  Delimfix("",		"sequence",			"@MtSeq"),
  Delimfix("__", 	"[seqobj,seqcont] => sequence", "@NmtSeq"),
  Delimfix("",		"seqcont",			"@MtSeqCont"),
  Delimfix(",/ __", 	"[seqobj,seqcont] => seqcont", 	"@SeqCont"),

  Delimfix("_",		"o => seqobj",			""),
  Delimfix("$_",	"ID => seqobj",		"@SeqId"),
  Delimfix("$_",	"VAR => seqobj",	"@SeqVar"),

  Infixl("=",	"['a,'a] => o",		50),
  Mixfix("~ _",	"o => o",	"not",	[40], 40),
  Infixr("&",	"[o,o] => o",		35),
  Infixr("|",	"[o,o] => o",		30),
  Infixr("-->",	"[o,o] => o",		25),
  Infixr("<->",	"[o,o] => o",		25),
  
  Binder("THE ", "('a => o) => 'a", "The", 0, 10),
  Binder("ALL ", "('a => o) => o",  "All", 0, 10),
  Binder("EX ",  "('a => o) => o",  "Ex",  0, 10) ];

val const_decs =
 [ (["True","False"],	"o"),
   (["Trueprop"], 	"[sobj=>sobj,sobj=>sobj] => prop"),
   (["Seqof"], 		"o => sobj=>sobj")];

val sext = Sext{mixfix=mixfix,
parse_translation = [("@Trueprop",true_tr)],
print_translation = [("Trueprop",true_tr')]};

end;
