(*  Title: 	HOL/wf.ML
    Author: 	Tobias Nipkow
    Copyright   1992  University of Cambridge

Well-founded Recursion
*)

structure WF =
struct

val thy = extend_theory Trancl.thy "WF_Rec"
 ([],[],[],[],
  [(["wf"], 		"('a * 'a)set => bool"),
   (["cut"], 		"['a => 'b, ('a * 'a)set, 'a] => 'a => 'b"),
   (["wftrec"], 	"[('a * 'a)set, ('a=>'b)=>('a=>'b), 'a] => 'b"),
   (["wfrec"], 		"[('a * 'a)set, 'a, ['a,'a=>'b]=>'b] => 'b"),
   (["is_recfun"], 
    "[('a * 'a)set, ('a=>'b)=>('a=>'b), 'a=>'b, 'a] => bool"),
   (["the_recfun"], 
    "[('a * 'a)set, ('a=>'b)=>('a=>'b), 'a] => 'a=>'b")
  ],
  None)
 [
  ("wf_def",
	 "wf(r) == (!P. (!x. (!y. <y,x>:r --> P(y)) --> P(x)) --> (!x.P(x)))"),
  ("cut_def", 	    "cut(f,r,x) == (%y. if(<y,x>:r, f(y), @z.True))"),

  ("is_recfun_def", 
   "is_recfun(r,H,f,a) == (f = cut(%x.H(cut(f,r,x),x), r, a))"),

  ("the_recfun_def", "the_recfun(r,H,a) == (@f.is_recfun(r,H,f,a))"),

  ("wftrec_def",     "wftrec(r,H,a) == H(the_recfun(r,H,a),a)"),

  (*version with swapped args, not requiring transitivity*)
  ("wfrec_def", "wfrec(r,a,J) == wftrec(trancl(r), %f x. J(x, cut(f,r,x)), a)")
 ];
end;

local val ax = get_axiom WF.thy
in
val wf_def = ax "wf_def";
val cut_def = ax "cut_def";
val is_recfun_def = ax "is_recfun_def";
val the_recfun_def = ax "the_recfun_def";
val wftrec_def = ax "wftrec_def";
val wfrec_def = ax "wfrec_def";
end;

val [H_cong] = mk_typed_congs WF.thy [("H","[?'a=>?'b,?'a]=>?'b")];
val H_cong2 = refl RSN (2,H_cong);

val ssw = HOL_ss addcongs (H_cong::mk_congs WF.thy ["cut"]);

val major::prems = goalw WF.thy [wf_def]
    "[| wf(r);          \
\       !!x.[| ! y. <y,x>: r --> P(y) |] ==> P(x) \
\    |]  ==>  P(a)";
by (rtac (major RS spec RS mp RS spec) 1);
by (fast_tac (HOL_cs addEs prems) 1);
val wf_induct = result();

(*Perform induction on i, then prove the wf(r) subgoal using prems. *)
fun wf_ind_tac a prems i = 
    EVERY [res_inst_tac [("a",a)] wf_induct i,
	   rename_last_tac a ["1"] (i+1),
	   ares_tac prems i];

val prems = goal WF.thy "[| wf(r);  <a,x>:r;  <x,a>:r |] ==> P";
by (subgoal_tac "! x. <a,x>:r --> <x,a>:r --> P" 1);
by (fast_tac (HOL_cs addIs prems) 1);
by (wf_ind_tac "a" prems 1);
by (fast_tac set_cs 1);
val wf_anti_sym = result();

val prems = goal WF.thy "[| wf(r);  <a,a>: r |] ==> P";
by (rtac wf_anti_sym 1);
by (REPEAT (resolve_tac prems 1));
val wf_anti_refl = result();

(*transitive closure of a WF relation is WF!*)
val [prem] = goal WF.thy "wf(r) ==> wf(trancl(r))";
by (rewtac wf_def);
by (strip_tac 1);
(*must retain the universal formula for later use!*)
by (rtac allE 1 THEN assume_tac 1);
by (etac mp 1);
by (res_inst_tac [("a","x")] (prem RS wf_induct) 1);
by (rtac (impI RS allI) 1);
by (etac tranclE 1);
by (fast_tac HOL_cs 1);
by (fast_tac HOL_cs 1);
val wf_trancl = result();


(** cut **)

goalw WF.thy [cut_def]
    "(cut(f,r,x) = cut(g,r,x)) = (!y. <y,x>:r --> f(y)=g(y))";
by (SIMP_CASE_TAC (ssw addrews [expand_fun_eq]) 1);
val cut_cut_eq = result();
val ssw = ssw addrews [cut_cut_eq];

val prems = goalw WF.thy [cut_def] "<x,a>:r ==> cut(f,r,a)(x) = f(x)";
by (SIMP_TAC (HOL_ss addrews prems) 1);
val cut_apply = result();


(*** is_recfun ***)

val prems = goalw WF.thy [is_recfun_def,cut_def]
    "[| is_recfun(r,H,f,a);  ~<b,a>:r |] ==> f(b) = (@z.True)";
by (cut_facts_tac prems 1);
by (etac ssubst 1);
by (ASM_SIMP_TAC ssw 1);
val is_recfun_undef = result();

(*eresolve_tac transD solves <a,b>:r using transitivity AT MOST ONCE
  mp amd allE  instantiate induction hypotheses*)
val indhyp_tac =
    DEPTH_SOLVE_1 o (ares_tac [TrueI] ORELSE' 
		     eresolve_tac [transD, mp, allE]);

(*** NOTE! the following simplifications need a different auto_tac!! ***)
val ssw' = set_auto_tac(ssw,indhyp_tac);

val prems = goalw WF.thy [is_recfun_def,cut_def]
    "[| wf(r);  trans(r);  is_recfun(r,H,f,a);  is_recfun(r,H,g,b) |] ==> \
    \ <x,a>:r --> <x,b>:r --> f(x)=g(x)";
by (EVERY1 [cut_facts_tac prems, etac wf_induct, strip_tac,
	    etac ssubst, etac ssubst,
	    ASM_SIMP_TAC ssw',
	    rtac (abs RS H_cong2), 
	    ASM_SIMP_CASE_TAC ssw']);
val is_recfun_equal = result() RS mp RS mp;

val prems as [a1,a2,a3,a4,_] = goalw WF.thy [cut_def]
    "[| wf(r);  trans(r); \
\       is_recfun(r,H,f,a);  is_recfun(r,H,g,b);  <b,a>:r |] ==> \
\    cut(f,r,b) = g";
by (EVERY1 [cut_facts_tac prems, 
	    rtac abs,
	    ASM_SIMP_CASE_TAC(ssw' addrews [a4 RS is_recfun_undef,
		a4 RS (a3 RS (a2 RS (a1 RS is_recfun_equal)))])]);
val is_recfun_cut = result();

(*** Main Existence Lemma -- Basic Properties of the_recfun ***)

val prems = goalw WF.thy [the_recfun_def]
    "is_recfun(r,H,f,a) ==> is_recfun(r,H, the_recfun(r,H,a), a)";
by (res_inst_tac [("P", "%x.is_recfun(r,H,x,a)")] selectI 1);
by (resolve_tac prems 1);
val is_the_recfun = result();

val prems = goal WF.thy
    "[|wf(r);  trans(r)|] ==> is_recfun(r, H, the_recfun(r,H,a), a)";
by (cut_facts_tac prems 1);
by (wf_ind_tac "a" prems 1);
by (res_inst_tac [("f", "cut(%y.H(the_recfun(r,H,y),y), r, a1)")] 
    is_the_recfun 1);
by (rewtac is_recfun_def);
by (rtac (cut_cut_eq RS ssubst) 1);
(*Applying the substitution: must keep the quantified assumption!!*)
by (EVERY1 [strip_tac, rtac H_cong2, rtac allE, atac,
            etac (mp RS ssubst), atac]);
by (fold_tac [is_recfun_def]);
(*check setting of auto_tac_ref!*)
by (ASM_SIMP_TAC (ssw' addrews [cut_apply,is_recfun_cut]) 1);
val unfold_the_recfun = result();

(*Beware incompleteness of unification!*)
val prems = goal WF.thy
    "[| wf(r);  trans(r);  <c,a>:r;  <c,b>:r |] \
\    ==> the_recfun(r,H,a,c) = the_recfun(r,H,b,c)";
by (DEPTH_SOLVE (ares_tac (prems@[is_recfun_equal,unfold_the_recfun]) 1));
val the_recfun_equal = result();

val prems = goal WF.thy
    "[| wf(r); trans(r); <b,a>:r |] \
\    ==> cut(the_recfun(r,H,a),r,b) = the_recfun(r,H,b)";
by (REPEAT (ares_tac (prems@[is_recfun_cut,unfold_the_recfun]) 1));
val the_recfun_cut = result();

(*** Unfolding wftrec ***)

val prems = goalw WF.thy [wftrec_def]
    "[| wf(r);  trans(r) |] ==> wftrec(r,H,a) = H(cut(wftrec(r,H),r,a), a)";
by (cut_facts_tac prems 1);
by (EVERY1 [stac (rewrite_rule [is_recfun_def] unfold_the_recfun),
	    REPEAT o atac, rtac H_cong2]);
by (rewtac wftrec_def);
by (ASM_SIMP_TAC (ssw addrews (prems RL [the_recfun_cut])) 1);
val wftrec_conv = result();

(*Unused but perhaps interesting*)
val prems = goal WF.thy
    "[| wf(r); trans(r); !f a. H(cut(f,r,a),a) = H(f,a) |] ==> \
\		wftrec(r,H) = H(wftrec(r,H))";
by (EVERY1 [cut_facts_tac prems, rtac abs, rtac trans, rtac wftrec_conv,
	  atac, atac, ASM_SIMP_TAC ssw]);
val wftrec_conv2 = result();

(** Removal of the premise trans(r); re-ordering of parameters **)

val [J_cong] = mk_typed_congs WF.thy [("J","[?'a, ?'a=>?'b]=>?'b")];

val [wfr] = goalw WF.thy [wfrec_def]
    "wf(r) ==> wfrec(r,a,J) = J(a, cut(%x.wfrec(r,x,J), r, a))";
by (rtac (wfr RS wf_trancl RS wftrec_conv RS ssubst) 1);
by (rtac trans_trancl 1);
by (rtac (refl RS J_cong) 1);
by (SIMP_TAC (ssw addrews [cut_apply, r_into_trancl]) 1);
val wfrec_conv = result();

(*This form avoids giant explosions in proofs.  NOTE USE OF == *)
val rew::prems = goal WF.thy
    "[| !!x. f(x)==wfrec(r,x,J);  wf(r) |] ==> f(a) = J(a, cut(%x.f(x),r,a))";
by (rewtac rew);
by (REPEAT (resolve_tac (prems@[wfrec_conv]) 1));
val wfrec_def_conv = result();
