(*  Title: 	HOL/univ
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Defines "Cartesian Product" and "Disjoint Sum" as set operations
within sexp.  Could <*> be generalized to a general summation?

*)

structure Univ =
struct
val sext = 
 Sext{mixfix = 
  [ Infixr("<*>", "[('a sexp)set, ('a sexp)set]=> ('a sexp)set", 80),
    Infixr("<+>", "[(('a+nat)sexp)set, (('a+nat)sexp)set]=> \
\                   (('a+nat)sexp)set",
	   70) ],
      parse_translation = [],
      print_translation = []};

val thy = extend_theory Sexp.thy "Univ"
 ([],[],[],[],
  [ 
    (["Atoms"],		"('a sexp)set"),  (**needed??*)
    (["ATOM"],      	"'a => ('a + 'b)sexp"),
    (["NUMB"],      	"nat => ('a+nat)sexp"),
    (["ATOMS"],		"(('a + 'b)sexp)set"),
    (["IN0","IN1"],	"('a+nat)sexp => ('a+nat)sexp")],
  Some sext)
 [
  ("uprod_def", 	"A<*>B == UN x:A. UN y:B. { (x.y) }"),
  ("usum_def", 		"A<+>B == (UN x:A. {IN0(x)}) Un (UN y:B. {IN1(y)})"),
  ("ATOM_def", 		"ATOM == (%a. Atom(Inl(a)))"),
  ("NUMB_def", 		"NUMB == (%k. Atom(Inr(k)))"),
  ("IN0_def", 		"IN0(M) == Atom(Inr(0)) . M"),
  ("IN1_def", 		"IN1(M) == Atom(Inr(Suc(0))) . M"),
  ("Atoms_def", 	"Atoms == range(Atom)"),
  ("ATOMS_def", 	"ATOMS == range(ATOM)")
 ];
end;

local val ax = get_axiom Univ.thy
in
val uprod_def =  ax"uprod_def";
val usum_def =   ax"usum_def";
val ATOM_def =  ax"ATOM_def";
val NUMB_def =  ax"NUMB_def";
val IN0_def =    ax"IN0_def";
val IN1_def =    ax"IN1_def";
val Atoms_def = ax"Atoms_def";
val ATOMS_def = ax"ATOMS_def";
end;


(*** Cartesian Product ***)

val prems = goalw Univ.thy [uprod_def]
    "[| a:A;  b:B |] ==> (a.b) : A<*>B";
by (REPEAT (resolve_tac(prems@[singletonI,UN_I]) 1));
val uprodI = result();

(*The general elimination rule*)
val major::prems = goalw Univ.thy [uprod_def]
    "[| c : A<*>B;  \
\       !!x y.[| x:A;  y:B;  c=x.y |] ==> P \
\    |] ==> P";
by (cut_facts_tac [major] 1);
by (REPEAT (eresolve_tac [asm_rl,singletonE,UN_E] 1
     ORELSE resolve_tac prems 1));
val uprodE = result();

(*Elimination of <a,b>:A*B -- introduces no eigenvariables*)
val prems = goal Univ.thy
    "[| (a.b) : A<*>B;    \
\       [| a:A;  b:B |] ==> P   \
\    |] ==> P";
by (rtac uprodE 1);
by (REPEAT (ares_tac prems 1 ORELSE eresolve_tac [Scons_inject,ssubst] 1));
val uprodE2 = result();

val prems = goalw Univ.thy [uprod_def] "[| A<=C;  B<=D |] ==> A<*>B <= C<*>D";
by (cfast_tac prems 1);
val uprod_mono = result();


(*** Atoms ***)

goalw Univ.thy [Atoms_def] "Atom(a) : Atoms";
by (rtac rangeI 1);
val AtomsI = result();

val [major,minor] = goalw Univ.thy [Atoms_def]
    "[| b: Atoms;  !!x. b=Atom(x) ==> P |] ==> P";
by (rtac (major RS rangeE) 1);
by (etac minor 1);
val AtomsE = result();

goalw Univ.thy [ATOM_def] "inj(ATOM)";
by (rtac injI 1);
by (etac (Atom_inject RS Inl_inject) 1);
val inj_ATOM = result();

val ATOM_inject = inj_ATOM RS injD;

goalw Univ.thy [NUMB_def] "inj(NUMB)";
by (rtac injI 1);
by (etac (Atom_inject RS Inr_inject) 1);
val inj_NUMB = result();

val NUMB_inject = inj_NUMB RS injD;


(*** Disjoint Sum ***)

(** introduction **)

val prems = goalw Univ.thy [usum_def] "a:A ==> IN0(a) : A<+>B";
by (fast_tac (set_cs addIs prems) 1);
val usum_IN0I = result();

val prems = goalw Univ.thy [usum_def] "b:B ==> IN1(b) : A<+>B";
by (fast_tac (set_cs addIs prems) 1);
val usum_IN1I = result();

(** elimination **)

val major::prems = goalw Univ.thy [usum_def]
    "[| u : A<+>B;  \
\       !!x. x:A ==> P(IN0(x)); \
\       !!y. y:B ==> P(IN1(y)) \
\    |] ==> P(u)";
by (rtac (major RS UnE) 1);
by (REPEAT (eresolve_tac (prems@[UN_E,singletonE,ssubst]) 1));
val usumE = result();


(*** Injection rules ***)

goalw Univ.thy [IN0_def,IN1_def] "~ (IN0(a) = IN1(b))";
by (rtac notI 1);
by (etac Scons_inject 1);
by (etac (Atom_inject RS Inr_inject RS Zero_neq_Suc) 1);
val IN0_not_IN1 = result();

val IN1_not_IN0 = standard (IN0_not_IN1 RS neg_sym);

val IN0_neq_IN1 = standard (IN0_not_IN1 RS notE);
val IN1_neq_IN0 = sym RS IN0_neq_IN1;

val [major] = goalw Univ.thy [IN0_def] "IN0(a) = IN0(b) ==>  a=b";
by (rtac (major RS Scons_inject) 1);
by (assume_tac 1);
val IN0_inject = result();

val [major] = goalw Univ.thy [IN1_def] "IN1(a) = IN1(b) ==>  a=b";
by (rtac (major RS Scons_inject) 1);
by (assume_tac 1);
val IN1_inject = result();

val prems = goalw Univ.thy [usum_def] "[| A<=C;  B<=D |] ==> A<+>B <= C<+>D";
by (cfast_tac prems 1);
val usum_mono = result();

