(*  Title: 	HOL/set
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Set theory for higher-order logic.  A set is simply a predicate.
*)

structure Set =
struct
val const_decs = 
 [
  (["Collect"],		"['a=>bool] => 'a set"),	(*comprehension*)
  (["Compl"], 		"('a set) => 'a set"),		(*complement*)
  (["Union", "Inter"],	"(('a set)set) => 'a set"),	(*...of a set*)
  (["UNION", "INTER"],	"['a set, 'a => 'b set] => 'b set"), (*general*)
  (["range"], 		"('a => 'b) => 'b set"),	(*range of function*)
  (["Ball","Bex"], 	"['a set, 'a=>bool]=>bool"),  	(*bounded quantifiers*)
  (["mono"], 		"['a set => 'b set] => bool"),	(*monotonicity*)
  (["inj","surj"], 	"('a => 'b) => bool"),      (*injective/surjective?*)
  (["inj_onto"], 	"['a => 'b, 'a set] => bool")
 ];

(** Bounded quantifiers have the form %x:A.P(x) **)
fun qnt_tr q [Free(x,T), A, P] = Const(q,dummyT) $ A $ absfree(x,T,P);

fun qnt_tr' q [A, Abs(x,T,P)] =
    let val (x',P') = variant_abs(x,T,P)
    in Const(q,dummyT) $ Free(x',T) $ A $ P' end;

val mixfix =
 [Delimfix("(1{_./ _})", "[ID,bool]=>'a set", "@Coll"),   (*collection*)
  Delimfix("{_}", "'a=> 'a set", "singleton"),
  Infixl("``",	"['a=>'b,'a set] => ('b set)",	90),
  Infixl("Int", "['a set,'a set]=> 'a set", 	70),
  Infixl("Un",	"['a set,'a set]=> 'a set", 	65),
  Infixl(":",	"['a,'a set]=>bool", 			50), (*membership*)
  Infixl("<=",	"['a set,'a set]=>bool", 		50),
      (*Big intersection/union*)
  Mixfix("(INT _:_./ _)", "[ID,'a set,'b set]=> 'b set", 
	 "@INTER", [], 10),
  Mixfix("(UN _:_./ _)", "[ID,'a set,'b set]=> 'b set", 
	 "@UNION",  [], 10),
      (*Bounded quantifiers*)
  Mixfix("(! _:_./ _)",		"[ID,'a set,bool]=>bool", "@Ball", [], 10),
  Mixfix("(ALL _:_./ _)", 	"[ID,'a set,bool]=>bool", "*Ball", [], 10),
  Mixfix("(? _:_./ _)",  	"[ID,'a set,bool]=>bool", "@Bex",  [], 10),
  Mixfix("(EX _:_./ _)",  	"[ID,'a set,bool]=>bool", "*Bex",  [], 10)
];

val sext = Sext{
  mixfix=mixfix,
  parse_translation=
    [mk_binder_tr("@Coll","Collect"),
     ("@INTER", qnt_tr "INTER"),
     ("@UNION", qnt_tr "UNION"),
     ("@Ball",  qnt_tr "Ball"),
     ("*Ball",  qnt_tr "Ball"),
     ("@Bex",   qnt_tr "Bex"),
     ("*Bex",   qnt_tr "Bex") ],
  print_translation=
    [mk_binder_tr'("Collect","@Coll"),
     ("INTER",  qnt_tr' "@INTER"),
     ("UNION",  qnt_tr' "@UNION"),
     ("Ball",   alt_tr' (qnt_tr' "@Ball", qnt_tr' "*Ball")),
     ("Bex",    alt_tr' (qnt_tr' "@Bex",  qnt_tr' "*Bex")) ]};

val thy = extend_theory HOL_Rule.thy "Set"
 ([], [], [], 
  [(["set"],([["term"]],"term"))],
  const_decs, Some(sext))
[
  ("mem_Collect_eq",	"(a : {x.P(x)}) = P(a)" ),
  ("Collect_mem_eq",	"{x.x:A} = A" ),

  ("Ball_def",		"Ball(A,P)  == ! x. x:A --> P(x)" ),
  ("Bex_def",		"Bex(A,P)   == ? x. x:A & P(x)" ),
  ("subset_def", 	"A <= B     == ! x:A. x:B" ),
  ("singleton_def",   	"{a}        == {x.x=a}" ),
  ("Un_def",		"A Un B     == {x.x:A | x:B}" ),
  ("Int_def",		"A Int B    == {x.x:A & x:B}" ),
  ("Compl_def",		"Compl(A)   == {x. ~x:A}" ),
  ("INTER_def",		"INTER(A,B) == {y. ! x:A. y: B(x)}" ),
  ("UNION_def",		"UNION(A,B) == {y. ? x:A. y: B(x)}" ),
  ("Inter_def",		"Inter(S)   == (INT x:S. x)" ),
  ("Union_def",		"Union(S)   ==  (UN x:S. x)" ),
  ("image_def",  	"f``A       == {y. ? x:A. y=f(x)}" ),
  ("range_def",		"range(f)   == {y. ? x. y=f(x)}" ),
  ("mono_def",		"mono(f)    == (!A B. A <= B --> f(A) <= f(B))" ),

  ("inj_def",	"inj(f) == ! x y. f(x)=f(y) --> x=y" ),

  ("inj_onto_def",	
   "inj_onto(f,A) == ! x:A. ! y:A. f(x)=f(y) --> x=y" ),

  ("surj_def",		"surj(f) == ! y. ? x. y=f(x)" ) ];

end;

local val ax = get_axiom Set.thy
in
val mem_Collect_eq = ax"mem_Collect_eq";
val Collect_mem_eq = ax"Collect_mem_eq";
val Ball_def    = ax"Ball_def";
val Bex_def     = ax"Bex_def";
val subset_def  = ax"subset_def";
val Un_def      = ax"Un_def";
val Int_def     = ax"Int_def";
val Compl_def   = ax"Compl_def";
val Inter_def   = ax"Inter_def";
val Union_def   = ax"Union_def";
val INTER_def   = ax"INTER_def";
val UNION_def   = ax"UNION_def";
val mono_def    = ax"mono_def";
val image_def   = ax"image_def";
val singleton_def    = ax"singleton_def";
val range_def   = ax"range_def";
val inj_def = ax"inj_def";
val inj_onto_def      = ax"inj_onto_def";
val surj_def    = ax"surj_def";
end;

val [prem] = goal Set.thy "[| P(a) |] ==> a : {x.P(x)}";
by (rtac (mem_Collect_eq RS ssubst) 1);
by (rtac prem 1);
val CollectI = result();

val prems = goal Set.thy "[| a : {x.P(x)} |] ==> P(a)";
by (resolve_tac (prems RL [mem_Collect_eq  RS subst]) 1);
val CollectD = result();

val [prem] = goal Set.thy "[| !!x. (x:A) = (x:B) |] ==> A = B";
by (rtac (prem RS abs RS arg_cong RS box_equals) 1);
by (rtac Collect_mem_eq 1);
by (rtac Collect_mem_eq 1);
val set_ext = result();

val [prem] = goal Set.thy "[| !!x. P(x)=Q(x) |] ==> {x. P(x)} = {x. Q(x)}";
by (rtac (prem RS abs RS arg_cong) 1);
val Collect_cong = result();

val CollectE = make_elim CollectD;

(*** Bounded quantifiers ***)

val prems = goalw Set.thy [Ball_def]
    "[| !!x. x:A ==> P(x) |] ==> ! x:A. P(x)";
by (REPEAT (ares_tac (prems @ [allI,impI]) 1));
val ballI = result();

val [major,minor] = goalw Set.thy [Ball_def]
    "[| ! x:A. P(x);  x:A |] ==> P(x)";
by (rtac (minor RS (major RS spec RS mp)) 1);
val bspec = result();

val major::prems = goalw Set.thy [Ball_def]
    "[| ! x:A. P(x);  P(x) ==> Q;  ~ x:A ==> Q |] ==> Q";
by (rtac (major RS spec RS impCE) 1);
by (REPEAT (eresolve_tac prems 1));
val ballE = result();

(*Takes assumptions ! x:A.P(x) and a:A; creates assumption P(a)*)
fun ball_tac i = etac ballE i THEN contr_tac (i+1);

val prems = goalw Set.thy [Bex_def]
    "[| P(x);  x:A |] ==> ? x:A. P(x)";
by (REPEAT (ares_tac (prems @ [exI,conjI]) 1));
val bexI = result();

val bexCI = prove_goal Set.thy 
   "[| ! x:A. ~P(x) ==> P(a);  a:A |] ==> ? x:A.P(x)"
 (fn prems=>
  [ (rtac classical 1),
    (REPEAT (ares_tac (prems@[bexI,ballI,notI,notE]) 1))  ]);

val major::prems = goalw Set.thy [Bex_def]
    "[| ? x:A. P(x);  !!x. [| x:A; P(x) |] ==> Q  |] ==> Q";
by (rtac (major RS exE) 1);
by (REPEAT (eresolve_tac (prems @ [asm_rl,conjE]) 1));
val bexE = result();

(*Trival rewrite rule;   (! x:A.P)=P holds only if A is nonempty!*)
val prems = goal Set.thy
    "(! x:A. True) = True";
by (REPEAT (ares_tac [TrueI,ballI,iffI] 1));
val ball_rew = result();

(** Congruence rules **)

val prems = goal Set.thy
    "[| A=A';  !!x. x:A' ==> P(x) = P'(x) |] ==> \
\    (! x:A. P(x)) = (! x:A'. P'(x))";
by (resolve_tac (prems RL [ssubst]) 1);
by (REPEAT (ares_tac [ballI,iffI] 1
     ORELSE eresolve_tac ([make_elim bspec, mp] @ (prems RL [iffE])) 1));
val ball_cong = result();

val prems = goal Set.thy
    "[| A=A';  !!x. x:A' ==> P(x) = P'(x) |] ==> \
\    (? x:A. P(x)) = (? x:A'. P'(x))";
by (resolve_tac (prems RL [ssubst]) 1);
by (REPEAT (etac bexE 1
     ORELSE ares_tac ([bexI,iffI] @ (prems RL [iffD1,iffD2])) 1));
val bex_cong = result();

(*** Rules for subsets ***)

val prems = goalw Set.thy [subset_def] "(!!x.x:A ==> x:B) ==> A <= B";
by (REPEAT (ares_tac (prems @ [ballI]) 1));
val subsetI = result();

(*Rule in Modus Ponens style*)
val major::prems = goalw Set.thy [subset_def] "[| A <= B;  c:A |] ==> c:B";
by (rtac (major RS bspec) 1);
by (resolve_tac prems 1);
val subsetD = result();

(*Classical elimination rule*)
val major::prems = goalw Set.thy [subset_def] 
    "[| A <= B;  ~(c:A) ==> P;  c:B ==> P |] ==> P";
by (rtac (major RS ballE) 1);
by (REPEAT (eresolve_tac prems 1));
val subsetCE = result();

(*Takes assumptions A<=B; c:A and creates the assumption c:B *)
fun set_mp_tac i = etac subsetCE i  THEN  mp_tac i;

val subset_refl = prove_goal Set.thy "A <= A"
 (fn _=> [ (REPEAT (ares_tac [subsetI] 1)) ]);

val prems = goal Set.thy "[| A<=B;  B<=C |] ==> A<=C";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [subsetI] 1 ORELSE set_mp_tac 1));
val subset_trans = result();


(*** Rules for equality ***)

(*Anti-symmetry of the subset relation*)
val prems = goal Set.thy "[| A <= B;  B <= A |] ==> A = B";
by (rtac (iffI RS set_ext) 1);
by (REPEAT (ares_tac (prems RL [subsetD]) 1));
val subset_antisym = result();
val equalityI = subset_antisym;

(* Equality rules from ZF set theory -- are they appropriate here? *)
val prems = goal Set.thy "A = B ==> A<=B";
by (resolve_tac (prems RL [subst]) 1);
by (rtac subset_refl 1);
val equalityD1 = result();

val prems = goal Set.thy "A = B ==> B<=A";
by (resolve_tac (prems RL [subst]) 1);
by (rtac subset_refl 1);
val equalityD2 = result();

val prems = goal Set.thy
    "[| A = B;  [| A<=B; B<=A |] ==> P |]  ==>  P";
by (resolve_tac prems 1);
by (REPEAT (resolve_tac (prems RL [equalityD1,equalityD2]) 1));
val equalityE = result();

val major::prems = goal Set.thy
    "[| A = B;  [| c:A; c:B |] ==> P;  [| ~ c:A; ~ c:B |] ==> P |]  ==>  P";
by (rtac (major RS equalityE) 1);
by (REPEAT (contr_tac 1 ORELSE eresolve_tac ([asm_rl,subsetCE]@prems) 1));
val equalityCE = result();

(*Lemma for creating induction formulae -- for "pattern matching" on p
  To make the induction hypotheses usable, apply "spec" or "bspec" to
  put universal quantifiers over the free variables in p. *)
val prems = goal Set.thy 
    "[| p:A;  !!z. z:A ==> p=z --> R |] ==> R";
by (rtac mp 1);
by (REPEAT (resolve_tac (refl::prems) 1));
val setup_induction = result();


(*** Rules for binary union -- Un ***)

val prems = goalw Set.thy [Un_def] "c:A ==> c : A Un B";
by (REPEAT (resolve_tac (prems @ [CollectI,disjI1]) 1));
val UnI1 = result();

val prems = goalw Set.thy [Un_def] "c:B ==> c : A Un B";
by (REPEAT (resolve_tac (prems @ [CollectI,disjI2]) 1));
val UnI2 = result();

(*Classical introduction rule: no commitment to A vs B*)
val UnCI = prove_goal Set.thy "(~c:B ==> c:A) ==> c : A Un B"
 (fn prems=>
  [ (rtac classical 1),
    (REPEAT (ares_tac (prems@[UnI1,notI]) 1)),
    (REPEAT (ares_tac (prems@[UnI2,notE]) 1)) ]);

val major::prems = goalw Set.thy [Un_def]
    "[| c : A Un B;  c:A ==> P;  c:B ==> P |] ==> P";
by (rtac (major RS CollectD RS disjE) 1);
by (REPEAT (eresolve_tac prems 1));
val UnE = result();


(*** Rules for small intersection -- Int ***)

val prems = goalw Set.thy [Int_def]
    "[| c:A;  c:B |] ==> c : A Int B";
by (REPEAT (resolve_tac (prems @ [CollectI,conjI]) 1));
val IntI = result();

val [major] = goalw Set.thy [Int_def] "c : A Int B ==> c:A";
by (rtac (major RS CollectD RS conjunct1) 1);
val IntD1 = result();

val [major] = goalw Set.thy [Int_def] "c : A Int B ==> c:B";
by (rtac (major RS CollectD RS conjunct2) 1);
val IntD2 = result();

val [major,minor] = goal Set.thy
    "[| c : A Int B;  [| c:A; c:B |] ==> P |] ==> P";
by (rtac minor 1);
by (rtac (major RS IntD1) 1);
by (rtac (major RS IntD2) 1);
val IntE = result();


(*** Rules for set complement -- Compl ***)

val prems = goalw Set.thy [Compl_def]
    "[| c:A ==> False |] ==> c : Compl(A)";
by (REPEAT (ares_tac (prems @ [CollectI,notI]) 1));
val ComplI = result();

(*This form, with negated conclusion, works well with the Classical prover.
  Negated assumptions behave like formulae on the right side of the notional
  turnstile...*)
val major::prems = goalw Set.thy [Compl_def]
    "[| c : Compl(A) |] ==> ~c:A";
by (rtac (major RS CollectD) 1);
val ComplD = result();

val ComplE = make_elim ComplD;


(*** Singleton sets ***)

goalw Set.thy [singleton_def] "a : {a}";
by (rtac CollectI 1);
by (rtac refl 1);
val singletonI = result();

val [major] = goalw Set.thy [singleton_def] "b : {a} ==> b=a"; 
by (rtac (major RS CollectD) 1);
val singletonD = result();

val singletonE = make_elim singletonD;

(*** Unions of families ***)

(*The order of the premises presupposes that A is rigid; b may be flexible*)
val prems = goalw Set.thy [UNION_def]
    "[| a:A;  b: B(a) |] ==> b: (UN x:A. B(x))";
by (REPEAT (resolve_tac (prems @ [bexI,CollectI]) 1));
val UN_I = result();

val major::prems = goalw Set.thy [UNION_def]
    "[| b : (UN x:A. B(x));  !!x.[| x:A;  b: B(x) |] ==> R |] ==> R";
by (rtac (major RS CollectD RS bexE) 1);
by (REPEAT (ares_tac prems 1));
val UN_E = result();


(*** Intersections of families -- INTER x:A. B(x) is Inter(B)``A ) *)

val prems = goalw Set.thy [INTER_def]
    "(!!x. x:A ==> b: B(x)) ==> b : (INT x:A. B(x))";
by (REPEAT (ares_tac ([CollectI,ballI] @ prems) 1));
val INT_I = result();

val major::prems = goalw Set.thy [INTER_def]
    "[| b : (INT x:A. B(x));  a:A |] ==> b: B(a)";
by (rtac (major RS CollectD RS bspec) 1);
by (resolve_tac prems 1);
val INT_D = result();

(*"Classical" elimination rule -- does not require proving X:C *)
val major::prems = goalw Set.thy [INTER_def]
    "[| b : (INT x:A. B(x));  b: B(a) ==> R;  ~ a:A ==> R |] ==> R";
by (rtac (major RS CollectD RS ballE) 1);
by (REPEAT (eresolve_tac prems 1));
val INT_E = result();

(*** Rules for Unions ***)

(*The order of the premises presupposes that C is rigid; A may be flexible*)
val prems = goalw Set.thy [Union_def]
    "[| X:C;  A:X |] ==> A : Union(C)";
by (REPEAT (resolve_tac (prems @ [UN_I]) 1));
val UnionI = result();

val major::prems = goalw Set.thy [Union_def]
    "[| A : Union(C);  !!X.[| A:X;  X:C |] ==> R |] ==> R";
by (rtac (major RS UN_E) 1);
by (REPEAT (ares_tac prems 1));
val UnionE = result();

(*** Rules for Inter ***)

val prems = goalw Set.thy [Inter_def]
    "[| !!X. X:C ==> A:X |] ==> A : Inter(C)";
by (REPEAT (ares_tac ([INT_I] @ prems) 1));
val InterI = result();

(*A "destruct" rule -- every X in C contains A as an element, but
  A:X can hold when X:C does not!  This rule is analogous to "spec". *)
val major::prems = goalw Set.thy [Inter_def]
    "[| A : Inter(C);  X:C |] ==> A:X";
by (rtac (major RS INT_D) 1);
by (resolve_tac prems 1);
val InterD = result();

(*"Classical" elimination rule -- does not require proving X:C *)
val major::prems = goalw Set.thy [Inter_def]
    "[| A : Inter(C);  A:X ==> R;  ~ X:C ==> R |] ==> R";
by (rtac (major RS INT_E) 1);
by (REPEAT (eresolve_tac prems 1));
val InterE = result();

