(*  Title: 	HOL/ex/arith
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

Arithmetic operators and their definitions

Proofs about elementary arithmetic: addition, multiplication, etc.
Tests definitions and simplifier.
*)

writeln"File HOL/ex/arith";

structure Arith =
struct
val mixfix = 
 [
  Infixl("+",	"[nat,nat]=>nat", 65),
  Infixl("-",	"[nat,nat]=>nat", 65),
  Infixl("*",	"[nat,nat]=>nat", 70),
  Infixl("'/'/","[nat,nat]=>nat", 70),
  Infixl("'/",	"[nat,nat]=>nat", 70)
 ];

val arith_sext = Sext{mixfix=mixfix,
		      parse_translation=[],
		      print_translation=[]};

val thy = extend_theory Nat.thy  "arith" 
    ([], [], [], [], [], Some arith_sext)
  [ 
    ("add_def",   "m+n == nat_rec(m, n, %u v.Suc(v))"),  
    ("diff_def",  "m-n == nat_rec(n, m, %u v. nat_rec(v, 0, %x y.x))"),  
    ("mult_def",  "m*n == nat_rec(m, 0, %u v. n + v)"),  
    ("mod_def",   
     "m//n == wfrec(trancl(pred_nat), m, %j f. if(j<n, j, f(j-n)))"),  
    ("quo_def",   
     "m/n == wfrec(trancl(pred_nat), m, %j f. if(j<n, 0, Suc(f(j-n))))") ];
end;

(*"Difference" is subtraction of natural numbers.
  There are no negative numbers; we have
     m - n = 0  iff  m<=n   and     m - n = Suc(k) iff m>n.
  Also, nat_rec(m, 0, %z w.z) is pred(m).   *)

local val ax = get_axiom Arith.thy in  
val add_def = ax"add_def";
val diff_def = ax"diff_def";
val mult_def = ax"mult_def";
val mod_def = ax"mod_def";
val quo_def = ax"quo_def";
end;


val nat_ss = 
  HOL_ss addcongs (mk_congs Nat.thy ["Suc","nat_case","nat_rec", "op <"])
         addrews [Suc_not_Zero, Zero_not_Suc, Suc_Suc_eq,
		  nat_case_0_conv, nat_case_Suc_conv,
		  nat_rec_0_conv, nat_rec_Suc_conv];

(*** Basic rewrite and congruence rules for the arithmetic operators ***)

(** Addition **)

val add_0_conv = add_def RS nat_rec_def_0_conv;
val add_Suc_conv = add_def RS nat_rec_def_Suc_conv; 

(** Multiplication **)

val mult_0_conv = standard (mult_def RS nat_rec_def_0_conv);
val mult_Suc_conv = standard (mult_def RS nat_rec_def_Suc_conv); 

(** Difference **)

val diff_0_conv = diff_def RS nat_rec_def_0_conv;

val diff_0_eq_0 = prove_goalw Arith.thy [diff_def] "0 - n = 0"
 (fn _ => [ (nat_ind_tac "n" 1), (ALLGOALS (ASM_SIMP_TAC nat_ss)) ]);

(*Must simplify BEFORE the induction!!  (Else we get a critical pair)
  Suc(m) - Suc(n)   rewrites to   pred(Suc(m) - n)  *)
val diff_Suc_Suc = prove_goalw Arith.thy [diff_def] "Suc(m) - Suc(n) = m - n"
 (fn _ =>
  [ (ASM_SIMP_TAC nat_ss 1),
    (nat_ind_tac "n" 1),
    (ALLGOALS (ASM_SIMP_TAC nat_ss)) ]);

goal Arith.thy "m - n < Suc(m)";
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (etac less_SucE 3);
by (all_simp_tac nat_ss [diff_0_conv,diff_0_eq_0,diff_Suc_Suc,
			 lessI,less_SucI]);
val diff_leq = result();

(*** Simplification over add, mult, diff ***)

val arith_rews = [add_0_conv, add_Suc_conv,
		  mult_0_conv, mult_Suc_conv,
		  diff_0_conv, diff_0_eq_0, diff_Suc_Suc];

val arith_congs = mk_congs Arith.thy ["op +", "op -", "op *"];

val arith_ss = nat_ss addcongs arith_congs addrews  arith_rews;


(**** Inductive properties of the operators ****)

(*** Addition ***)

val add_right0 = prove_goal Arith.thy "m + 0 = m"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss))  ]);

val add_right_Suc = prove_goal Arith.thy "m + Suc(n) = Suc(m+n)"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss))  ]);

val arith_ss = arith_ss addrews [add_right0,add_right_Suc];

(*Associative law for addition*)
val add_assoc = prove_goal Arith.thy "(m + n) + k = m + (n + k)"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss)) ]);

(*Commutative law for addition.  Must simplify after first induction!
  Orientation of rewrites is delicate*)  
val add_commute = prove_goal Arith.thy "m + n = n + m"
 (fn _ =>
  [ (nat_ind_tac "m" 1),
    (ALLGOALS (ASM_SIMP_TAC arith_ss))]);

(*** Multiplication ***)

(*right annihilation in product*)
val mult_right0 = prove_goal Arith.thy "m * 0 = 0"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss))  ]);

(*right Sucessor law for multiplication*)
val mult_right_Suc = prove_goal Arith.thy 
    "m * Suc(n) = m + (m * n)"
 (fn _ =>
  [ (nat_ind_tac "m" 1),
    (all_simp_tac arith_ss ([add_assoc RS sym])),
       (*The final goal requires the commutative law for addition*)
    (REPEAT (ares_tac ([refl,add_commute]@arith_congs) 1))  ]);

val arith_ss = arith_ss addrews [mult_right0,mult_right_Suc];

(*Commutative law for multiplication*)
val mult_commute = prove_goal Arith.thy "m * n = n * m"
 (fn _ => [ (nat_ind_tac "m" 1), (all_simp_tac arith_ss []) ]);

(*addition distributes over multiplication*)
val add_mult_dist = prove_goal Arith.thy "(m + n) * k = (m * k) + (n * k)"
 (fn _ =>
  [ (nat_ind_tac "m" 1),
    (all_simp_tac arith_ss [add_assoc RS sym]) ]);

(*Associative law for multiplication*)
val mult_assoc = prove_goal Arith.thy "(m * n) * k = m * (n * k)"
 (fn _ => [ (nat_ind_tac "m" 1), (all_simp_tac arith_ss [add_mult_dist]) ]);


(*** Difference ***)

val diff_self_eq_0 = prove_goal Arith.thy "m - m = 0"
 (fn _ => [ (nat_ind_tac "m" 1), (ALLGOALS (ASM_SIMP_TAC arith_ss)) ]);

(*Addition is the inverse of subtraction: if n<=m then n+(m-n) = m. *)
val [prem] = goal Arith.thy "[| ~ m<n |] ==> n + (m-n) = m";
by (rtac (prem RS rev_mp) 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (all_simp_tac arith_ss [zero_less_Suc, Suc_less_eq]);
val add_inverse_diff = result();


(*** Remainder ***)

(*In ordinary notation: if 0<n and n<=m then m-n < m *)
val prems = goal Arith.thy "[| 0<n; ~ m<n |] ==> m - n < m";
by (subgoal_tac "0<n --> ~ m<n --> m - n < m" 1);
by (cfast_tac prems 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (all_simp_tac nat_ss (prems@[less_not_refl,diff_leq,diff_Suc_Suc]));
val quorem_termination = result();

val wf_less_trans = wf_pred_nat RS wf_trancl RSN (2, wfrec_def_conv RS trans);

goalw Nat.thy [less_def] "<m,n> : trancl(pred_nat) = (m<n)";
by (rtac refl 1);
val less_eq = result();

val quorem_ss = nat_ss addrews [quorem_termination, cut_apply, less_eq];

val prems = goal Arith.thy "m<n ==> m//n = m";
by (rtac (mod_def RS wf_less_trans) 1);
by (SIMP_TAC (nat_ss addrews prems) 1);
val mod_less_conv = result();

val prems = goal Arith.thy "[| 0<n;  ~m<n |] ==> m//n = (m-n)//n";
by (rtac (mod_def RS wf_less_trans) 1);
by (SIMP_TAC (quorem_ss addrews prems) 1);
val mod_geq_conv = result();


(*** Quotient ***)

val prems = goal Arith.thy "m<n ==> m/n = 0";
by (rtac (quo_def RS wf_less_trans) 1);
by (SIMP_TAC (nat_ss addrews prems) 1);
val quo_less_conv = result();

val prems = goal Arith.thy "[| 0<n;  ~m<n |] ==> m/n = Suc((m-n)/n)";
by (rtac (quo_def RS wf_less_trans) 1);
by (SIMP_TAC (quorem_ss addrews prems) 1);
val quo_geq_conv = result();

(*Main Result about quotient and remainder.*)
val prems = goal Arith.thy "[| 0<n |] ==> (m/n)*n + m//n = m";
by (res_inst_tac [("n","m")] less_induct 1);
by (rename_tac "k" 1);    (*Variable name used in line below*)
by (res_inst_tac [("Q","k<n")] (excluded_middle RS disjE) 1);
by (all_simp_tac arith_ss (prems @
        [mod_less_conv, mod_geq_conv, quo_less_conv, quo_geq_conv,
	 add_assoc, add_inverse_diff, quorem_termination]));
val mod_quo_equality = result();


(*** More results about difference ***)

val [prem] = goal Arith.thy "m < Suc(n) ==> m-n = 0";
by (rtac (prem RS rev_mp) 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (SIMP_TAC (arith_ss addrews [not_less0,less_Suc_eq]) 1);
by (SIMP_TAC arith_ss 1);
by (ASM_SIMP_TAC (arith_ss addrews [Suc_less_eq]) 1);
val less_imp_diff_is_0 = result();

val prems = goal Arith.thy "m-n = 0  -->  n-m = 0  -->  m=n";
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (ALLGOALS (SIMP_TAC arith_ss));
val diffs0_imp_equal_lemma = result();

(*  [| m-n = 0;  n-m = 0 |] ==> m=n  *)
val diffs0_imp_equal = standard (diffs0_imp_equal_lemma RS mp RS mp);

val [prem] = goal Arith.thy "m<n ==> 0<n-m";
by (rtac (prem RS rev_mp) 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (SIMP_TAC (arith_ss addrews [not_less0]) 1);
by (SIMP_TAC arith_ss 1);
by (ASM_SIMP_TAC (arith_ss addrews [Suc_less_eq]) 1);
val less_imp_diff_positive = result();

val [prem] = goal Arith.thy "n < Suc(m) ==> Suc(m)-n = Suc(m-n)";
by (rtac (prem RS rev_mp) 1);
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (SIMP_TAC arith_ss 1);
by (SIMP_TAC (arith_ss addrews [Suc_less_eq,not_less0]) 1);
by (ASM_SIMP_TAC (arith_ss addrews [Suc_less_eq]) 1);
val Suc_diff_n = result();

goal Arith.thy "Suc(m)-n = if(m<n, 0, Suc(m-n))";
by (SIMP_CASE_TAC (arith_ss addrews [Suc_less_eq, less_imp_diff_is_0, 
				     not_less_eq, Suc_diff_n]) 1);
val if_Suc_diff_n = result();

val ss1 = arith_ss addcongs [read_instantiate [("f","P")] arg_cong];

goal Arith.thy "P(k) --> (!n. P(Suc(n))--> P(n)) --> P(k-i)";
by (res_inst_tac [("m","k"),("n","i")] diff_induct 1);
by (ALLGOALS (strip_tac THEN' SIMP_TAC ss1));
by (fast_tac HOL_cs 1);
val zero_induct_lemma = result();

val prems = goal Arith.thy "[| P(k);  !!n. P(Suc(n)) ==> P(n) |] ==> P(0)";
by (rtac (diff_self_eq_0 RS subst) 1);
by (rtac (zero_induct_lemma RS mp RS mp) 1);
by (REPEAT (ares_tac ([impI,allI]@prems) 1));
val zero_induct = result();


writeln"Reached end of file.";

(*13 July 1992: loaded in 105.7s*)
