(*  Title: CTT/ruleshell
    Author: Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Constructive Type Theory rules

!!!After updating, rebuild  ".rules.ML"  by calling make-rulenames!!! 

Equality forms of rules that follow by reflexivity are omitted.
*)

signature CTT_RULE =
  sig
  val sign: Sign.sg
  val thy: theory
(*INSERT-RULESIG -- file produced by make-rulenames*)
  val refl_red: thm
  val red_if_equal: thm
  val trans_red: thm
  val refl_type: thm
  val refl_elem: thm
  val sym_type: thm
  val sym_elem: thm
  val trans_type: thm
  val trans_elem: thm
  val equal_types: thm
  val equal_typesL: thm
  val subst_type: thm
  val subst_typeL: thm
  val subst_elem: thm
  val subst_elemL: thm
  val NF: thm
  val NI0: thm
  val NI_succ: thm
  val NI_succL: thm
  val NE: thm
  val NEL: thm
  val NC0: thm
  val NC_succ: thm
  val zero_ne_succ: thm
  val ProdF: thm
  val ProdFL: thm
  val ProdI: thm
  val ProdIL: thm
  val ProdE: thm
  val ProdEL: thm
  val ProdC: thm
  val ProdC2: thm
  val SumF: thm
  val SumFL: thm
  val SumI: thm
  val SumIL: thm
  val SumE: thm
  val SumEL: thm
  val SumC: thm
  val fst_def: thm
  val snd_def: thm
  val PlusF: thm
  val PlusFL: thm
  val PlusI_inl: thm
  val PlusI_inlL: thm
  val PlusI_inr: thm
  val PlusI_inrL: thm
  val PlusE: thm
  val PlusEL: thm
  val PlusC_inl: thm
  val PlusC_inr: thm
  val EqF: thm
  val EqFL: thm
  val EqI: thm
  val EqE: thm
  val EqC: thm
  val FF: thm
  val FE: thm
  val FEL: thm
  val TF: thm
  val TI: thm
  val TE: thm
  val TEL: thm
  val TC: thm
  end;


functor CTT_RuleFun (CTT_Syntax: CTT_SYNTAX) : CTT_RULE = 
struct

val thy = extend_theory pure_thy "CTT"
( [],
  [],
  [],
  [(["i","t","o"], ([],"logic"))],
  CTT_Syntax.const_decs,
  Some CTT_Syntax.sext)
[ 
  (*Reduction: a weaker notion than equality;  a hack for simplification.
    Reduce(a,b) means either that  a=b:A  for some A or else that "a" and "b" 
    are textually identical.*)

  (*does not verify a:A!  Sound because only trans_red uses a Reduce premise
    No new theorems can be proved about the standard judgements.*)
  ("refl_red", "Reduce(a,a)"),

  ("red_if_equal", "a = b : A ==> Reduce(a,b)"),

  ("trans_red", "[| a = b : A;  Reduce(b,c) |] ==> a = c : A"),

  (*Reflexivity*)

  ("refl_type", "A type ==> A = A"), 
  ("refl_elem", "a : A ==> a = a : A"),

  (*Symmetry*)

  ("sym_type",  "A = B ==> B = A"),
  ("sym_elem",  "a = b : A ==> b = a : A"),

  (*Transitivity*)

  ("trans_type",   "[| A = B;  B = C |] ==> A = C"),
  ("trans_elem",   "[| a = b : A;  b = c : A |] ==> a = c : A"),

  ("equal_types",  "[| a : A;  A = B |] ==> a : B"),
  ("equal_typesL", "[| a = b : A;  A = B |] ==> a = b : B"),

  (*Substitution*)

  ("subst_type",   "[| a : A;  !!z. z:A ==> B(z) type |] ==> B(a) type"),
  ("subst_typeL",
    "[| a = c : A;  !!z. z:A ==> B(z) = D(z) |] ==> B(a) = D(c)"),

  ("subst_elem",   "[| a : A;  !!z. z:A ==> b(z):B(z) |] ==> b(a):B(a)"),
  ("subst_elemL",
    "[| a=c : A;  !!z. z:A ==> b(z)=d(z) : B(z) |] ==> b(a)=d(c) : B(a)"),


  (*The type N -- natural numbers*)

  ("NF", "N type"),
  ("NI0", "0 : N"),
  ("NI_succ", "a : N ==> succ(a) : N"),
  ("NI_succL",  "a = b : N ==> succ(a) = succ(b) : N"),

  ("NE",
   "[| p: N;  a: C(0);  !!u v. [| u: N; v: C(u) |] ==> b(u,v): C(succ(u)) |] \
\   ==> rec(p, a, %u v.b(u,v)) : C(p)"),

  ("NEL",
   "[| p = q : N;  a = c : C(0);  \
\      !!u v. [| u: N; v: C(u) |] ==> b(u,v) = d(u,v): C(succ(u)) |] \
\   ==> rec(p, a, %u v.b(u,v)) = rec(q,c,d) : C(p)"),

  ("NC0",
   "[| a: C(0);  !!u v. [| u: N; v: C(u) |] ==> b(u,v): C(succ(u)) |] \
\   ==> rec(0, a, %u v.b(u,v)) = a : C(0)"),

  ("NC_succ",
   "[| p: N;  a: C(0);  \
\       !!u v. [| u: N; v: C(u) |] ==> b(u,v): C(succ(u)) |] ==>  \
\   rec(succ(p), a, %u v.b(u,v)) = b(p, rec(p, a, %u v.b(u,v))) : C(succ(p))"),

  (*The fourth Peano axiom.  See page 91 of Martin-Lof's book*)
  ("zero_ne_succ",
    "[| a: N;  0 = succ(a) : N |] ==> 0: F"),


  (*The Product of a family of types*)

  ("ProdF",  "[| A type; !!x. x:A ==> B(x) type |] ==> PROD x:A.B(x) type"),

  ("ProdFL",
   "[| A = C;  !!x. x:A ==> B(x) = D(x) |] ==> \
\   PROD x:A.B(x) = PROD x:C.D(x)"),

  ("ProdI", 
   "[| A type;  !!x. x:A ==> b(x):B(x)|] ==> lam x.b(x) : PROD x:A.B(x)"),

  ("ProdIL",
   "[| A type;  !!x. x:A ==> b(x) = c(x) : B(x)|] ==> \
\   lam x.b(x) = lam x.c(x) : PROD x:A.B(x)"),

  ("ProdE",  "[| p : PROD x:A.B(x);  a : A |] ==> p`a : B(a)"),
  ("ProdEL", "[| p=q: PROD x:A.B(x);  a=b : A |] ==> p`a = q`b : B(a)"),

  ("ProdC",
   "[| a : A;  !!x. x:A ==> b(x) : B(x)|] ==> \
\   (lam x.b(x)) ` a = b(a) : B(a)"),

  ("ProdC2",
   "p : PROD x:A.B(x) ==> (lam x. p`x) = p : PROD x:A.B(x)"),
 

  (*The Sum of a family of types*)

  ("SumF",  "[| A type;  !!x. x:A ==> B(x) type |] ==> SUM x:A.B(x) type"),
  ("SumFL", 
    "[| A = C;  !!x. x:A ==> B(x) = D(x) |] ==> SUM x:A.B(x) = SUM x:C.D(x)"),

  ("SumI",  "[| a : A;  b : B(a) |] ==> <a,b> : SUM x:A.B(x)"),
  ("SumIL", "[| a=c:A;  b=d:B(a) |] ==> <a,b> = <c,d> : SUM x:A.B(x)"),

  ("SumE",
    "[| p: SUM x:A.B(x);  !!x y. [| x:A; y:B(x) |] ==> c(x,y): C(<x,y>) |] \
\    ==> split(p, %x y.c(x,y)) : C(p)"),

  ("SumEL",
    "[| p=q : SUM x:A.B(x); \
\       !!x y. [| x:A; y:B(x) |] ==> c(x,y)=d(x,y): C(<x,y>)|] \
\    ==> split(p, %x y.c(x,y)) = split(q, % x y.d(x,y)) : C(p)"),

  ("SumC",
    "[| a: A;  b: B(a);  !!x y. [| x:A; y:B(x) |] ==> c(x,y): C(<x,y>) |] \
\    ==> split(<a,b>, %x y.c(x,y)) = c(a,b) : C(<a,b>)"),

  ("fst_def",   "fst(a) == split(a, %x y.x)"),
  ("snd_def",   "snd(a) == split(a, %x y.y)"),


  (*The sum of two types*)

  ("PlusF",   "[| A type;  B type |] ==> A+B type"),
  ("PlusFL",  "[| A = C;  B = D |] ==> A+B = C+D"),

  ("PlusI_inl",   "[| a : A;  B type |] ==> inl(a) : A+B"),
  ("PlusI_inlL", "[| a = c : A;  B type |] ==> inl(a) = inl(c) : A+B"),

  ("PlusI_inr",   "[| A type;  b : B |] ==> inr(b) : A+B"),
  ("PlusI_inrL", "[| A type;  b = d : B |] ==> inr(b) = inr(d) : A+B"),

  ("PlusE",
    "[| p: A+B;  !!x. x:A ==> c(x): C(inl(x));  \
\                !!y. y:B ==> d(y): C(inr(y)) |] \
\    ==> when(p, %x.c(x), %y.d(y)) : C(p)"),
 
  ("PlusEL",  
    "[| p = q : A+B;  !!x. x: A ==> c(x) = e(x) : C(inl(x));   \
\                     !!y. y: B ==> d(y) = f(y) : C(inr(y)) |] \
\    ==> when(p, %x.c(x), %y.d(y)) = when(q, %x.e(x), %y.f(y)) : C(p)"),
 
  ("PlusC_inl",
    "[| a: A;  !!x. x:A ==> c(x): C(inl(x));  \
\              !!y. y:B ==> d(y): C(inr(y)) |] \
\    ==> when(inl(a), %x.c(x), %y.d(y)) = c(a) : C(inl(a))"),
 
  ("PlusC_inr",
    "[| b: B;  !!x. x:A ==> c(x): C(inl(x));  \
\              !!y. y:B ==> d(y): C(inr(y)) |] \
\    ==> when(inr(b), %x.c(x), %y.d(y)) = d(b) : C(inr(b))"),


  (*The type Eq*)

  ("EqF",    "[| A type;  a : A;  b : A |] ==> Eq(A,a,b) type"),
  ("EqFL", "[| A=B;  a=c: A;  b=d : A |] ==> Eq(A,a,b) = Eq(B,c,d)"),
  ("EqI", "a = b : A ==> eq : Eq(A,a,b)"),
  ("EqE", "p : Eq(A,a,b) ==> a = b : A"),

  (*By equality of types, can prove C(p) from C(eq), an elimination rule*)
  ("EqC", "p : Eq(A,a,b) ==> p = eq : Eq(A,a,b)"),

  (*The type F*)

  ("FF", "F type"),
  ("FE", "[| p: F;  C type |] ==> contr(p) : C"),
  ("FEL",  "[| p = q : F;  C type |] ==> contr(p) = contr(q) : C"),

  (*The type T 
     Martin-Lof's book (page 68) discusses elimination and computation.
     Elimination can be derived by computation and equality of types,
     but with an extra premise C(x) type x:T.
     Also computation can be derived from elimination. *)

  ("TF", "T type"),
  ("TI", "tt : T"),
  ("TE", "[| p : T;  c : C(tt) |] ==> c : C(p)"),
  ("TEL", "[| p = q : T;  c = d : C(tt) |] ==> c = d : C(p)"),
  ("TC", "p : T ==> p = tt : T")
];

val sign = sign_of thy;
val ax = get_axiom thy;
(*INSERT-RULENAMES -- file produced by make-rulenames*)
val refl_red = ax"refl_red";
val red_if_equal = ax"red_if_equal";
val trans_red = ax"trans_red";
val refl_type = ax"refl_type";
val refl_elem = ax"refl_elem";
val sym_type = ax"sym_type";
val sym_elem = ax"sym_elem";
val trans_type = ax"trans_type";
val trans_elem = ax"trans_elem";
val equal_types = ax"equal_types";
val equal_typesL = ax"equal_typesL";
val subst_type = ax"subst_type";
val subst_typeL = ax"subst_typeL";
val subst_elem = ax"subst_elem";
val subst_elemL = ax"subst_elemL";
val NF = ax"NF";
val NI0 = ax"NI0";
val NI_succ = ax"NI_succ";
val NI_succL = ax"NI_succL";
val NE = ax"NE";
val NEL = ax"NEL";
val NC0 = ax"NC0";
val NC_succ = ax"NC_succ";
val zero_ne_succ = ax"zero_ne_succ";
val ProdF = ax"ProdF";
val ProdFL = ax"ProdFL";
val ProdI = ax"ProdI";
val ProdIL = ax"ProdIL";
val ProdE = ax"ProdE";
val ProdEL = ax"ProdEL";
val ProdC = ax"ProdC";
val ProdC2 = ax"ProdC2";
val SumF = ax"SumF";
val SumFL = ax"SumFL";
val SumI = ax"SumI";
val SumIL = ax"SumIL";
val SumE = ax"SumE";
val SumEL = ax"SumEL";
val SumC = ax"SumC";
val fst_def = ax"fst_def";
val snd_def = ax"snd_def";
val PlusF = ax"PlusF";
val PlusFL = ax"PlusFL";
val PlusI_inl = ax"PlusI_inl";
val PlusI_inlL = ax"PlusI_inlL";
val PlusI_inr = ax"PlusI_inr";
val PlusI_inrL = ax"PlusI_inrL";
val PlusE = ax"PlusE";
val PlusEL = ax"PlusEL";
val PlusC_inl = ax"PlusC_inl";
val PlusC_inr = ax"PlusC_inr";
val EqF = ax"EqF";
val EqFL = ax"EqFL";
val EqI = ax"EqI";
val EqE = ax"EqE";
val EqC = ax"EqC";
val FF = ax"FF";
val FE = ax"FE";
val FEL = ax"FEL";
val TF = ax"TF";
val TI = ax"TI";
val TE = ax"TE";
val TEL = ax"TEL";
val TC = ax"TC";
end;

