(*  Title: 	ZF/ex/ramsey
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

Ramsey's Theorem

Based upon the article
D Basin and M Kaufmann,
The Boyer-Moore Prover and Nuprl: An Experimental Comparison.
In G Huet and G Plotkin, editors, Logical Frameworks.
(CUP, 1991), pages 89--119
*)

writeln"ZF/ex/ramsey";

structure Ramsey =
struct

val const_decs =
    [ (["Symmetric"],			"i=>o"),
      (["Atleast"],			"[i,i]=>o"),
      (["Clique","Indept","Ramsey"],	"[i,i,i]=>o") ];

val thy = extend_theory Arith.thy "ramsey"
     ([], [], [], [], const_decs, None)
 [ 
  ("Symmetric_def",
      "Symmetric(E) == (ALL x y. <x,y>:E --> <y,x>:E)"),
  ("Clique_def",
      "Clique(C,V,E) == (C<=V) & (ALL x:C. ALL y:C. ~ x=y --> <x,y>:E)"),
  ("Indept_def",
      "Indept(I,V,E) == (I<=V) & (ALL x:I. ALL y:I. ~ x=y --> ~ <x,y>:E)"),
  ("Atleast_def",
      "Atleast(n,S) == (EX f. f: inj(n,S))"),
  ("Ramsey_def",
      "Ramsey(n,i,j) == ALL V E. Symmetric(E) & Atleast(n,V) -->  \
\         (EX C. Clique(C,V,E) & Atleast(i,C)) |       \
\         (EX I. Indept(I,V,E) & Atleast(j,I))")
  ];
end;

local val ax = get_axiom Ramsey.thy in 
val Symmetric_def   = ax"Symmetric_def";
val Clique_def  = ax"Clique_def";
val Indept_def  = ax"Indept_def";
val Atleast_def = ax"Atleast_def";
val Ramsey_def  = ax"Ramsey_def";
end;


val ramsey_congs = mk_congs Ramsey.thy ["Atleast"];
val ramsey_ss = arith_ss addcongs ramsey_congs;

(*** Cliques and Independent sets ***)

goalw Ramsey.thy [Clique_def] "Clique(0,V,E)";
by (fast_tac ZF_cs 1);
val Clique0 = result();

val prems = goalw Ramsey.thy [Clique_def]
    "[| Clique(C,V',E);  V'<=V |] ==> Clique(C,V,E)";
by (cfast_tac prems 1);
val Clique_superset = result();

goalw Ramsey.thy [Indept_def] "Indept(0,V,E)";
by (fast_tac ZF_cs 1);
val Indept0 = result();

val prems = goalw Ramsey.thy [Indept_def]
    "[| Indept(I,V',E);  V'<=V |] ==> Indept(I,V,E)";
by (cfast_tac prems 1);
val Indept_superset = result();

(*** Atleast ***)

goalw Ramsey.thy [Atleast_def,inj_def] "Atleast(0,A)";
by (fast_tac (ZF_cs addIs [PiI]) 1);
val Atleast0 = result();

(*SHOULD BE MOVED TO func.ML*)
(*Weaking one function type to another*)
val prems = goal ZF_Rule.thy "[| f: A->B;  B<=D |] ==> f: A->D";
by (cut_facts_tac prems 1);
by (etac PiE 1);
by (rtac PiI 1);
by (rtac subsetI 1);
by (set_mp_tac 1);
by (fast_tac ZF_cs 1);
by (etac bspec 1);
by (assume_tac 1);
val Pi_weaken_type = result();

val prems = goalw Perm.thy [inj_def]  (*SHOULD BE MOVED TO perm.ML*)
    "[| f: inj(A,B);  B<=D |] ==> f: inj(A,D)";
by (cut_facts_tac prems 1);
by (fast_tac (ZF_cs addSEs [Pi_weaken_type]) 1);
val inj_weaken_type = result();

val [major] = goal Perm.thy  (*SHOULD BE MOVED TO perm.ML*)
    "[| f: inj(succ(m), A) |] ==> restrict(f,m) : inj(m, A-{f`m})";
by (rtac (major RS restrict_bij RS bij_is_inj RS inj_weaken_type) 1);
by (fast_tac ZF_cs 1);
by (cut_facts_tac [major] 1);
by (rewtac inj_def);
by (safe_tac ZF_cs);
by (etac range_type 1);
by (assume_tac 1);
by (dtac apply_equality 1);
by (assume_tac 1);
by (res_inst_tac [("a","m")] mem_anti_refl 1);
by (fast_tac ZF_cs 1);
val inj_succ_restrict = result();

val [major] = goalw Ramsey.thy [Atleast_def]
    "Atleast(succ(m),A) ==> EX x:A. Atleast(m, A-{x})";
by (rtac (major RS exE) 1);
by (rtac bexI 1);
by (etac (inj_is_fun RS apply_type) 2);
by (rtac succI1 2);
by (rtac exI 1);
by (etac inj_succ_restrict 1);
val Atleast_succD = result();

val major::prems = goalw Ramsey.thy [Atleast_def]
    "[| Atleast(n,A);  A<=B |] ==> Atleast(n,B)";
by (rtac (major RS exE) 1);
by (rtac exI 1);
by (etac inj_weaken_type 1);
by (resolve_tac prems 1);
val Atleast_superset = result();

val prems = goalw Perm.thy [inj_def]  (*SHOULD BE MOVED TO perm.ML*)
    "[| f: inj(A,B);  ~ a:A;  ~ b:B |]  ==> \
\    cons(<a,b>,f) : inj(cons(a,A), cons(b,B))";
by (cut_facts_tac prems 1);
by (dtac CollectD1 1);
by (rtac CollectI 1);
by (etac fun_extend 1);
by (assume_tac 1);
by (REPEAT (rtac ballI 1));
by (REPEAT_FIRST (eresolve_tac [consE,ssubst]));
by (fast_tac ZF_cs 1);
by (ASM_SIMP_TAC (ramsey_ss addrews [fun_extend_apply2,fun_extend_apply1]) 1);
by (fast_tac (ZF_cs addIs [apply_type]) 1);
by (ASM_SIMP_TAC (ramsey_ss addrews [fun_extend_apply2,fun_extend_apply1]) 1);
by (fast_tac (ZF_cs addIs [apply_type]) 1);
by (ASM_SIMP_TAC (ramsey_ss addrews [fun_extend_apply2,fun_extend_apply1]) 1);
by (cut_facts_tac (prems RL [CollectD2]) 1);
by (fast_tac ZF_cs 1);
val inj_extend = result();

val prems = goalw Ramsey.thy [Atleast_def,succ_def]
    "[| Atleast(m,B);  ~ b: B |] ==> Atleast(succ(m), cons(b,B))";
by (cut_facts_tac prems 1);
by (etac exE 1);
by (rtac exI 1);
by (etac inj_extend 1);
by (rtac notI 1);
by (etac mem_anti_refl 1);
by (assume_tac 1);
val Atleast_succI = result();

val prems = goal Ramsey.thy
    "[| Atleast(m, B-{x});  x: B |] ==> Atleast(succ(m), B)";
by (cut_facts_tac prems 1);
by (etac (Atleast_succI RS Atleast_superset) 1);
by (fast_tac ZF_cs 1);
by (fast_tac ZF_cs 1);
val Atleast_Diff_succI = result();

(*** Main Cardinality Lemma ***)

val add_succ_right_conv = prove_goal Arith.thy
    "[| m:nat;  n:nat |] ==> m #+ succ(n) = succ(m #+ n)"
 (fn prems=>
    [ (res_inst_tac [("n1", "succ(n)")] (add_commute RS ssubst) 1), 
      (res_inst_tac [("n1", "n")] (add_commute RS ssubst) 3), 
      (REPEAT(resolve_tac (prems@[add_succ_conv,nat_succ_I]) 1)) ]); 

val prems = goal Ramsey.thy
    "m: nat ==> \
\    ALL n: nat. ALL A B. Atleast(m#+n, A Un B) -->   \
\                         Atleast(m,A) | Atleast(n,B)";
by (nat_ind_tac "m" prems 1);
by (fast_tac (ZF_cs addSIs [Atleast0]) 1);
by (ASM_SIMP_TAC ramsey_ss 1);
by (rtac ballI 1);
by (nat_ind_tac "n" [] 1);
by (fast_tac (ZF_cs addSIs [Atleast0]) 1);
by (ASM_SIMP_TAC (ramsey_ss addrews [add_succ_right_conv]) 1);
by (safe_tac ZF_cs);
by (etac (Atleast_succD RS bexE) 1);
by (etac UnE 1);
(*ensure that the correct quantified assumption is instantiated*)
by (dres_inst_tac [("xb","succ(n1)"), ("xa","A-{x}"), ("x","B")] 
    (standard(bspec RS spec RS spec)) 1);
by (etac nat_succ_I 1);
by (etac (mp RS disjE) 1);
by (mp_tac 3);
by (ASM_SIMP_TAC (ramsey_ss addrews [add_succ_right_conv]) 1);
by (etac Atleast_superset 1);
by (fast_tac ZF_cs 1);
by (etac Atleast_Diff_succI 1);
by (assume_tac 1);
by (dres_inst_tac [("x","A")] spec 1);
by (dres_inst_tac [("x","B-{x}")] spec 1);
by (etac (mp RS disjE) 1);
by (assume_tac 2);
by (etac Atleast_superset 1);
by (fast_tac ZF_cs 1);
by (etac notE 1);
by (etac Atleast_Diff_succI 1);
by (assume_tac 1);
val Atleast_Un_lemma = result();

(* [| m:nat; n:nat; Atleast(m#+n, A Un B) |] ==> Atleast(m,A) | Atleast(n,B) *)
val Atleast_Un = standard (Atleast_Un_lemma RS bspec RS spec RS spec RS mp);


(**** Ramsey's Theorem ****)

(** Base cases of induction **)

goalw Ramsey.thy [Ramsey_def] "Ramsey(0,0,j)";
by (fast_tac (ZF_cs addIs [Clique0,Atleast0]) 1);
val Ramsey00j = result();

goalw Ramsey.thy [Ramsey_def] "Ramsey(0,i,0)";
by (fast_tac (ZF_cs addIs [Indept0,Atleast0]) 1);
val Ramsey0i0 = result();

(** Lemmas for induction step **)

val prems = goal Ramsey.thy
    "[| Atleast(m#+n, A);  m: nat;  n: nat |] ==> \
\    Atleast(m, {x:A. ~P(x)}) | Atleast(n, {x:A. P(x)})";
by (rtac Atleast_Un 1);
by (rtac Atleast_superset 3);
by (REPEAT (resolve_tac prems 1));
by (fast_tac ZF_cs 1);
val Atleast_partition = result();

(*For the Atleast part, proves ~(a:I) from the second premise!*)
val prems = goalw Ramsey.thy [Symmetric_def,Indept_def]
    "[| Symmetric(E);  Indept(I, {z: V-{a}. ~ <a,z>:E}, E);  a: V;  \
\       Atleast(j,I) |] ==>   \
\    Indept(cons(a,I), V, E) & Atleast(succ(j), cons(a,I))";
by (cut_facts_tac prems 1);
by (fast_tac (ZF_cs addSEs [Atleast_succI]) 1);	 (*57 secs*)
val Indept_succ = result();

val prems = goalw Ramsey.thy [Symmetric_def,Clique_def]
    "[| Symmetric(E);  Clique(I, {z: V-{a}. <a,z>:E}, E);  a: V;  \
\       Atleast(j,I) |] ==>   \
\    Clique(cons(a,I), V, E) & Atleast(succ(j), cons(a,I))";
by (cut_facts_tac prems 1);
by (fast_tac (ZF_cs addSEs [Atleast_succI]) 1);  (*70 secs*)
val Clique_succ = result();

(** Induction step **)

val ram1::ram2::prems = goalw Ramsey.thy [Ramsey_def] 
   "[| Ramsey(m, succ(i), j);  Ramsey(n, i, succ(j));  m: nat;  n: nat |] ==> \
\   Ramsey(succ(m#+n), succ(i), succ(j))";
by (safe_tac ZF_cs);
by (etac (Atleast_succD RS bexE) 1);
by (eres_inst_tac [("P1","%z.<x,z>:E")] (Atleast_partition RS disjE) 1);
by (REPEAT (resolve_tac prems 1));
(*case m*)
by (rtac (ram1 RS spec RS spec RS mp RS disjE) 1);
by (fast_tac ZF_cs 1);
by (fast_tac (ZF_cs addEs [Clique_superset]) 1); (*easy -- given a Clique*)
by (safe_tac ZF_cs);
by (eresolve_tac (swapify [exI]) 1);
by (REPEAT (ares_tac [Indept_succ] 1));  	 (*make a bigger Indept*)
(*case n*)
by (rtac (ram2 RS spec RS spec RS mp RS disjE) 1);
by (fast_tac ZF_cs 1);
by (safe_tac ZF_cs);
by (rtac exI 1);
by (REPEAT (ares_tac [Clique_succ] 1));  	 (*make a bigger Clique*)
by (fast_tac (ZF_cs addEs [Indept_superset]) 1); (*easy -- given an Indept*)
val Ramsey_step_lemma = result();


(** The actual proof **)

val prems = goal Ramsey.thy "i: nat ==> ALL j: nat. EX n:nat. Ramsey(n,i,j)";
by (nat_ind_tac "i" prems 1);
by (fast_tac (ZF_cs addSIs [nat_0_I,Ramsey00j]) 1);
by (rtac ballI 1);
by (nat_ind_tac "j" [] 1);
by (fast_tac (ZF_cs addSIs [nat_0_I,Ramsey0i0]) 1);
by (dres_inst_tac [("x","succ(j1)")] bspec 1);
by (REPEAT (eresolve_tac [nat_succ_I,bexE] 1));
by (rtac bexI 1);
by (rtac Ramsey_step_lemma 1);
by (REPEAT (ares_tac [nat_succ_I,add_type] 1));
val ramsey = result();

writeln"Reached end of file.";
