(*  Title: 	Provers/hypsubst
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

Martin Coen's tactic for substitution in the hypotheses
*)

signature HYPSUBST_DATA =
  sig
  val dest_eq: term -> term*term
  val imp_intr: thm	(* (P ==> Q) ==> P-->Q *)
  val rev_cut_eq: thm	(* [| a=b;  a=b ==> R |] ==> R *)
  val rev_mp: thm	(* [| P;  P-->Q |] ==> Q *)
  val subst: thm	(* [| a=b;  P(a) |] ==> P(b) *)
  val sym: thm		(* a=b ==> b=a *)
  end;

signature HYPSUBST =
  sig
  val hyp_subst_tac: int -> tactic
  val lasthyp_subst_tac: int -> tactic
    (*exported purely for debugging purposes*)
  val eq_var: term -> term * thm
  val inspect_pair: term * term -> term * thm
  val liftvar: int -> typ list -> term
  end;

functor HypsubstFun(Data: HYPSUBST_DATA): HYPSUBST = 
struct

local open Data in

fun REPEATN 0 tac = all_tac
  | REPEATN n tac = Tactic(fn state =>
                           tapply(tac THEN REPEATN (n-1) tac,  state));

(*Substitute an equality -- the last assumption -- throughout the subgoal*)
fun lasthyp_subst_tac i = DETERM (STATE(fn state =>
      let val (_,_,Bi,_) = dest_state(state,i)
          val n = length(Logic.strip_assums_hyp Bi) - 1
      in REPEATN n (etac rev_mp i) THEN
         etac subst i THEN REPEATN n (rtac imp_intr i)
      end
      handle THM _ => no_tac));

local
  val T = case #1 (types_sorts rev_cut_eq) ("a",0) of
	      Some T => T
   	    | None   => error"No such variable in rev_cut_eq"
in
  fun liftvar inc paramTs = Var(("a",inc), paramTs ---> incr_tvar inc T);
end;

(*Substitute an equality -- the last assumption -- throughout the subgoal*)
fun lasthyp_subst_tac i = DETERM (STATE(fn state =>
      let val (_,_,Bi,_) = dest_state(state,i)
          val n = length(Logic.strip_assums_hyp Bi) - 1
      in REPEATN n (etac rev_mp i) THEN
         etac subst i THEN REPEATN n (rtac imp_intr i)
      end
      handle THM _ => no_tac));

exception EQ_VAR;

fun loose (i,t) = 0 mem add_loose_bnos(t,i,[]);

(*It's not safe to substitute for a constant; consider 0=1.
  It's not safe to substitute for x=t[x] since x is not eliminated.
  It's not safe to substitute for a variable free in the premises,
    but how could we check for this?*)
fun inspect_pair (t,u) =
  case (Logic.eta_contract t, Logic.eta_contract u) of
       (Bound i, _) => if loose(i,u) then raise Match else (t, asm_rl)
     | (_, Bound i) => if loose(i,t) then raise Match else (u, sym)
     | (Free _, _) => if Logic.occs(t,u) then raise Match else (t, asm_rl)
     | (_, Free _) => if Logic.occs(u,t) then raise Match else (u, sym) 
     | _ => raise Match;

 (* Extracts the name of the variable on the left (resp. right) of an equality
   assumption.  Returns the rule asm_rl (resp. sym). *)
fun eq_var (Const("all",_) $ Abs(_,_,t)) = eq_var t
  | eq_var (Const("==>",_) $ A $ B) = 
	(inspect_pair (dest_eq A) (*Match comes from inspect_pair or dest_eq*)
	 handle Match => eq_var B)
  | eq_var _ = raise EQ_VAR;

(*Lift and instantiate a rule wrt the given state and subgoal number *)
fun lift_instpair (state, i, t, rule) =
  let val {maxidx,sign,...} = rep_thm state
      val (_, _, Bi, _) = dest_state(state,i)
      val params = Logic.strip_params Bi
      val var = liftvar (maxidx+1) (map #2 params)
      and u   = Unify.rlist_abs(rev params, t)
      and cterm = Sign.cterm_of sign
  in cterm_instantiate [(cterm var, cterm u)] (lift_rule (state,i) rule)
  end;

fun eres_instpair_tac t rule i = STATE (fn state => 
	   compose_tac (true, lift_instpair (state, i, t, rule),
			length(prems_of rule)) i);

val ssubst = sym RS subst;

(*Select a suitable equality assumption and substitute throughout the subgoal*)
fun hyp_subst_tac i = DETERM (STATE(fn state =>
      let val (_,_,Bi,_) = dest_state(state,i)
	  val n = length(Logic.strip_assums_hyp Bi) - 1
	  val (t,symopt) = eq_var Bi
      in eres_instpair_tac t (symopt RS rev_cut_eq) i  THEN
         REPEATN n (etac rev_mp i) THEN
	 etac ssubst i THEN REPEATN n (rtac imp_intr i)
      end
      handle THM _ => no_tac | EQ_VAR => no_tac));

end
end;

