(*  Title: 	HOL/list
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

Definition of type 'a list by a least fixed point

Also, the List "functor" for defining other recursive types

simpsets need tidying
*)


structure List =
struct
val thy = extend_theory Univ.thy "List"
([],[],[],
 [(["list"], ([["term"]],"term"))],
 [ 
  (["List"],  	    "(('a + nat)sexp)set => (('a + nat)sexp)set"),
  (["Rep_List"], 	"'a list => ('a + nat)sexp"),
  (["Abs_List"], 	"('a + nat)sexp => 'a list"),
  (["NIL"],      	"('a + nat)sexp"),
  (["CONS"],     	"[('a + nat)sexp, ('a + nat)sexp] => ('a + nat)sexp"),
  (["Nil"],      	"'a list"),
  (["Cons"],     	"['a, 'a list] => 'a list"),
  (["List_rec"],	
    "[('a+nat)sexp, 'b, [('a+nat)sexp , ('a+nat)sexp, 'b]=>'b] => 'b"),
  (["list_rec"],        "['a list, 'b, ['a ,'a list, 'b]=>'b] => 'b"),
  (["Rep_map"],
    "('b => ('a+nat)sexp) => ('b list => ('a+nat)sexp)"),
  (["Abs_map"],
   "(('a+nat)sexp => 'b) => ('a+nat)sexp => 'b list"),
  (["list_all"],        "('a => bool) => ('a list => bool)"),
  (["map"],		"('a=>'b) => ('a list => 'b list)")
 ],
 None)
 [
  ("List_def",	"List(A) == lfp(%Z. {NUMB(0)} <+> A <*> Z)"),
    (*faking a type definition...*)
  ("Rep_List", 		"Rep_List(xs): List(range(ATOM))"),
  ("Rep_List_inverse", 	"Abs_List(Rep_List(xs)) = xs"),
  ("Abs_List_inverse", 	"M: List(range(ATOM)) ==> Rep_List(Abs_List(M)) = M"),
     (*defining the concrete constructors*)
  ("NIL_def",  	"NIL == IN0(NUMB(0))"),
  ("CONS_def", 	"CONS(M,z) == IN1(M . z)"),
     (*defining the abstract constructors*)
  ("Nil_def",  		"Nil == Abs_List(NIL)"),
  ("Cons_def", 	 "Cons(x,xs) == Abs_List(CONS(ATOM(x), Rep_List(xs)))"),

     (*list recursion*)
  ("List_rec_def",	
   "List_rec(M,c,d) == wfrec(trancl(pred_sexp), M, \
\                    %z g. @u. z=NIL & u=c | \
\                             (? M y. z=CONS(M,y) & u = d(M,y,g(y))))" ),

  ("list_rec_def",
   "list_rec(l,c,d) == \
\   List_rec(Rep_List(l), c, %x y r. d(Inv(ATOM,x), Abs_List(y), r))" ),

     (*Generalized map functionals*)
  ("Rep_map_def", 
   "Rep_map(f,xs) == list_rec(xs, NIL, %x l r. CONS(f(x), r))" ),
  ("Abs_map_def", 
   "Abs_map(g,M) == List_rec(M, Nil, %N L r. Cons(g(N), r))" ),

  ("list_all_def", "list_all(P,xs) == list_rec(xs, True, %x l r. P(x) & r)" ),
  ("map_def", 	"map(f,xs) == list_rec(xs, Nil, %x l r. Cons(f(x), r))" )
 ];
end;

local val ax = get_axiom List.thy
in
val List_def    = ax"List_def";
val Rep_List    = ax"Rep_List";
val Rep_List_inverse    = ax"Rep_List_inverse";
val Abs_List_inverse    = ax"Abs_List_inverse";
val NIL_def     = ax"NIL_def";
val CONS_def    = ax"CONS_def";
val Nil_def     = ax"Nil_def";
val Cons_def    = ax"Cons_def";
val List_rec_def        = ax"List_rec_def";
val list_rec_def        = ax"list_rec_def";
val Rep_map_def = ax"Rep_map_def";
val Abs_map_def = ax"Abs_map_def";
val list_all_def        = ax"list_all_def";
val map_def     = ax"map_def";
end;


(** the list functional **)

goal List.thy "mono(%Z. {NUMB(0)} <+> A <*> Z)";
by (REPEAT (ares_tac [monoI, subset_refl, usum_mono, uprod_mono] 1));
val List_fun_mono = result();

val List_unfold = List_fun_mono RS (List_def RS Tarski_def_theorem);

(*This justifies using List in other recursive type definitions*)
val prems = goalw List.thy [List_def] "[| A<=B |] ==> List(A) <= List(B)";
by (REPEAT (ares_tac (prems@[monoI, subset_refl, lfp_mono, 
			     usum_mono, uprod_mono]) 1));
val List_mono = result();

(** Induction **)

(*Induction for the set List(A) *)
val major::prems = goalw List.thy [List_def,NIL_def,CONS_def]
    "[| M: List(A);  P(NIL);   \
\       !!M N. [| M: A;  N: List(A);  P(N) |] ==> P(CONS(M,N)) |]  \
\    ==> P(M)";
by (rtac (List_fun_mono RS (major RS general_induction)) 1);
by (fast_tac (set_cs addIs prems addEs [usumE,uprodE]) 1);
val List_induct = result();

(*Induction for the type 'a list *)
val prems = goalw List.thy [Nil_def,Cons_def]
    "[| P(Nil);   \
\       !!x xs. [| P(xs) |] ==> P(Cons(x,xs)) |]  ==> P(l)";
by (rtac (Rep_List_inverse RS subst) 1);   (*types force good instantiation*)
by (rtac (Rep_List RS List_induct) 1);
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [rangeE, ssubst, Abs_List_inverse RS subst] 1));
val list_induct = result();

(*Perform induction on xs. *)
fun list_ind_tac a M = 
    EVERY [res_inst_tac [("l",a)] list_induct M,
	   rename_last_tac a ["1"] (M+1)];

(** Introduction rules for List constructors **)

(* ?c : {NUMB(0)} <+> ?Ga5 <*> List(?Ga5) ==> ?c : List(?Ga5) *)
val ListI = List_unfold RS equalityD2 RS subsetD;

(* Nil is a List -- this also justifies the type definition*)
goalw List.thy [NIL_def] "NIL: List(A)";
by (REPEAT (resolve_tac [singletonI, ListI, usum_IN0I] 1));
val NIL_I = result();

val prems = goalw List.thy [CONS_def]
    "[| a: A;  M: List(A) |] ==> CONS(a,M) : List(A)";
by (REPEAT (resolve_tac (prems@[ListI, uprodI, usum_IN1I]) 1));
val CONS_I = result();

(*** Isomorphisms ***)

goal List.thy "inj(Rep_List)";
by (rtac inj_inverseI 1);
by (rtac Rep_List_inverse 1);
val inj_Rep_List = result();

goal List.thy "inj_onto(Abs_List,List(range(ATOM)))";
by (rtac inj_onto_inverseI 1);
by (etac Abs_List_inverse 1);
val inj_onto_Abs_List = result();

(** Distinctness of constructors **)

goalw List.thy [NIL_def,CONS_def] "~ CONS(a,M) = NIL";
by (rtac IN1_not_IN0 1);
val CONS_not_NIL = result();
val NIL_not_CONS = standard (CONS_not_NIL RS neg_sym);

val CONS_neq_NIL = standard (CONS_not_NIL RS notE);
val NIL_neq_CONS = sym RS CONS_neq_NIL;

goalw List.thy [Nil_def,Cons_def] "~ Cons(x,xs) = Nil";
by (rtac (CONS_not_NIL RS (inj_onto_Abs_List RS inj_onto_contraD)) 1);
by (REPEAT (resolve_tac [rangeI, NIL_I, CONS_I, Rep_List] 1));
val Cons_not_Nil = result();

val Nil_not_Cons = standard (Cons_not_Nil RS neg_sym);

val Cons_neq_Nil = standard (Cons_not_Nil RS notE);
val Nil_neq_Cons = sym RS Cons_neq_Nil;

(** Injectiveness of Cons **)

val [major,minor] = goalw List.thy [CONS_def]
    "[| CONS(K,M)=CONS(L,N);  [| K=L;  M=N |] ==> P \
\    |] ==> P";
by (rtac (major RS (IN1_inject RS Scons_inject)) 1);
by (REPEAT (ares_tac [minor] 1));
val CONS_inject = result();

val [major,minor] = goalw List.thy [Cons_def]
    "[| Cons(x,xs)=Cons(y,ys);  [| x=y;  xs=ys |] ==> P \
\    |] ==> P";
by (rtac (major RS (inj_onto_Abs_List RS inj_ontoD RS CONS_inject)) 1);
by (REPEAT (resolve_tac [rangeI, NIL_I, CONS_I, Rep_List] 1));
by (rtac minor 1);
by (REPEAT (eresolve_tac [ATOM_inject, inj_Rep_List RS injD] 1));
val Cons_inject = result();

val [major] = goal List.thy "CONS(M,N): List(A) ==> M: A & N: List(A)";
by (rtac (major RS setup_induction) 1);
by (etac List_induct 1);
by (ALLGOALS (fast_tac (set_cs addEs [CONS_neq_NIL,CONS_inject])));
val CONS_D = result();

(** Some rewrite rules **)

goal List.thy "(Cons(x,xs)=Cons(y,ys)) = (x=y & xs=ys)";
by (fast_tac (HOL_cs addEs [Cons_inject]) 1);
val Cons_Cons_eq = result();

goal List.thy "(CONS(K,M)=CONS(L,N)) = (K=L & M=N)";
by (fast_tac (HOL_cs addEs [CONS_inject]) 1);
val CONS_CONS_eq = result();

(*Basic ss with constructors and their freeness*)
val list_free_ss = 
    HOL_ss addcongs (mk_congs List.thy ["Cons","CONS","op :"])
           addrews [Cons_not_Nil, Nil_not_Cons, Cons_Cons_eq,
		    CONS_not_NIL, NIL_not_CONS, CONS_CONS_eq,
		    NIL_I, CONS_I];

goal List.thy "!x. ~(l=Cons(x,l))";
by (list_ind_tac "l" 1);
by (rtac (Nil_not_Cons RS allI) 1);
by (ASM_SIMP_TAC list_free_ss 1);
val l_not_Cons_l = result();


(*** List_rec -- by wf recursion on pred_sexp ***)

val List_rec_unfold =
    wf_pred_sexp RS wf_trancl RS (List_rec_def RS wfrec_def_conv);

(** conversion rules **)

goalw List.thy [CONS_def,IN1_def] "<M, CONS(M,N)> : trancl(pred_sexp)";
by (rtac (pred_sexpI1 RS trancl_into_trancl2) 1);
by (rtac (pred_sexpI2 RS r_into_trancl) 1);
val pred_sexp_CONS1 = result();

goalw List.thy [CONS_def,IN1_def] "<N, CONS(M,N)> : trancl(pred_sexp)";
by (rtac (pred_sexpI2 RS trancl_into_trancl2) 1);
by (rtac (pred_sexpI2 RS r_into_trancl) 1);
val pred_sexp_CONS2 = result();

goal List.thy "List_rec(NIL,c,h) = c";
by (rtac (List_rec_unfold RS trans) 1);
by (fast_tac (HOL_cs addIs [select_equality]
	             addEs [NIL_neq_CONS]) 1);
val List_rec_NIL_conv = result();

goal List.thy "List_rec(CONS(a,N), c, h) = h(a, N, List_rec(N,c,h))";
by (rtac (List_rec_unfold RS trans) 1);
by (rtac (select_equality RS trans) 1);
by (fast_tac HOL_cs 1);
by (fast_tac (HOL_cs addEs [CONS_neq_NIL, CONS_inject]) 1);
by (SIMP_TAC (HOL_ss 
	addcongs (mk_typed_congs List.thy [("h", "[?'x, ?'x, ?'y]=>?'y")])
	addrews [pred_sexp_CONS2, cut_apply]) 1);
val List_rec_CONS_conv = result();

(*** list_rec -- by List_rec ***)

goalw List.thy [list_rec_def, Nil_def] "list_rec(Nil,c,h) = c";
by (SIMP_TAC (HOL_ss addcongs (mk_congs List.thy ["List_rec"])
                     addrews [List_rec_NIL_conv, Abs_List_inverse, NIL_I]) 1);
val list_rec_Nil_conv = result();


goalw List.thy [list_rec_def, Cons_def]
    "list_rec(Cons(a,l), c, h) = h(a, l, list_rec(l,c,h))";
by (SIMP_TAC (HOL_ss 
      addcongs (mk_congs List.thy ["List_rec"] @ 
		mk_typed_congs List.thy [("h", "[?'a , ?'a list, ?'b]=>?'b")])
      addrews [List_rec_CONS_conv, Abs_List_inverse, CONS_I,
	       Rep_List, rangeI, inj_ATOM, Inv_f_f, Rep_List_inverse]) 1);
val list_rec_Cons_conv = result();

val list_rec_ss = list_free_ss
    	addcongs (mk_congs List.thy ["list_rec","List_rec"])
        addrews [List_rec_NIL_conv, List_rec_CONS_conv,
		 list_rec_Nil_conv, list_rec_Cons_conv];

(*Type checking.  Useful?*)
val major::prems = goal List.thy
    "[| M: List(A);    \
\       c: C(NIL);       \
\       !!x y r. [| x: A;  y: List(A);  r: C(y) |] ==> h(x,y,r): C(CONS(x,y)) \
\    |] ==> List_rec(M,c,h) : C(M)";
by (rtac (major RS List_induct) 1);
by (ALLGOALS (ASM_SIMP_TAC (list_rec_ss addrews prems)));
val List_rec_type = result();

(** Generalized map functionals **)

goalw List.thy [Rep_map_def] "Rep_map(f,Nil) = NIL";
by (rtac list_rec_Nil_conv 1);
val Rep_map_Nil_conv = result();

goalw List.thy [Rep_map_def]
    "Rep_map(f, Cons(x,xs)) = CONS(f(x), Rep_map(f,xs))";
by (rtac list_rec_Cons_conv 1);
val Rep_map_Cons_conv = result();

val prems = goalw List.thy [Rep_map_def]
    "(!!x. f(x): A) ==> Rep_map(f,xs): List(A)";
by (rtac list_induct 1);
by (ALLGOALS (ASM_SIMP_TAC (list_rec_ss addrews prems)));
val Rep_map_type = result();

goalw List.thy [Abs_map_def] "Abs_map(g,NIL) = Nil";
by (rtac List_rec_NIL_conv 1);
val Abs_map_NIL_conv = result();

goalw List.thy [Abs_map_def]
    "Abs_map(g, CONS(M,N)) = Cons(g(M), Abs_map(g,N))";
by (rtac List_rec_CONS_conv 1);
val Abs_map_CONS_conv = result();

(** The functional "map" **)

goalw List.thy [map_def] "map(f,Nil) = Nil";
by (rtac list_rec_Nil_conv 1);
val map_Nil_conv = result();

goalw List.thy [map_def] "map(f, Cons(x,xs)) = Cons(f(x), map(f,xs))";
by (rtac list_rec_Cons_conv 1);
val map_Cons_conv = result();

val map_ss = list_free_ss
    	addcongs (mk_congs List.thy ["Rep_map","Abs_map","map"])
        addrews [Abs_map_NIL_conv, Abs_map_CONS_conv, 
		 Rep_map_Nil_conv, Rep_map_Cons_conv, 
		 map_Nil_conv, map_Cons_conv];

val [major,minor] = goal List.thy 
    "[| M: List(A);  !!z. z: A ==> f(g(z)) = z |] \
\    ==> Rep_map(f, Abs_map(g,M)) = M";
by (rtac (major RS List_induct) 1);
by (ALLGOALS (ASM_SIMP_TAC (map_ss addrews [minor])));
val Abs_map_inverse = result();

(*Rep_map_inverse is obtained via Abs_Rep_map and map_ident*)


(** The functional "list_all" -- creates predicates over lists **)

goalw List.thy [list_all_def] "list_all(P,Nil) = True";
by (rtac list_rec_Nil_conv 1);
val list_all_Nil_conv = result();

goalw List.thy [list_all_def]
    "list_all(P, Cons(x,xs)) = (P(x) & list_all(P,xs))";
by (rtac list_rec_Cons_conv 1);
val list_all_Cons_conv = result();

val list_ss = map_ss
	addcongs (mk_congs List.thy ["list_all"])
        addrews [list_all_Nil_conv, list_all_Cons_conv];

(** Additional mapping lemmas **)

goal List.thy "map(%x.x, xs) = xs";
by (list_ind_tac "xs" 1);
by (ALLGOALS (ASM_SIMP_TAC map_ss));
val map_ident = result();

goal List.thy "Abs_map(g, Rep_map(f,xs)) = map(%t. g(f(t)), xs)";
by (list_ind_tac "xs" 1);
by (ALLGOALS (ASM_SIMP_TAC map_ss));
val Abs_Rep_map = result();


