(*  Title: 	ZF/nat
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Natural numbers in Zermelo-Fraenkel Set Theory 
*)

writeln"File ZF/nat";

structure Nat =
struct
val const_decs = 
 [ 
   (["nat"],		"i"),
   (["nat_case"],	"[i, i, i=>i]=>i"),
   (["nat_rec"],	"[i, i, [i,i]=>i]=>i")
 ];

val thy = extend_theory Ord.thy "nat"
     ([], [], [], [], const_decs, None)
 [ 
  ("nat_def", "nat == lfp(lam r: Pow(Inf). {0} Un RepFun(r,succ))"),

  ("nat_case_def",	
   "nat_case(n,a,b) == THE y. n=0 & y=a | (EX x. n=succ(x) & y=b(x))"),

  ("nat_rec_def",	
   "nat_rec(k,a,b) ==   \
\   wfrec(Memrel(nat), k, %n f. nat_case(n, a, %m. b(m, f`m)))")
 ];
end;

local val ax = get_axiom Nat.thy
in 
val nat_def = ax"nat_def";
val nat_case_def = ax"nat_case_def";
val nat_rec_def = ax"nat_rec_def";
end;


(* 0:Inf *)
val InfI0 = infinity RS conjunct1;    

(* ?a : Inf ==> succ(?a) : Inf *)
val InfIsucc = infinity RS conjunct2 RS bspec; 

goal Nat.thy
    "(lam r: Pow(Inf). {0} Un RepFun(r,succ)) : mono(Pow(Inf), Pow(Inf))";
by (rtac lam_mono_Powtype 1);
by (REPEAT (ares_tac [subset_refl, RepFun_mono, Un_mono, 
		      subsetI, InfI0, InfIsucc] 1 
     ORELSE eresolve_tac [UnE,singletonE, RepFunE, ssubst, subsetD] 1));
val natfun_mono = result();

goalw Nat.thy [nat_def] "nat = {0} Un RepFun(nat,succ)";
by (rtac (natfun_mono RS lam_Tarski_theorem) 1);
val nat_unfold = result();

(** Type checking of 0 and successor **)

goal Nat.thy "0 : nat";
by (rtac (nat_unfold RS ssubst) 1);
by (rtac (singletonI RS UnI1) 1);
val nat_0_I = result();

val prems = goal Nat.thy "n : nat ==> succ(n) : nat";
by (rtac (nat_unfold RS ssubst) 1);
by (rtac (RepFunI RS UnI2) 1);
by (resolve_tac prems 1);
val nat_succ_I = result();

(** Injectivity properties and induction **)

(*Mathematical induction*)
val prems = goalw Nat.thy [nat_def]
    "[| n: nat;  P(0);  !!x. [| x: nat;  P(x) |] ==> P(succ(x)) \
\    |] ==> P(n)";
by (rtac lam_general_induction 1);
by (REPEAT (ares_tac (natfun_mono::prems) 1
     ORELSE eresolve_tac [UnE,singletonE, RepFunE, CollectE, ssubst] 1));
val nat_induct = result();

(*Perform induction on n, then prove the n:nat subgoal using prems. *)
fun nat_ind_tac a prems i = 
    EVERY [res_inst_tac [("n",a)] nat_induct i,
	   rename_last_tac a ["1"] (i+2),
	   ares_tac prems i];

val prems = goal Nat.thy "n: nat ==> n=0 | (EX y:nat. n=succ(y))";
by (nat_ind_tac "n" prems 1);
by (fast_tac ZF_cs 1);
by (fast_tac ZF_cs 1);
val natE = result();

val prems = goal Nat.thy "n: nat ==> Ord(n)";
by (nat_ind_tac "n" prems 1);
by (REPEAT (ares_tac [Ord_0, Ord_succ] 1));
val naturals_are_ordinals = result();

val prems = goal Nat.thy "n: nat ==> n=0 | 0:n";
by (nat_ind_tac "n" prems 1);
by (fast_tac ZF_cs 1);
by (fast_tac (ZF_cs addIs [naturals_are_ordinals RS Ord_0_mem_succ]) 1);
val natE0 = result();

goal Nat.thy "Ord(nat)";
by (rtac OrdI 1);
by (etac (naturals_are_ordinals RS Ord_is_Transset) 2);
by (rewtac Transset_def);
by (rtac ballI 1);
by (etac nat_induct 1);
by (REPEAT (ares_tac [empty_subsetI,succ_subsetI] 1));
val Ord_nat = result();

(** Variations on mathematical induction **)

val prem1::prems = goal Nat.thy
    "[| m: n;  n: nat;  \
\       P(m);  \
\       !!x. [| x: nat;  m: succ(x);  P(x) |] ==> P(succ(x)) \
\    |] ==> P(n)";
by (rtac (prem1 RS rev_mp) 1);
by (nat_ind_tac "n" prems 1);
by (fast_tac (ZF_cs addIs (prems@[Ord_nat, prem1 RS Ord_trans])) 2);
by (fast_tac ZF_cs 1);
val nat_induct2 = result();

val prems = goal Nat.thy
    "[| m <= n;  m: nat;  n: nat;  \
\       P(m);  \
\       !!x. [| x: nat;  m<=x;  P(x) |] ==> P(succ(x)) \
\    |] ==> P(n)";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl, naturals_are_ordinals,
			  member_succI RS succE, subst] 1));
by (eres_inst_tac [("n","n")] nat_induct2 1);
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [member_succD, Ord_nat RS Ord_in_Ord] 1));
val nat_induct3 = result();

(*Mathematical induction suitable for subtraction and less-than*)
val prems = goal Nat.thy
    "[| m: nat;  n: nat;  \
\       !!x. [| x: nat |] ==> P(x,0);  \
\       !!y. [| y: nat |] ==> P(0,succ(y));  \
\       !!x y. [| x: nat;  y: nat;  P(x,y) |] ==> P(succ(x),succ(y))  \
\    |] ==> P(m,n)";
by (res_inst_tac [("x","m")] bspec 1);
by (resolve_tac prems 2);
by (nat_ind_tac "n" prems 1);
by (rtac ballI 2);
by (nat_ind_tac "x" [] 2);
by (REPEAT (ares_tac (prems@[ballI]) 1 ORELSE etac bspec 1));
val diff_induct = result();

(** nat_case **)

goalw Nat.thy [nat_case_def] "nat_case(0,a,b) = a";
by (fast_tac (ZF_cs addIs [the_equality]) 1);
val nat_case_0_conv = result();

goalw Nat.thy [nat_case_def] "nat_case(succ(m),a,b) = b(m)";
by (fast_tac (ZF_cs addIs [the_equality]) 1);
val nat_case_succ_conv = result();

val major::prems = goal Nat.thy
    "[| n: nat;  a: C(0);  !!m. m: nat ==> b(m): C(succ(m))  \
\    |] ==> nat_case(n,a,b) : C(n)";
by (rtac (major RS nat_induct) 1);
by (REPEAT (resolve_tac [nat_case_0_conv RS ssubst,
			 nat_case_succ_conv RS ssubst] 1 
       THEN resolve_tac prems 1));
by (assume_tac 1);
val nat_case_type = result();

val prems = goalw Nat.thy [nat_case_def]
    "[| n=n';  a=a';  !!m z. b(m)=b'(m)  \
\    |] ==> nat_case(n,a,b)=nat_case(n',a',b')";
by (REPEAT (resolve_tac [the_cong,disj_cong,ex_cong] 1
     ORELSE EVERY1 (map rtac ((prems RL [ssubst]) @ [iff_refl]))));
val nat_case_cong = result();


(** nat_rec **)

val nat_rec_trans = wf_Memrel RS (nat_rec_def RS wfrec_def_conv RS trans);

goal Nat.thy "nat_rec(0,a,b) = a";
by (rtac nat_rec_trans 1);
by (rtac nat_case_0_conv 1);
val nat_rec_0_conv = result();

val [prem] = goal Nat.thy 
    "m: nat ==> nat_rec(succ(m),a,b) = b(m, nat_rec(m,a,b))";
val nat_rec_ss = ZF_ss 
      addcongs (mk_typed_congs Nat.thy [("b", "[i,i]=>i")])
      addrews [prem, nat_case_succ_conv, nat_succ_I,
	       Memrel_iff, under_iff];
by (rtac nat_rec_trans 1);
by (SIMP_TAC nat_rec_ss 1);
val nat_rec_succ_conv = result();

val major::prems = goal Nat.thy
    "[| n: nat;  \
\       a: C(0);  \
\       !!m z. [| m: nat;  z: C(m) |] ==> b(m,z): C(succ(m))  \
\    |] ==> nat_rec(n,a,b) : C(n)";
by (rtac (major RS nat_induct) 1);
by (SIMP_TAC (ZF_ss addrews (prems@[nat_rec_0_conv])) 1);
by (ASM_SIMP_TAC (ZF_ss addrews (prems@[nat_rec_succ_conv])) 1);
val nat_rec_type = result();

val prems = goalw Nat.thy [nat_rec_def,Memrel_def]
    "[| n=n';  a=a';  !!m z. b(m,z)=b'(m,z)  \
\    |] ==> nat_rec(n,a,b)=nat_rec(n',a',b')";
by (SIMP_TAC (ZF_ss addcongs [wfrec_cong,nat_case_cong] 
		    addrews (prems RL [sym])) 1);
val nat_rec_cong = result();

(** The union of two natural numbers is a natural number -- their maximum **)

(*  [| ?i : nat; ?j : nat |] ==> ?i Un ?j : nat  *)
val Un_nat_type = standard (Ord_nat RSN (3,Ord_member_UnI));

(*  [| ?i : nat; ?j : nat |] ==> ?i Int ?j : nat  *)
val Int_nat_type = standard (Ord_nat RSN (3,Ord_member_IntI));

