(*  Title: 	ZF/ex/sexp
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

S-expressions, i.e. general binary trees
*)

writeln"File ZF/ex/sexp";

structure Sexp =
struct

val const_decs = 
 [ 
   (["sexp","Atom"],	"i=>i"),
   (["sexp_case"],	"[i, i=>i, [i,i]=>i] => i") ,
   (["sexp_rec"],	"[i, i=>i, [i,i,i,i]=>i] => i") 
 ];

val thy = extend_theory Arith.thy "sexp"
     ([], [], [], [], const_decs, None)
 [ 
  ("sexp_def",		
   "sexp(A) == lfp(lam r: Pow(univ(A)). RepFun(A,Atom) Un r*r)" ),

  ("Atom_def",	"Atom(a) == {0,a}" ),

  ("sexp_case_def",	
   "sexp_case(M,c,d) == THE z. (EX x.   M=Atom(x) & z=c(x))  \
\                            | (EX N1 N2. M = <N1,N2>  & z=d(N1,N2))" ),

  ("sexp_rec_def",	
   "sexp_rec(M,c,d) ==   \
\       VVrec(M, %M g. sexp_case(M, c, %N1 N2. d(N1, N2, g`N1, g`N2)))" )
 ];
end;

local val ax = get_axiom Sexp.thy
in 
val sexp_def =  ax"sexp_def";
val Atom_def =  ax"Atom_def";
val sexp_case_def =      ax"sexp_case_def";
val sexp_rec_def =       ax"sexp_rec_def";
end;

(** Monotonicity and unfolding of the function **)

val prems = goalw Sexp.thy [Atom_def] "a: A ==> Atom(a) : univ(A)";
by (rtac (zero_in_univ RS doubleton_in_univ) 1);
by (rtac (A_subset_univ RS subsetD) 1);
by (resolve_tac prems 1);
val Atom_in_univ = result();

goal Sexp.thy
    "(lam r: Pow(univ(A)). RepFun(A,Atom) Un r*r)   \
\      : mono(Pow(univ(A)), Pow(univ(A)))";
by (rtac lam_mono_Powtype 1);
by (REPEAT (ares_tac [subset_refl, product_subset_univ, 
		      RepFun_subset, Atom_in_univ,
		      RepFun_mono, Un_mono, Un_least, product_mono] 1));
val sexpfun_mono = result();

goalw Sexp.thy [sexp_def] "sexp(A) = RepFun(A,Atom) Un sexp(A)*sexp(A)";
by (rtac (sexpfun_mono RS lam_Tarski_theorem) 1);
val sexp_unfold = result();


(** Type checking of "Atom" and Pair **)

val prems = goal Sexp.thy "a: A ==> Atom(a) : sexp(A)";
by (rtac (sexp_unfold RS ssubst) 1);
by (rtac (RepFunI RS UnI1) 1);
by (resolve_tac prems 1);
val sexp_AtomI = result();

val prems = goal Sexp.thy
    "[| M: sexp(A);  N: sexp(A) |] ==> <M,N> : sexp(A)";
by (rtac (sexp_unfold RS ssubst) 1);
by (rtac (SigmaI RS UnI2) 1);
by (REPEAT (resolve_tac prems 1));
val sexp_PairI = result();


(** Structural induction on sexps **)
val prems = goalw Sexp.thy [sexp_def]
  "[| M: sexp(A);  \
\     !!x. x: A ==> P(Atom(x));        \
\     !!N1 N2. [| N1: sexp(A);  N2: sexp(A);  P(N1);  P(N2) |] ==> P(<N1,N2>) \
\  |] ==> P(M)";
by (rtac lam_general_induction 1);
by (REPEAT (ares_tac (sexpfun_mono::prems) 1
     ORELSE eresolve_tac [UnE, RepFunE, SigmaE, CollectE, ssubst] 1));
val sexp_induct = result();

(*Perform induction on M, then prove the major premise using prems. *)
fun sexp_ind_tac a prems i = 
    EVERY [res_inst_tac [("M",a)] sexp_induct i,
	   rename_last_tac a ["1","2"] (i+2),
	   ares_tac prems i];


(** Injectivity properties **)

val [major] = goalw Sexp.thy [Atom_def,Pair_def]
    "Atom(a) = <M,N> ==> P";
by (cut_facts_tac [consI1 RS (major RS equalityD1 RS subsetD)] 1);
by (fast_tac (ZF_cs addEs [sym RS equals0D]) 1);
val Atom_neq_Pair = result();

val [major] = goalw Sexp.thy [Atom_def] "Atom(a) = Atom(b) ==>  a=b";
by (rtac (major RS equalityE) 1);
by (fast_tac ZF_cs 1);
val Atom_inject = result();

(** sexp_case **)

goalw Sexp.thy [sexp_case_def] "sexp_case(Atom(a),c,d) = c(a)";
by (fast_tac (ZF_cs addIs [the_equality] 
	            addEs [make_elim Atom_inject, Atom_neq_Pair]) 1);
val sexp_case_Atom_conv = result();

goalw Sexp.thy [sexp_case_def] "sexp_case(<M,N>, c, d) = d(M,N)";
by (fast_tac (ZF_cs addIs [the_equality] 
	            addEs [Pair_inject, sym RS Atom_neq_Pair]) 1);
val sexp_case_Pair_conv = result();

(*Type checking rules are perhaps unnecessary*)
val major::prems = goal Sexp.thy
    "[| M: sexp(A);  \
\       !!x. x: A ==> c(x) : C(Atom(x));        \
\       !!x y. [| x: sexp(A);  y: sexp(A) |] ==> d(x,y) : C(<x,y>) \
\    |] ==> sexp_case(M,c,d) : C(M)";
by (rtac (major RS sexp_induct) 1);
by (rtac (sexp_case_Pair_conv RS ssubst) 2);
by (rtac (sexp_case_Atom_conv RS ssubst) 1);
by (REPEAT (ares_tac prems 1));
val sexp_case_type = result();


(*** sexp_rec -- by VV recursion ***)

(*Used just to verify sexp_rec*)
val sexp_rec_ss = ZF_ss 
      addcongs (mk_typed_congs Sexp.thy [("h", "[i,i,i,i]=>i")])
      addrews [sexp_case_Atom_conv, sexp_case_Pair_conv];

(** conversion rules **)

goal Sexp.thy "sexp_rec(Atom(a), c, h) = c(a)";
by (rtac (sexp_rec_def RS VVrec_def_conv RS trans) 1);
by (SIMP_TAC sexp_rec_ss 1);
val sexp_rec_Atom_conv = result();

goal Sexp.thy
    "sexp_rec(<M,N>, c, h) = h(M, N, sexp_rec(M,c,h), sexp_rec(N,c,h))";
by (rtac (sexp_rec_def RS VVrec_def_conv RS trans) 1);
by (SIMP_TAC (sexp_rec_ss addrews [VV_rank_I, rank_pair1, rank_pair2]) 1);
val sexp_rec_Pair_conv = result();

val sexp_ss = ZF_ss addrews
  [sexp_case_Atom_conv,sexp_case_Pair_conv,
   sexp_rec_Atom_conv,sexp_rec_Pair_conv,
   sexp_AtomI, sexp_PairI];

(*Type checking.  This proof is vastly simpler than using wfrec_type*)
val prems = goal Sexp.thy
 "[| M: sexp(A);    \
\    !!x. x: A ==> c(x): C(Atom(x));       \
\    !!N1 N2 r1 r2. [| N1: sexp(A);  N2: sexp(A);  r1: C(N1);  r2: C(N2) \
\        	    |] ==> h(N1,N2,r1,r2): C(<N1,N2>) \
\ |] ==> sexp_rec(M,c,h) : C(M)";
by (sexp_ind_tac "M" prems 1);
by (SIMP_TAC (sexp_ss addrews prems) 1);
by (SIMP_TAC (sexp_ss addrews prems) 1);
val sexp_rec_type = result();
