(*  Title: 	ZF/epsilon
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

Epsilon induction and recursion
*)

structure Epsilon =
struct
val const_decs = 
    [ (["eclose","rank"],	"i=>i"),
      (["transrec"],		"[i, [i,i]=>i] =>i") ];

val thy = extend_theory Nat.thy "epsilon"
     ([], [], [], [], const_decs, None)
 [ 
  ("eclose_def",  "eclose(A) == UN n:nat. nat_rec(n, {A}, %m r. Union(r))"),
  ("transrec_def",  "transrec(a,H) == wfrec(Memrel(eclose(a)), a, H)"),
  ("rank_def",    "rank(a) == transrec(a, %x f. UN y:x. succ(f`y))")
 ];
end;

local val ax = get_axiom Epsilon.thy
in 
val eclose_def = ax"eclose_def";
val transrec_def  = ax"transrec_def";
val rank_def    = ax"rank_def";
end;

(*** Basic closure properties ***)

goalw Epsilon.thy [eclose_def] "A : eclose(A)";
by (rtac (nat_0_I RS UN_I) 1);
by (rtac (nat_rec_0_conv RS ssubst) 1);
by (rtac singletonI 1);
val A_in_eclose = result();

goalw Epsilon.thy [eclose_def,Transset_def] "Transset(eclose(A))";
by (rtac (subsetI RS ballI) 1);
by (etac UN_E 1);
by (rtac (nat_succ_I RS UN_I) 1);
by (assume_tac 1);
by (etac (nat_rec_succ_conv RS ssubst) 1);
by (etac UnionI 1);
by (assume_tac 1);
val Transset_eclose = result();

(* x : eclose(A) ==> x <= eclose(A) *)
val eclose_subset = 
    standard (rewrite_rule [Transset_def] Transset_eclose RS bspec);

(* [| A : eclose(B); c : A |] ==> c : eclose(B) *)
val ecloseD = standard (eclose_subset RS subsetD);

val [prem] = goal Epsilon.thy "A: B ==> A: eclose(B)";
by (rtac (prem RS (A_in_eclose RS ecloseD)) 1);
val A_into_eclose = result();

(* [| a: eclose(A);  !!x. [| x: eclose(A); ALL y:x. P(y) |] ==> P(x) 
   |] ==> P(a) *)
val eclose_induct = standard (Transset_eclose RSN (2, Transset_induct));

(*Epsilon induction*)
val prems = goal Epsilon.thy
    "[| !!x.[| ALL y:x. P(y) |] ==> P(x) \
\    |]  ==>  P(a)";
by (rtac (A_in_eclose RS eclose_induct) 1);
by (eresolve_tac prems 1);
val eps_induct = result();

(*Perform epsilon-induction on i. *)
fun eps_ind_tac a = 
    EVERY' [res_inst_tac [("a",a)] eps_induct,
	    rename_last_tac a ["1"]];


(*** Leastness of eclose ***)

val [transsx,ainx,nnat] = goalw Epsilon.thy [Transset_def]
    "[| Transset(X);  A:X;  n: nat |] ==> \
\    nat_rec(n, {A}, %m r. Union(r)) <= X";
(*** SPECIAL AUTO_TAC ***)
val ZF_ss' = set_auto_tac(ZF_ss,
	standard_auto_tac ORELSE' fast_tac (ZF_cs addIs [ainx]));
by (rtac (nnat RS nat_induct) 1);
by (SIMP_TAC (ZF_ss' addrews [nat_rec_0_conv]) 1);
by (ASM_SIMP_TAC (ZF_ss' addrews [nat_rec_succ_conv, Union_least,
				 transsx RS bspec]) 1);
val eclose_least_lemma = result();


val prems = goalw Epsilon.thy [eclose_def]
     "[| Transset(X);  A:X |] ==> eclose(A) <= X";
by (REPEAT (ares_tac (prems@[UN_least,eclose_least_lemma]) 1));
val eclose_least = result();


(*** Epsilon recursion ***)

val prems = goal Epsilon.thy
    "[| A: eclose(B);  B: eclose(C) |] ==> A: eclose(C)";
by (rtac (Transset_eclose RS eclose_least RS subsetD) 1);
by (resolve_tac prems 1);
by (resolve_tac prems 1);
val mem_eclose_trans = result();

val prems = goalw Epsilon.thy [Transset_def,under_def]
    "[| Transset(i);  j:i |] ==> under(Memrel(i), j) = j";
by (cut_facts_tac prems 1);
by (fast_tac (ZF_cs addSIs [equalityI,MemrelI] addSEs [MemrelE]) 1);
val under_Memrel = result();

(* j : eclose(A1) ==> under(Memrel(eclose(A1)),j) = j *)
val under_Memrel_eclose = Transset_eclose RS under_Memrel;

val wfrec_ssubst = standard (wf_Memrel RS wfrec_conv RS ssubst);

val [kmemj,jmemi] = goal Epsilon.thy
    "[| k:eclose(j);  j:eclose(i) |] ==> \
\    wfrec(Memrel(eclose(i)), k, H) = wfrec(Memrel(eclose(j)), k, H)";
by (rtac (kmemj RS Transset_induct) 1);
by (rtac Transset_eclose 1);
by (rtac wfrec_ssubst 1);
by (rtac wfrec_ssubst 1);
by (ASM_SIMP_TAC (wf_ss addrews [under_Memrel_eclose,
				 jmemi RSN (2,mem_eclose_trans)]) 1);
val wfrec_eclose_eq = result();

val [prem] = goal Epsilon.thy
    "k: i ==> wfrec(Memrel(eclose(i)),k,H) = wfrec(Memrel(eclose(k)),k,H)";
by (rtac (A_in_eclose RS wfrec_eclose_eq) 1);
by (rtac (prem RS A_into_eclose) 1);
val wfrec_eclose_succ_eq = result();

goalw Epsilon.thy [transrec_def]
    "transrec(a,H) = H(a, lam x:a. transrec(x,H))";
by (rtac wfrec_ssubst 1);
by (SIMP_TAC (wf_ss addrews [wfrec_eclose_succ_eq,
			     A_in_eclose, under_Memrel_eclose]) 1);
val transrec_conv = result();

(*This form avoids giant explosions in proofs.  NOTE USE OF == *)
val rew::prems = goal Epsilon.thy
    "[| !!x. f(x)==transrec(x,H) |] ==> f(a) = H(a, lam x:a. f(x))";
by (rewtac rew);
by (REPEAT (resolve_tac (prems@[transrec_conv]) 1));
val transrec_def_conv = result();

(** Two typing theorems **)

val prems = goal Epsilon.thy
    "[| !!x u. [| x:eclose(a);  u: Pi(x,B) |] ==> H(x,u) : B(x)   \
\    |]  ==> transrec(a,H) : B(a)";
by (res_inst_tac [("A", "a")] (A_in_eclose RS eclose_induct) 1);
by (rtac (transrec_conv RS ssubst) 1);
by (REPEAT (ares_tac (prems @ [lam_type]) 1 ORELSE etac bspec 1));
val transrec_type = result();


val prems = goal Epsilon.thy "Ord(i) ==> eclose(i) <= succ(i)";
by (rtac (Ord_is_Transset RS Transset_succ RS eclose_least) 1);
by (resolve_tac prems 1);
by (rtac succI1 1);
val eclose_Ord = result();

val prems = goal Epsilon.thy
    "[| j: i;  Ord(i);  \
\       !!x u. [| x: i;  u: Pi(x,B) |] ==> H(x,u) : B(x)   \
\    |]  ==> transrec(j,H) : B(j)";
by (rtac transrec_type 1);
by (resolve_tac prems 1);
by (rtac (Ord_in_Ord RS eclose_Ord RS subsetD RS succE) 1);
by (DEPTH_SOLVE (ares_tac prems 1 ORELSE eresolve_tac [ssubst,Ord_trans] 1));
val Ord_transrec_type = result();

(** Congruence **)

val prems = goalw Epsilon.thy [transrec_def,Memrel_def]
    "[| a=a';  !!x u. H(x,u)=H'(x,u) |]  ==> transrec(a,H)=transrec(a',H')";
val transrec_ss = 
    ZF_ss addcongs ([wfrec_cong] @ mk_congs Epsilon.thy ["eclose"])
	  addrews (prems RL [sym]);
by (SIMP_TAC transrec_ss 1);
val transrec_cong = result();

(*** Rank ***)

val ord_ss = ZF_ss addcongs (mk_congs Ord.thy ["Ord"]);

(*NOT SUITABLE FOR REWRITING -- RECURSIVE!*)
goal Epsilon.thy "rank(a) = (UN y:a. succ(rank(y)))";
by (rtac (rank_def RS transrec_def_conv RS ssubst) 1);
by (SIMP_TAC ZF_ss 1);
val rank_conv = result();

goal Epsilon.thy "Ord(rank(a))";
by (eps_ind_tac "a" 1);
by (rtac (rank_conv RS ssubst) 1);
by (rtac (Ord_succ RS Ord_UN) 1);
by (etac bspec 1);
by (assume_tac 1);
val Ord_rank = result();

val [major] = goal Epsilon.thy "Ord(i) ==> rank(i) = i";
by (rtac (major RS trans_induct) 1);
by (rtac (rank_conv RS ssubst) 1);
by (ASM_SIMP_TAC (ord_ss addrews [Ord_equality]) 1);
val rank_of_Ord = result();

val [prem] = goal Epsilon.thy "a:b ==> rank(a) : rank(b)";
by (res_inst_tac [("a1","b")] (rank_conv RS ssubst) 1);
by (rtac (prem RS UN_I) 1);
by (rtac succI1 1);
val rank_lt = result();


(*A COMPLETELY DIFFERENT induction principle from eclose_induct!!*)
val major::prems = goal Epsilon.thy
    "[| a: eclose(b);  P(b);                              \
\       !!y z. [| y: eclose(b);  P(y);  z: y |] ==> P(z)  \
\    |] ==> P(a)";
by (rtac (major RSN (3, eclose_least RS subsetD RS CollectD2)) 1);
by (rtac (A_in_eclose RS CollectI) 2);
by (resolve_tac prems 2);
by (rewtac Transset_def);
by (fast_tac (ZF_cs addEs (prems@[ecloseD])) 1);
val eclose_induct_down = result();

val [major] = goal Epsilon.thy "a: eclose(b) ==> rank(a) <= rank(b)";
by (rtac (major RS eclose_induct_down) 1);
by (rtac subset_refl 1);
by (rtac subset_trans 1);
by (assume_tac 2);
by (etac (rank_lt RS OrdmemD) 1);
by (rtac Ord_rank 1);
val rank_leq = result();

val [prem] = goal Epsilon.thy "a<=b ==> rank(a) <= rank(b)";
by (rtac (rank_conv RS ssubst) 1);
by (rtac (rank_conv RS ssubst) 1);
by (rtac (prem RS UN_mono) 1);
by (rtac subset_refl 1);
val rank_mono = result();

goal Epsilon.thy "rank(Pow(a)) = succ(rank(a))";
by (res_inst_tac [("a1","Pow(a)")] (rank_conv RS ssubst) 1);
by (rtac equalityI 1);
by (fast_tac ZF_cs 2);
by (rtac UN_least 1);
by (etac (PowD RS rank_mono RS Ord_succ_mono) 1);
by (rtac Ord_rank 1);
by (rtac Ord_rank 1);
val rank_Pow = result();

goal Epsilon.thy "rank(Union(A)) = (UN x:A. rank(x))";
by (rtac equalityI 1);
by (rtac (rank_mono RS UN_least) 2);
by (etac Union_upper 2);
by (rtac (rank_conv RS ssubst) 1);
by (rtac UN_least 1);
by (etac UnionE 1);
by (rtac subset_trans 1);
by (etac (RepFunI RS Union_upper) 2);
by (etac (rank_lt RS Ord_succ_subsetI) 1);
by (rtac Ord_rank 1);
val rank_Union = result();

goal Epsilon.thy "rank(eclose(a)) = succ(rank(a))";
by (rtac equalityI 1);
by (rtac (A_in_eclose RS rank_lt RS Ord_succ_subsetI) 2);
by (rtac Ord_rank 2);
by (res_inst_tac [("a1","eclose(a)")] (rank_conv RS ssubst) 1);
by (rtac UN_least 1);
by (etac (rank_leq RS Ord_succ_mono) 1);
by (rtac Ord_rank 1);
by (rtac Ord_rank 1);
val rank_eclose = result();

goalw Epsilon.thy [Pair_def] "rank(a) : rank(<a,b>)";
by (rtac (consI1 RS rank_lt RS Ord_trans) 1);
by (rtac (consI1 RS consI2 RS rank_lt) 1);
by (rtac Ord_rank 1);
val rank_pair1 = result();

goalw Epsilon.thy [Pair_def] "rank(b) : rank(<a,b>)";
by (rtac (consI1 RS consI2 RS rank_lt RS Ord_trans) 1);
by (rtac (consI1 RS consI2 RS rank_lt) 1);
by (rtac Ord_rank 1);
val rank_pair2 = result();

goalw Epsilon.thy [Inl_def] "rank(a) : rank(Inl(a))";
by (rtac rank_pair2 1);
val rank_Inl = result();

goalw Epsilon.thy [Inr_def] "rank(a) : rank(Inr(a))";
by (rtac rank_pair2 1);
val rank_Inr = result();

val [major] = goal Epsilon.thy "i: rank(a) ==> (EX x:a. i<=rank(x))";
by (resolve_tac ([major] RL [rank_conv RS subst] RL [UN_E]) 1);
by (rtac bexI 1);
by (etac member_succD 1);
by (rtac Ord_rank 1);
by (assume_tac 1);
val rank_implies_mem = result();


(*** Corollaries of leastness -- currently unused?? ***)

(*Note: monotonicity fails to hold*)
val [major] = goal Epsilon.thy "A:B ==> eclose(A)<=eclose(B)";
by (rtac (Transset_eclose RS eclose_least) 1);
by (rtac (major RS A_into_eclose) 1);
val mem_eclose_subset = result();

(** Repetition of eclose **)

goal Epsilon.thy "eclose(eclose(A)) <= succ(eclose(A))";
by (rtac (Transset_eclose RS Transset_succ RS eclose_least) 1);
by (rtac succI1 1);
val eclose2_subset = result();

goal Epsilon.thy "eclose(eclose(A)) = succ(eclose(A))";
by (rtac equalityI 1);
by (rtac eclose2_subset 1);
by (rtac (A_in_eclose RS succ_subsetI) 1);
by (rtac (A_in_eclose RS mem_eclose_subset) 1);
val eclose2_eq = result();

(*** Simple corollaries of downwards closure ***)

val [major] = goal Epsilon.thy "{a}: eclose(A) ==> a : eclose(A)";
by (rtac (major RS ecloseD) 1);
by (rtac singletonI 1);
val singleton_eclose = result();

val [major] = goal Epsilon.thy "{a,b}: eclose(A) ==> a : eclose(A)";
by (rtac (major RS ecloseD) 1);
by (rtac consI1 1);
val doubleton_eclose1 = result();

val [major] = goal Epsilon.thy "{a,b}: eclose(A) ==> b : eclose(A)";
by (rtac (major RS ecloseD) 1);
by (rtac (consI1 RS consI2) 1);
val doubleton_eclose2 = result();

val [major] = goalw Epsilon.thy [Pair_def]
    "<a,b>: eclose(A) ==> a : eclose(A)";
by (rtac (major RS doubleton_eclose1 RS doubleton_eclose1) 1);
val pair_eclose1 = result();

val [major] = goalw Epsilon.thy [Pair_def]
    "<a,b>: eclose(A) ==> b : eclose(A)";
by (rtac (major RS doubleton_eclose2 RS doubleton_eclose2) 1);
val pair_eclose2 = result();
