(*  Title: 	HOL/trancl
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1992  University of Cambridge

Transitive closure of a relation
*)

structure Trancl =
struct

val sext = 
  Sext{mixfix	= 
         [Infixr("O", "[('b * 'c)set, ('a * 'b)set] => ('a * 'c)set",  60)],
       parse_translation	= [],
       print_translation	= []};

val const_decs = 
 [
  (["trans"], 		 "('a * 'a)set => bool"),
  (["id"],		 "('a * 'a)set"),
  (["rtrancl","trancl"], "('a * 'a)set => ('a * 'a)set")
 ];

val thy = extend_theory Fixedpt.thy  "trancl" 
    ([], [], [], [], const_decs, Some sext)
 [    
  ("trans_def",     "trans(r) == (!x y z. <x,y>:r --> <y,z>:r --> <x,z>:r)"),

  ("comp_def",	(*composition of relations*)
      "r O s == {xz. ? x y z. xz=<x,z> & <x,y>:s & <y,z>:r}"),
  ("id_def",		(*the identity relation*)
      "id == {p. ? x. p=<x,x>}"),

  ("rtrancl_def", "rtrancl(r) == lfp(%s. id Un (r O s))"),

  ("trancl_def", "trancl(r) == r O rtrancl(r)")
 ];

end;

local val ax = get_axiom Trancl.thy
in
val trans_def = ax "trans_def";
val comp_def    = ax"comp_def";
val id_def      = ax"id_def";
val trancl_def = ax"trancl_def";
val rtrancl_def = ax"rtrancl_def";
end;


(** Natural deduction for trans(r) **)

val prems = goalw Trancl.thy [trans_def]
    "(!! x y z. [| <x,y>:r;  <y,z>:r |] ==> <x,z>:r) ==> trans(r)";
by (REPEAT (ares_tac (prems@[allI,impI]) 1));
val transI = result();

val major::prems = goalw Trancl.thy [trans_def]
    "[| trans(r);  <a,b>:r;  <b,c>:r |] ==> <a,c>:r";
by (cut_facts_tac [major] 1);
by (fast_tac (HOL_cs addIs prems) 1);
val transD = result();

(** Identity relation **)

goalw Trancl.thy [id_def] "<a,a> : id";  
by (rtac CollectI 1);
by (rtac exI 1);
by (rtac refl 1);
val idI = result();

val major::prems = goalw Trancl.thy [id_def]
    "[| p: id;  !!x.[| p=<x,x> |] ==> P  \
\    |] ==>  P";  
by (rtac (major RS CollectE) 1);
by (etac exE 1);
by (eresolve_tac prems 1);
val idE = result();

(** Composition of two relations **)

val prems = goalw Trancl.thy [comp_def]
    "[| <a,b>:s; <b,c>:r |] ==> <a,c> : r O s";
by (fast_tac (set_cs addIs prems) 1);
val compI = result();

(*proof requires higher-level assumptions or a delaying of hyp_subst_tac*)
val prems = goalw Trancl.thy [comp_def]
    "[| xz : r O s;  \
\       !!x y z. [| xz=<x,z>;  <x,y>:s;  <y,z>:r |] ==> P \
\    |] ==> P";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [CollectE, exE, conjE] 1 ORELSE ares_tac prems 1));
val compE = result();

val prems = goal Trancl.thy
    "[| <a,c> : r O s;  \
\       !!y. [| <a,y>:s;  <y,c>:r |] ==> P \
\    |] ==> P";
by (rtac compE 1);
by (REPEAT (ares_tac prems 1 ORELSE eresolve_tac [Pair_inject,ssubst] 1));
val compEpair = result();

val comp_cs = set_cs addIs [compI,idI] 
		       addEs [compE,idE] 
		       addSEs [Pair_inject];

val prems = goal Trancl.thy
    "[| r'<=r; s'<=s |] ==> (r' O s') <= (r O s)";
by (cut_facts_tac prems 1);
by (fast_tac comp_cs 1);
val comp_mono = result();

(** The relation rtrancl **)

goal Trancl.thy "mono(%s. id Un (r O s))";
by (rtac monoI 1);
by (REPEAT (ares_tac [monoI, subset_refl, comp_mono, Un_mono] 1));
val rtrancl_fun_mono = result();

val rtrancl_unfold = rtrancl_fun_mono RS (rtrancl_def RS Tarski_def_theorem);

(*Reflexivity of rtrancl*)
goal Trancl.thy "<a,a> : rtrancl(r)";
by (stac rtrancl_unfold 1);
by (fast_tac comp_cs 1);
val rtrancl_refl = result();

(*Closure under composition with r*)
val prems = goal Trancl.thy
    "[| <a,b> : rtrancl(r);  <b,c> : r |] ==> <a,c> : rtrancl(r)";
by (stac rtrancl_unfold 1);
by (fast_tac (comp_cs addIs prems) 1);
val rtrancl_into_rtrancl = result();

(*rtrancl of r contains r*)
val [prem] = goal Trancl.thy "[| <a,b> : r |] ==> <a,b> : rtrancl(r)";
by (rtac (rtrancl_refl RS rtrancl_into_rtrancl) 1);
by (rtac prem 1);
val r_into_rtrancl = result();


(** standard induction rule **)

val major::prems = goalw Trancl.thy [rtrancl_def]
  "[| <a,b> : rtrancl(r); \
\     !!x. P(<x,x>); \
\     !!x y z.[| P(<x,y>); <x,y>: rtrancl(r); <y,z>: r |]  ==>  P(<x,z>) |] \
\  ==>  P(<a,b>)";
by (rtac (rtrancl_fun_mono RS (major RS general_induction)) 1);
by (fast_tac (comp_cs addIs prems) 1);
val rtrancl_full_induct = result();

(*nice induction rule*)
val major::prems = goal Trancl.thy
    "[| <a::'a,b> : rtrancl(r);    \
\       P(a); \
\	!!y z.[| <a,y> : rtrancl(r);  <y,z> : r;  P(y) |] ==> P(z) |]  \
\     ==> P(b)";
(*by induction on this formula*)
by (subgoal_tac "! y. <a::'a,b> = <a,y> --> P(y)" 1);
(*now solve first subgoal: this formula is sufficient*)
by (fast_tac HOL_cs 1);
(*now do the induction*)
by (resolve_tac [major RS rtrancl_full_induct] 1);
by (fast_tac (comp_cs addIs prems) 1);
by (fast_tac (comp_cs addIs prems) 1);
val rtrancl_induct = result();

(*transitivity of transitive closure!! -- by induction.*)
goal Trancl.thy "trans(rtrancl(r))";
by (rtac transI 1);
by (res_inst_tac [("b","z")] rtrancl_induct 1);
by (DEPTH_SOLVE (eresolve_tac [asm_rl, rtrancl_into_rtrancl] 1));
val trans_rtrancl = result();

(*elimination of rtrancl -- by induction on a special formula*)
val major::prems = goal Trancl.thy
    "[| <a::'a,b> : rtrancl(r);  (a = b) ==> P; \
\	!!y.[| <a,y> : rtrancl(r); <y,b> : r |] ==> P |] \
\    ==> P";
by (subgoal_tac "a::'a = b  | (? y. <a,y> : rtrancl(r) & <y,b> : r)" 1);
by (rtac (major RS rtrancl_induct) 2);
by (fast_tac (set_cs addIs prems) 2);
by (fast_tac (set_cs addIs prems) 2);
by (REPEAT (eresolve_tac ([asm_rl,exE,disjE,conjE]@prems) 1));
val rtranclE = result();


(**** The relation trancl ****)

(** Conversions between trancl and rtrancl **)

val [major] = goalw Trancl.thy [trancl_def]
    "[| <a,b> : trancl(r) |] ==> <a,b> : rtrancl(r)";
by (resolve_tac [major RS compEpair] 1);
by (REPEAT (ares_tac [rtrancl_into_rtrancl] 1));
val trancl_into_rtrancl = result();

(*trancl(r) contains r*)
val [prem] = goalw Trancl.thy [trancl_def]
   "[| <a,b> : r |] ==> <a,b> : trancl(r)";
by (REPEAT (ares_tac [prem,compI,rtrancl_refl] 1));
val r_into_trancl = result();

(*intro rule by definition: from rtrancl and r*)
val prems = goalw Trancl.thy [trancl_def]
    "[| <a,b> : rtrancl(r);  <b,c> : r |]   ==>  <a,c> : trancl(r)";
by (REPEAT (resolve_tac ([compI]@prems) 1));
val rtrancl_into_trancl1 = result();

(*intro rule from r and rtrancl*)
val prems = goal Trancl.thy
    "[| <a,b> : r;  <b,c> : rtrancl(r) |]   ==>  <a,c> : trancl(r)";
by (resolve_tac (prems RL [rtranclE]) 1);
by (etac subst 1);
by (resolve_tac (prems RL [r_into_trancl]) 1);
by (rtac (trans_rtrancl RS transD RS rtrancl_into_trancl1) 1);
by (REPEAT (ares_tac (prems@[r_into_rtrancl]) 1));
val rtrancl_into_trancl2 = result();

(*elimination of trancl(r) -- NOT an induction rule*)
val major::prems = goal Trancl.thy
    "[| <a::'a,b> : trancl(r);  \
\       <a,b> : r ==> P; \
\	!!y.[| <a,y> : trancl(r);  <y,b> : r |] ==> P  \
\    |] ==> P";
by (subgoal_tac "<a::'a,b> : r | (? y. <a,y> : trancl(r)  &  <y,b> : r)" 1);
by (REPEAT (eresolve_tac ([asm_rl,disjE,exE,conjE]@prems) 1));
by (rtac (rewrite_rule [trancl_def] major RS compEpair) 1);
by (etac rtranclE 1);
by (fast_tac comp_cs 1);
by (fast_tac (comp_cs addSIs [rtrancl_into_trancl1]) 1);
val tranclE = result();

(*Transitivity of trancl(r).
  Proved by unfolding since it uses transitivity of rtrancl. *)
goalw Trancl.thy [trancl_def] "trans(trancl(r))";
by (rtac transI 1);
by (REPEAT (etac compEpair 1));
by (rtac (rtrancl_into_rtrancl RS (trans_rtrancl RS transD RS compI)) 1);
by (REPEAT (assume_tac 1));
val trans_trancl = result();

val prems = goal Trancl.thy
    "[| <a,b> : r;  <b,c> : trancl(r) |]   ==>  <a,c> : trancl(r)";
by (rtac (r_into_trancl RS (trans_trancl RS transD)) 1);
by (resolve_tac prems 1);
by (resolve_tac prems 1);
val trancl_into_trancl2 = result();
