(*  Title: 	ZF/ex/wf.ML
    Author: 	Tobias Nipkow and Lawrence C Paulson
    Copyright   1991  University of Cambridge

Well-founded Recursion
*)

structure WF =
struct
val const_decs = 
  [(["wf"], 		"i=>o"),
   (["trans"], 		"i=>o"),
   (["under"], 		"[i,i]=>i"),
   (["wfrec"], 		"[i, i, [i,i]=>i] =>i"),
   (["is_recfun"], 	"[i, [i,i]=>i, i, i] =>o"),
   (["the_recfun"], 	"[i, [i,i]=>i, i] =>i")];

val thy = extend_theory plus_thy "WF_Rec"
 ([], [], [], const_decs, None)
 [
  ("wf_def",  (*r is a well-founded relation*)
     "wf(r) == ALL Z. Z=0 | (EX x:Z. ALL y. <y,x>:r --> ~ y:Z)"),

    (*should use --> instead of & below*)
  ("trans_def",	"trans(r) == ALL x y z. <x,y>: r & <y,z>: r --> <x,z>: r"),
  ("under_def",	"under(r,x) == {y: domain(r). <y,x>:r}"),
  ("is_recfun_def",  
   "is_recfun(r,H,f,a) == \
\   (f = (lam x: under(r,a). H(x, restrict(f, under(r,x)))))"),
  ("the_recfun_def",	"the_recfun(r,H,a) == (THE f.is_recfun(r,H,f,a))"),
  ("wfrec_def", "wfrec(r,a,H) == H(a, the_recfun(r,H,a))")
 ];
end;

local val ax = get_axiom WF.thy
in
val wf_def =    ax"wf_def";
val trans_def = ax"trans_def";
val under_def = ax"under_def";
val is_recfun_def =     ax"is_recfun_def";
val the_recfun_def =    ax"the_recfun_def";
val wfrec_def = ax"wfrec_def";
end;

val [H_cong] = mk_typed_congs WF.thy[("H","[i,i]=>i")];

val wf_ss = ZF_ss addcongs [H_cong];


(*** Well-founded relations ***)

(*If every subset of field(r) possesses an r-minimal element then wf(r).
  Seems impossible to prove this for domain(r) instead...*)
val [prem1,prem2] = goalw WF.thy [wf_def]
    "[| field(r)<=A;  \
\       !!Z u. [| Z<=A;  u:Z;  ALL x:Z. EX y:Z. <y,x>:r |] ==> False |] \
\    ==>  wf(r)";
by (rtac (equals0I RS disjCI RS allI) 1);
by (rtac prem2 1);
by (res_inst_tac [ ("B", "Z") ] (prem1 RS Int_lowerbound1) 1);
by (fast_tac ZF_cs 1);
by (fast_tac ZF_cs 1);
val wfI = result();

(*If r allows well-founded induction then wf(r)*)
val [prem1,prem2] = goal WF.thy
    "[| field(r)<=A;  \
\       !!B. ALL x:A. (ALL y. <y,x>: r --> y:B) --> x:B ==> A<=B |]  \
\    ==>  wf(r)";
by (rtac (prem1 RS wfI) 1);
by (res_inst_tac [ ("B", "A-Z") ] (prem2 RS subsetCE) 1);
by (fast_tac ZF_cs 3);
by (fast_tac ZF_cs 2);
by (fast_tac ZF_cs 1);
val wfI2 = result();


(** Well-founded Induction **)

(*Consider the least z in domain(r) Un {a} such that P(z) does not hold...*)
val major::prems = goalw WF.thy [wf_def]
    "[| wf(r);          \
\       !!x.[| ALL y. <y,x>: r --> P(y) |] ==> P(x) \
\    |]  ==>  P(a)";
by (res_inst_tac [ ("x", "{z:domain(r) Un {a}. ~P(z)}") ]  (major RS allE) 1);
by (etac disjE 1);
by (rtac classical 1);
by (etac equals0D 1);
by (etac (singletonI RS UnI2 RS CollectI) 1);
by (etac bexE 1);
by (etac CollectE 1);
by (etac swap 1);
by (resolve_tac prems 1);
by (fast_tac ZF_cs 1);
val wf_induct = result();

(*Perform induction on i, then prove the wf(r) subgoal using prems. *)
fun wf_ind_tac a prems i = 
    EVERY [res_inst_tac [("a",a)] wf_induct i,
	   rename_last_tac a ["1"] (i+1),
	   ares_tac prems i];

(*** the following might be provable, or perhaps with <x,a>: rtrancl(r) 
    "[| wf(r);  trans(r);  \
\       !!x.[| <x,a>: r;  ALL y. <y,x>: r --> P(y) |] ==> P(x) \
\    |]  ==>  a: domain(r) --> P(a)";
***)

val wfr::amem::prems = goal WF.thy
    "[| wf(r);  a:A;  field(r)<=A;  \
\       !!x.[| x: A;  ALL y. <y,x>: r --> P(y) |] ==> P(x) \
\    |]  ==>  P(a)";
by (rtac (amem RS rev_mp) 1);
by (wf_ind_tac "a" [wfr] 1);
by (rtac impI 1);
by (eresolve_tac prems 1);
by (fast_tac (ZF_cs addIs (prems RL [subsetD])) 1);
val wf_induct2 = result();

val major::prems = goalw WF.thy [trans_def]
    "[| trans(r);  <a,b>:r;  <b,c>:r |] ==> <a,c>:r";
by (rtac (major RS spec RS spec RS spec RS mp) 1);
by (REPEAT (resolve_tac (prems@[conjI]) 1));
val transD = result();

val prems = goal WF.thy "[| wf(r);  <a,x>:r;  <x,a>:r |] ==> False";
by (subgoal_tac "ALL x. <a,x>:r --> <x,a>:r --> False" 1);
by (wf_ind_tac "a" prems 2);
by (fast_tac ZF_cs 2);
by (fast_tac (FOL_cs addIs prems) 1);
val wf_anti_sym = result();

(** under **)

val prems = goalw WF.thy [under_def] "b : under(r,a) <-> <b,a>:r";
by (fast_tac ZF_cs 1);
val under_iff = result();

val underI = standard (under_iff RS iffD1);
val underD = standard (under_iff RS iffD2);

val trans_tac = EVERY' [etac transD, atac, atac];

(**NOTE! the following simplifications need transitivity reasoning!**)
auto_tac_ref := (standard_auto_tac ORELSE' trans_tac);

(* is_recfun Lemmas *)
val prems = goalw WF.thy [is_recfun_def]
    "[| wf(r); trans(r); is_recfun(r,H,f,a); is_recfun(r,H,g,b) |] ==> \
\    <x,a>:r --> <x,b>:r --> f`x=g`x";
by (cut_facts_tac prems 1);
by (wf_ind_tac "x" prems 1);
by (REPEAT (rtac impI 1 ORELSE etac ssubst 1));
(*check setting of auto_tac_ref!*)
by (ASM_SIMP_TAC (ZF_ss addrews [under_iff]) 1); 
by (REPEAT (dtac underD 1
     ORELSE resolve_tac [refl, restrict_eqI, H_cong] 1));
(*check setting of auto_tac_ref!*)
by (ASM_SIMP_TAC (wf_ss addrews [under_iff]) 1);
val is_recfun_equal_lemma = result();
val is_recfun_equal = standard (is_recfun_equal_lemma RS mp RS mp);

val [major] = goalw WF.thy [is_recfun_def]
    "[| is_recfun(r,H,f,a) |] ==> f: under(r,a) -> range(f)";
by (rtac (major RS ssubst) 1);
by (rtac (lamI RS rangeI RS lam_type) 1);
by (assume_tac 1);
val is_recfun_type = result();

val prems as [wfr,transr,recf,recg,_] = goal WF.thy
    "[| wf(r); trans(r);       \
\       is_recfun(r,H,f,a); is_recfun(r,H,g,b); <b,a>:r |] ==> \
\    restrict(f, under(r,b)) = g";
by (cut_facts_tac prems 1);
by (rtac fun_extension 1);
val rec_equal = recg RS (recf RS (transr RS (wfr RS is_recfun_equal)));
by (rtac (rec_equal RS subst) 3);
by (REPEAT (ares_tac [restrict_type,apply_type,is_recfun_type,underI,
		      restrict_conv] 1
     ORELSE trans_tac 1 ORELSE dtac underD 1));
val is_recfun_cut = result();

val prems = goal WF.thy
    "[| wf(r); trans(r); is_recfun(r,H,f,a); is_recfun(r,H,g,a) |]  ==>  f=g";
by (cut_facts_tac prems 1);
by (rtac fun_extension 1);
by (REPEAT (ares_tac [is_recfun_equal] 1
     ORELSE eresolve_tac [is_recfun_type,underD] 1));
val is_recfun_functional = result();

(*** Main Existence Lemma ***)

val [major] = goalw WF.thy [is_recfun_def,restrict_def]
    "[| is_recfun(r,H,f,a) |] ==> restrict(f,under(r,a)) = f";
by (rtac (major RS ssubst) 1);
by (SIMP_TAC ZF_ss 1);    (*check setting of auto_tac_ref!*)
val cut_recfun = result();

val [isrec,rel] = goalw WF.thy [is_recfun_def]
    "[| is_recfun(r,H,f,a); <x,a>:r |] ==> f`x = H(x, restrict(f,under(r,x)))";
(*This tactic causes backtracking over substitution occurrences!*)
by (rtac (isrec RS ssubst) 1 THEN rtac (rel RS underI RS beta_conv) 1);
val apply_recfun = result();

val prems = goalw WF.thy [the_recfun_def]
    "[| is_recfun(r,H,f,a); wf(r); trans(r) |]  ==> the_recfun(r,H,a) = f";
by (REPEAT (ares_tac (prems@[the_equality,is_recfun_functional]) 1));
val the_recfun = result();

val prems = goal WF.thy "[| wf(r); trans(r) |] ==> EX f.is_recfun(r,H,f,a)";
by (cut_facts_tac prems 1);
by (wf_ind_tac "a" prems 1);
by (res_inst_tac
    [("x", "lam y: under(r,a1). \
\             H(y, restrict(the_recfun(r,H,y), under(r,y)))")] exI 1);
by (rewtac is_recfun_def);
by (REPEAT (dtac underD 1
     ORELSE resolve_tac [refl, lam_cong, restrict_eqI, H_cong] 1));
by (fold_tac [is_recfun_def]);
by (EVERY' [rtac (spec RS impE), atac, atac, etac exE] 1);
by (EVERY' [etac (spec RS impE), trans_tac, etac exE] 1);
(*check setting of auto_tac_ref!*)
by (ASM_SIMP_TAC (wf_ss addrews [the_recfun, underI RS beta_conv,
				 apply_recfun,is_recfun_cut,cut_recfun]) 1);
val ex_is_recfun = result();

auto_tac_ref := standard_auto_tac;    (** RESTORATION OF AUTO_TAC **)

(*** Properties of the_recfun ***)

val prems = goal WF.thy
     "[| !!f. is_recfun(r,H,f,a) ==> P(f);  wf(r);  trans(r) |] ==> \
\     P(the_recfun(r,H,a))";
by (rtac (ex_is_recfun RS exE) 1 THEN REPEAT (resolve_tac prems 1));
by (rtac (the_recfun RS ssubst) 1 THEN REPEAT (ares_tac prems 1));
val the_recfunI = result();

val prems = goal WF.thy
    "[| wf(r); trans(r);  <c,a>:r;  <c,b>:r|] ==> \
\    the_recfun(r,H,a) ` c = the_recfun(r,H,b) ` c";
by (rtac the_recfunI 1);
by (rtac the_recfunI 1);
by (REPEAT (ares_tac (prems@[is_recfun_equal]) 1));
val the_recfun_equal = result();

val prems = goal WF.thy
    "[| wf(r);  trans(r);  <b,a>:r |] ==> \
\    restrict(the_recfun(r,H,a), under(r,b)) = the_recfun(r,H,b)";
by (rtac the_recfunI 1);
by (rtac the_recfunI 1);
by (REPEAT (ares_tac (prems@[is_recfun_cut]) 1));
val the_recfun_cut = result();

val prems = goalw WF.thy [the_recfun_def]
    "[|wf(r);  trans(r)|] ==> is_recfun(r, H, the_recfun(r,H,a), a)";
by (rtac (ex_is_recfun RS exE) 1);
by (rtac (ex1I RS theI) 3);
by (REPEAT (ares_tac (prems@[is_recfun_functional]) 1));
val unfold_the_recfun = result();

(*** Unfolding wfrec ***)

val prems = goalw WF.thy [wfrec_def]
    "[| wf(r);  trans(r) |] ==> \
\    wfrec(r,a,H) = H(a, lam x: under(r,a). wfrec(r,x,H))";
by (rtac (rewrite_rule [is_recfun_def] unfold_the_recfun RS ssubst) 1);
frs prems;
frs prems;
by (SIMP_TAC (wf_ss addrews (prems@
         [under_iff RS iff_sym, the_recfun_cut])) 1);
val wfrec_conv = result();

(*This form avoids giant explosions in proofs.  NOTE USE OF == *)
val rew::prems = goal WF.thy
    "[| !!x. f(x)==wfrec(r,x,H);  wf(r);  trans(r) |] ==> \
\    f(a) = H(a, lam x: under(r,a). f(x))";
by (rewtac rew);
by (REPEAT (resolve_tac (prems@[wfrec_conv]) 1));
val wfrec_def_conv = result();

val prems = goal WF.thy
    "[| wf(r);  trans(r);  a:A;  field(r)<=A;  \
\       !!x u. [| x: A;  u: Pi(under(r,x), B) |] ==> H(x,u) : B(x)   \
\    |] ==> wfrec(r,a,H) : B(a)";
by (res_inst_tac [("a","a")] wf_induct2 1);
by (rtac (wfrec_conv RS ssubst) 4);
by (REPEAT (ares_tac (prems@[lam_type]) 1
     ORELSE eresolve_tac [spec RS mp, underD] 1));
val wfrec_type = result();

val prems = goalw WF.thy [wfrec_def,the_recfun_def,is_recfun_def,under_def]
    "[| r=r';  !!x u. H(x,u)=H'(x,u);  a=a' |] \
\    ==> wfrec(r,a,H)=wfrec(r',a',H')";
by (SIMP_TAC (wf_ss addrews (prems RL [sym])) 1);
val wfrec_cong = result();
