(*  Title: 	ZF/ex/sexp
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

S-expressions, i.e. general binary trees
*)

writeln"File ZF/ex/sexp";

structure Sexp =
struct
val mixfix = 
    [ Infixr(".",	"[i,i]=>i",	60) ];

val sext = Sext{mixfix			= mixfix,
		parse_translation	= [],
		print_translation	= []};

val const_decs = 
 [ 
   (["sexp","Atom"],	"i=>i"),
   (["sexp_case"],	"[i, i=>i, [i,i]=>i] => i") ,
   (["pred_sexp"],	"i => i"),
   (["sexp_rec"],	"[i, i, i=>i, [i,i,i,i]=>i] => i") 
 ];

val thy = extend_theory Univ.thy "sexp"
     ([], [], [], const_decs, Some sext)
 [ 
  ("sexp_def",		
   "sexp(A) == lfp(lam r: Pow(univ(A)). univ(A) + r*r)" ),

  ("Atom_def",	"Atom(a) == Inl(a)" ),

  ("Scons_def",	"M.N == Inr(<M,N>)" ),

  ("sexp_case_def",	
   "sexp_case(M,c,d) == THE z. (EX x.   M=Atom(x) & z=c(x))  \
\                            | (EX N1 N2. M = N1.N2  & z=d(N1,N2))" ),

  ("pred_sexp_def",	
   "pred_sexp(A) == UN M: sexp(A). UN N: sexp(A). {<M, M.N>, <N, M.N>}" ),

  ("sexp_rec_def",	
   "sexp_rec(A,M,c,d) == wfrec(trancl(pred_sexp(A)),  M, \
\             %M rec. sexp_case(M, c, %N1 N2. d(N1, N2, rec`N1, rec`N2)))" )
 ];
end;

local val ax = get_axiom Sexp.thy
in 
val sexp_def =  ax"sexp_def";
val Atom_def =  ax"Atom_def";
val Scons_def = ax"Scons_def";
val sexp_case_def =      ax"sexp_case_def";
val pred_sexp_def =       ax"pred_sexp_def";
val sexp_rec_def =       ax"sexp_rec_def";
end;


(** Monotonicity and unfolding of the function **)

goal Sexp.thy
    "(lam r: Pow(univ(A)). univ(A) + r*r) : mono(Pow(univ(A)), Pow(univ(A)))";
by (rtac lam_mono_Powtype 1);
by (REPEAT (ares_tac [subset_refl, plus_subset_univ, product_subset_univ, 
		      plus_mono, Sigma_mono] 1));
val sexpfun_mono = result();

goalw Sexp.thy [sexp_def] "sexp(A) = univ(A) + sexp(A)*sexp(A)";
by (rtac (sexpfun_mono RS lam_Tarski_theorem) 1);
val sexp_unfold = result();

(** Type checking of "Atom" and "." **)

val prems = goalw Sexp.thy [Atom_def]
    "a: univ(A) ==> Atom(a) : sexp(A)";
by (rtac (sexp_unfold RS ssubst) 1);
by (rtac plus_InlI 1);
by (resolve_tac prems 1);
val sexp_AtomI = result();

val prems = goalw Sexp.thy [Scons_def]
    "[| M: sexp(A);  N: sexp(A) |] ==> (M.N) : sexp(A)";
by (rtac (sexp_unfold RS ssubst) 1);
by (REPEAT (resolve_tac (prems @ [SigmaI, plus_InrI]) 1));
val sexp_SconsI = result();


(** Structural induction on sexps **)
val prems = goalw Sexp.thy [sexp_def,Atom_def,Scons_def]
  "[| M: sexp(A);  \
\     !!x. x: univ(A) ==> P(Atom(x));        \
\     !!N1 N2. [| N1: sexp(A);  N2: sexp(A);  P(N1);  P(N2) |] ==> P(N1.N2) \
\  |] ==> P(M)";
by (rtac lam_general_induction 1);
by (REPEAT (ares_tac (sexpfun_mono::prems) 1
     ORELSE eresolve_tac [plusE, SigmaE, CollectE, ssubst] 1));
val sexp_induct = result();

(*Perform induction on M, then prove the major premise using prems. *)
fun sexp_ind_tac a prems i = 
    EVERY [res_inst_tac [("M",a)] sexp_induct i,
	   rename_last_tac a ["1","2"] (i+2),
	   ares_tac prems i];


(** Injectivity properties **)

val [major] = goalw Sexp.thy [Atom_def,Scons_def]
    "Atom(a) = M.N ==> P";
by (rtac (major RS Inl_neq_Inr) 1);
val Atom_neq_Scons = result();

val [major] = goalw Sexp.thy [Atom_def]
    "Atom(a) = Atom(b) ==>  a=b";
by (rtac (major RS Inl_inject) 1);
val Atom_inject = result();

val [major] = goalw Sexp.thy [Scons_def]
    "M.N = M'.N' ==> M=M'";
by (rtac (major RS Inr_inject RS Pair_inject1) 1);
val Scons_inject1 = result();

val [major] = goalw Sexp.thy [Scons_def]
    "M.N = M'.N' ==> N=N'";
by (rtac (major RS Inr_inject RS Pair_inject2) 1);
val Scons_inject2 = result();

val [major,minor] = goal Sexp.thy
    "[| M.N = M'.N';  [| M=M'; N=N' |] ==> P |] ==> P";
by (rtac minor 1);
by (rtac (major RS Scons_inject1) 1);
by (rtac (major RS Scons_inject2) 1);
val Scons_inject = result();

(** sexp_case **)

goalw Sexp.thy [sexp_case_def] "sexp_case(Atom(a),c,d) = c(a)";
by (fast_tac (ZF_cs addIs [the_equality] 
	            addEs [make_elim Atom_inject, Atom_neq_Scons]) 1);
val sexp_case_Atom_conv = result();

goalw Sexp.thy [sexp_case_def] "sexp_case(M.N, c, d) = d(M,N)";
by (fast_tac (ZF_cs addIs [the_equality] 
	            addEs [Scons_inject, sym RS Atom_neq_Scons]) 1);
val sexp_case_Scons_conv = result();

val major::prems = goal Sexp.thy
    "[| M: sexp(A);  \
\       !!x. x: univ(A) ==> c(x) : C(Atom(x));        \
\       !!x y. [| x: sexp(A);  y: sexp(A) |] ==> d(x,y) : C(x.y) \
\    |] ==> sexp_case(M,c,d) : C(M)";
by (rtac (major RS sexp_induct) 1);
by (rtac (sexp_case_Scons_conv RS ssubst) 2);
by (rtac (sexp_case_Atom_conv RS ssubst) 1);
by (REPEAT (ares_tac prems 1));
val sexp_case_type = result();


(*** The pred_sexp relation ***)

goalw Sexp.thy [pred_sexp_def] "pred_sexp(A) <= sexp(A) * sexp(A)";
by (fast_tac (ZF_cs addIs [sexp_SconsI]) 1);
val pred_sexp_type = result();

(** Introduction rules for 'pred_sexp' **)

val prems = goalw Sexp.thy [pred_sexp_def] 
    "[| M: sexp(A);  N: sexp(A) |] ==> <M, M.N> : pred_sexp(A)";
by (fast_tac (ZF_cs addIs prems) 1);
val pred_sexpI1 = result();

val prems = goalw Sexp.thy [pred_sexp_def] 
    "[| M: sexp(A);  N: sexp(A) |] ==> <N, M.N> : pred_sexp(A)";
by (fast_tac (ZF_cs addIs prems) 1);
val pred_sexpI2 = result();

val major::prems = goalw Sexp.thy [pred_sexp_def]
    "[| p : pred_sexp(A);  \
\       !!M N. [| M: sexp(A);  N: sexp(A);  p = <M, M.N> |] ==> R; \
\       !!M N. [| M: sexp(A);  N: sexp(A);  p = <N, M.N> |] ==> R  \
\    |] ==> R";
by (cut_facts_tac [major] 1);
by (DEPTH_SOLVE (eresolve_tac [UN_E,emptyE,consE] 1
          ORELSE ares_tac prems 1));
val pred_sexpE = result();

goal Sexp.thy "wf(pred_sexp(A))";
by (rtac (pred_sexp_type RS field_rel_subset RS wfI2) 1);
by (rtac subsetI 1);
by (etac sexp_induct 1);
by (fast_tac (ZF_cs addIs [sexp_SconsI]
	            addSEs [bspec RS mp, pred_sexpE, Scons_inject]) 2);
by (fast_tac (ZF_cs addIs [sexp_AtomI]
	            addSEs [bspec RS mp, pred_sexpE, Atom_neq_Scons]) 1);
val wf_pred_sexp = result();


(*** sexp_rec -- by wf recursion on pred_sexp ***)

(*Used just to verify sexp_rec*)
val sexp_rec_ss = ZF_ss 
      addcongs (mk_typed_congs Sexp.thy [("h", "[i,i,i,i]=>i")])
      addrews [trans_trancl, wf_pred_sexp RS wf_trancl, 
	       sexp_rec_def RS wfrec_def_conv,
	       sexp_case_Atom_conv, sexp_case_Scons_conv];

(** conversion rules **)

(*The premise a:univ(A) is NOT required!*)
goal Sexp.thy "sexp_rec(A, Atom(a), c, h) = c(a)";
by (SIMP_TAC sexp_rec_ss 1);
val sexp_rec_Atom_conv = result();

val prems = goal Sexp.thy
    "[| M: sexp(A);  N: sexp(A) |] ==> \
\    sexp_rec(A, M.N, c, h) = h(M, N, sexp_rec(A,M,c,h), sexp_rec(A,N,c,h))";
by (SIMP_TAC (sexp_rec_ss addrews 
		([under_iff,r_sub_trancl,pred_sexpI1,pred_sexpI2]@prems)) 1);
val sexp_rec_Scons_conv = result();

val sexp_ss = ZF_ss addrews
  [sexp_case_Atom_conv,sexp_case_Scons_conv,
   sexp_rec_Atom_conv,sexp_rec_Scons_conv,
   sexp_AtomI, sexp_SconsI];

(*Type checking.  This proof is vastly simpler than using wfrec_type*)
val prems = goal Sexp.thy
 "[| M: sexp(A);    \
\    !!x. x: univ(A) ==> c(x): C(Atom(x));       \
\    !!N1 N2 r1 r2. [| N1: sexp(A);  N2: sexp(A);  \
\        	   r1: C(N1);  r2: C(N2) |] ==> h(N1,N2,r1,r2): C(N1.N2) \
\ |] ==> sexp_rec(A,M,c,h) : C(M)";
by (sexp_ind_tac "M" prems 1);
by (SIMP_TAC (sexp_ss addrews prems) 1);
by (SIMP_TAC (sexp_ss addrews prems) 1);
val sexp_rec_type = result();
