(*  Title: 	ZF/ex/fixedpt
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

The Knaster-Tarski Theorem and the Schroeder-Bernstein Theorem

Proved in the lattice of subsets of D, namely Pow(D), with Inter as glb
*)

writeln"File ZF/ex/fixedpt";

structure Fixedpt =
struct
val const_decs = 
  [ (["mono"],	"[i,i]=>i"),
    (["lfp"],	"i=>i") ];

val thy = extend_theory Ord.thy  "fixedpt" 
    ([], [], [], const_decs, None)
  [ 
    ("mono_def",	(*monotone functions from A to B*)
     "mono(A,B) == { f: A->B. ALL w:A. ALL x:A. w<=x --> f`w<=f`x}"),

    ("lfix_def", "lfp(f) == Inter({u: domain(f). f`u <= u})") ];

end;

local val ax = get_axiom Fixedpt.thy
in
  val mono_def = ax"mono_def";
  val lfix_def = ax"lfix_def";
end;


(*** Monotone function space ***)

val prems = goalw Fixedpt.thy [mono_def]
    "[| f: A->B;  \
\       !!w x. [| w: A;  x: A;  w<=x |] ==> f`w<=f`x  \
\    |] ==> f: mono(A,B)";
by (REPEAT (ares_tac (prems@[CollectI,impI,ballI]) 1));
val monoI = result();

val [major] = goalw Fixedpt.thy [mono_def]
    "f: mono(A,B) ==> f: A-> B";
by (rtac (major RS CollectE) 1);
by (assume_tac 1);
val mono_type = result();

val major::prems = goalw Fixedpt.thy [mono_def]
    "[| f: mono(A,B);  a<=c;  a:A;  c:A \
\    |] ==> f`a <= f`c";
by (rtac (major RS CollectE) 1);
by (etac (bspec RS bspec RS mp) 1);
by (REPEAT (ares_tac prems 1));
val monoD = result();


(*** Proof of Knaster-Tarski Theorem ***)

val [prem] = goal Fixedpt.thy "f: Pow(D)->Pow(D) ==> f`D <= D";
by (rtac PowD 1);
by (rtac (prem RS apply_type) 1);
by (rtac Pow_top 1);
val subset_top = result();

val prems = goalw Fixedpt.thy [lfix_def]
    "f: Pow(D)->Pow(D) ==> lfp(f): Pow(D)";
by (rtac (CollectI RS Inter_lower RS PowI) 1);
by (rtac (domain_of_fun RS ssubst) 1);
by (REPEAT (resolve_tac (prems@[Pow_top, subset_top]) 1));
val lfix_type = result();

(*lfp is the least fixed point*)
val prems = goalw Fixedpt.thy [lfix_def]
    "[| f`a <= a;  f: Pow(D)->Pow(D);  a : Pow(D) |] ==> lfp(f) <= a";
by (rtac (domain_of_fun RS ssubst) 1);
by (REPEAT (resolve_tac (prems@[Inter_lower,CollectI]) 1));
val lfix_lowerbound = result();

val prems = goalw Fixedpt.thy [lfix_def]
    "[| !!u. [| f`u <= u;  u: Pow(D) |] ==> a<=u;  \
\       f: Pow(D)->Pow(D) |] ==> \
\    a <= lfp(f)";
by (rtac (domain_of_fun RS ssubst) 1);
by (REPEAT (etac CollectE 1
     ORELSE ares_tac ([subset_top,Pow_top,CollectI,Inter_greatest]@prems) 1));
val lfix_greatest = result();

val prems = goal Fixedpt.thy
    "[| f`a<=a;  a : Pow(D);  f: mono(Pow(D),Pow(D))  \
\    |] ==> f`lfp(f) <= a";
by (rtac subset_trans 1);
by (REPEAT (resolve_tac (prems@[lfix_type,lfix_lowerbound,
				mono_type,monoD]) 1));
val fix_lemma1 = result();

val prems = goal Fixedpt.thy
    "[| f: mono(Pow(D),Pow(D)) |] ==> f`lfp(f) <= lfp(f)";
by (REPEAT (ares_tac (prems@[lfix_type,lfix_greatest,fix_lemma1,
			     mono_type]) 1));
val fix_lemma2 = result();

val prems = goal Fixedpt.thy
    "[| f: mono(Pow(D),Pow(D)) |] ==> lfp(f) <= f`lfp(f)";
by (REPEAT (ares_tac (prems@[lfix_type,lfix_lowerbound,fix_lemma2,
			     mono_type,monoD,apply_type]) 1));
val fix_lemma3 = result();

val prems = goal Fixedpt.thy
    "[| f: mono(Pow(D),Pow(D)) |] ==> lfp(f) = f`lfp(f)";
by (REPEAT (resolve_tac (prems@[equalityI,fix_lemma2,fix_lemma3]) 1));
val Tarski_theorem = result();


(*** General induction rule for least fixed points ***)

val prems = goal Fixedpt.thy
    "a: Pow(D) ==> {x: a . P(x)} : Pow(D)";
by (EVERY1 [rtac PowI, rtac subsetI, rtac (PowD RS subsetD),
            resolve_tac prems, etac CollectD1]);
val Collect_Pow_type = result();

val prems = goal Fixedpt.thy
    "[| !!x. [| x : f ` Collect(lfp(f),P) |] ==> P(x);  \
\       f: mono(Pow(D),Pow(D)) \
\    |] ==> f ` Collect(lfp(f),P) <= Collect(lfp(f),P)";
by (rtac subsetI 1);
by (rtac CollectI 1);
by (rtac subsetD 1);
by (assume_tac 2);
by (rtac subset_trans 1);
by (rtac fix_lemma2 2);
by (REPEAT (ares_tac (prems@[monoD,Collect_subset,Collect_Pow_type,
			     mono_type,lfix_type]) 1));
val Collect_is_fixedpt = result();

val prems = goal Fixedpt.thy
    "[| a : lfp(f);   \
\       !!x. [| x : f ` Collect(lfp(f),P) |] ==> P(x);  \
\       f: mono(Pow(D),Pow(D)) \
\    |] ==> P(a)";
by (rtac (Collect_is_fixedpt RS lfix_lowerbound RS subsetD RS CollectD2) 1);
by (REPEAT (ares_tac (prems@[mono_type, lfix_type RS Collect_Pow_type]) 1));
val general_induction = result();


(*** For working with lambda abstractions ***)

(*monotonicity of a lambda abstraction*)
val prems = goal Fixedpt.thy
    "[| !!x.x:A ==> b(x): B;  \
\       !!w x. [| w: A;  x: A;  w<=x |] ==> b(w) <= b(x)  \
\    |] ==> (lam x:A.b(x)) : mono(A,B)";  
by (rtac monoI 1);
by (rtac (beta_conv RS ssubst) 2);
by (rtac (beta_conv RS ssubst) 3);
by (REPEAT (ares_tac (prems @ [lam_type]) 1));
val lam_mono_type = result();

val prems = goal Fixedpt.thy
    "[| !!x.x<=A ==> b(x)<=B;  \
\       !!w x. [| w<=A;  x<=A;  w<=x |] ==> b(w) <= b(x)  \
\    |] ==> (lam x:Pow(A).b(x)) : mono(Pow(A),Pow(B))";  
by (rtac lam_mono_type 1);
by (REPEAT (ares_tac (prems @ [PowI,PowD]) 1));
val lam_mono_Powtype = result();

val prems = goal Fixedpt.thy
    "[| (lam x:Pow(D). b(x)) : mono(Pow(D),Pow(D))  \
\    |] ==> lfp(lam x:Pow(D). b(x)) = b(lfp(lam x:Pow(D). b(x)))";
by (REPEAT (ares_tac (prems @ [Tarski_theorem RS trans, mono_type,
			       beta_conv, lfix_type]) 1));
val lam_Tarski_theorem = result();


val prems = goal Fixedpt.thy
    "[| a : lfp(lam x:Pow(D). b(x));   \
\       !!x. [| x : b(Collect(lfp(lam x:Pow(D). b(x)), P)) |] ==> P(x);  \
\       (lam x:Pow(D). b(x)) : mono(Pow(D),Pow(D)) \
\    |] ==> P(a)";
by (rtac general_induction 1);
by (resolve_tac prems 1);
by (resolve_tac prems 1);
(*performing beta-reduction in the assumption...*)
by (etac rev_mp 1);
by (rtac (beta_conv RS ssubst) 1);
by (REPEAT (ares_tac (prems @ [impI, 
		      mono_type RS lfix_type RS Collect_Pow_type]) 1));
val lam_general_induction = result();

(*experimental rule -- trying to avoid complications with definitions*)
val eqprem::prems = goal Fixedpt.thy
    "[| A = lfp(lam x:Pow(D). b(x));              \
\       a : A;   \
\       !!x. [| x : b(Collect(A, P)) |] ==> P(x);  \
\       (lam x:Pow(D). b(x)) : mono(Pow(D),Pow(D)) \
\    |] ==> P(a)";
by (rtac lam_general_induction 1);
by (rtac (eqprem RS subst) 1);
by (resolve_tac prems 1);
by (resolve_tac prems 1);
by (etac (eqprem RS ssubst) 1);
by (resolve_tac prems 1);
val lam_general_induction2 = result();

(*** The Schroeder-Bernstein Theorem ***)

goal Fixedpt.thy "(lam s: Pow(X). X - g``(Y - f``s)) : mono(Pow(X), Pow(X))";
by (REPEAT (ares_tac [lam_mono_type, subset_refl, Diff_mono, image_mono, 
		      subsetI RS PowI] 1 
     ORELSE etac DiffE 1));
val decompfun_mono = result();

goal Fixedpt.thy "lfp(lam s: Pow(X). X - g``(Y - f``s)) <= X";
by (rtac (decompfun_mono RS mono_type RS lfix_type RS PowD) 1);
val decomp_subset = result();

(*Banach's Decomposition Theorem -- see Davey & Priestly, page 106*)
val prems = goal Fixedpt.thy
    "[| f: X->Y;  g: Y->X |] ==>   \
\    EX XA XB YA YB. (XA Int XB = 0) & (XA Un XB = X) &    \
\                    (YA Int YB = 0) & (YA Un YB = Y) &    \
\                    f``XA=YA & g``YB=XB";
by (res_inst_tac [("x", "lfp(lam s: Pow(X). X - g``(Y - f``s))")] exI 1);
by (rtac exI 1);
by (res_inst_tac [("x", "f``?XA")] exI 1);
by (rtac exI 1);
by (REPEAT (resolve_tac (prems@[refl,decomp_subset,Diff_disjoint,
                  Diff_partition,fun_is_rel,image_subset,conjI]) 1));
by (rtac (trans RS sym) 1);
by (rtac (decompfun_mono RS lam_Tarski_theorem RS ssubst) 1);
by (rtac double_complement 1);
by (REPEAT (resolve_tac (prems@[refl,subset_refl,fun_is_rel,
				image_subset,Diff_subset]) 1));
val decomposition = result();

val prems = goal Fixedpt.thy
    "[| f: inj(X,Y);  g: inj(Y,X) |] ==> EX h. h: bij(X,Y)";
by (cut_facts_tac prems 1);
by (res_inst_tac [("f","f"), ("g","g")] (make_elim decomposition) 1);
by (REPEAT (ares_tac [inj_is_fun] 1
     ORELSE eresolve_tac [restrict_bij,exE,conjE] 1
     ORELSE hyp_subst_tac 1
     ORELSE resolve_tac [Un_upper1,Un_upper2,bij_disjoint_Un,
			 bij_converse_bij,exI] 1));
(* The instantiation of exI to "restrict(f,XA) Un converse(restrict(g,YB))"
   is forced by the context!! *)
val schroeder_bernstein = result();
