(*  Title: 	HOL/simpdata
    Author: 	Tobias Nipkow
    Copyright   1990  University of Cambridge

Invoking the simplifier
*)


fun prover s = prove_goal thy s (fn _=>[fast_tac HOL_cs 1]);

val auto_tac_ref = ref(resolve_tac [truth]);

(* Conversion into rewrite rules *)

fun basify thm =
    let val _$(_$t$_) = concl_of thm
    in  case fastype_of([],t) of
	   Type("fun",_) => basify(thm RS ap_thm)
         | _ => thm
    end;

val not_P_imp_P_iff_F = prover "~P --> (P = False)" RS mp;

fun mk_eq r = case concl_of r of
	_$(Const("op =",_)$_$_) => r
    |	_$(Const("not",_)$_) => r RS not_P_imp_P_iff_F
    |   _ => r RS eqTrueI;

fun atomize r = case concl_of r of
	_$(Const("op -->",_)$_$_) => atomize(r RS mp)
    |   _$(Const("op &",_)$_$_) => atomize(r RS conjunct1) @
				atomize(r RS conjunct2)
    |	_$(Const("All",_)$_) => atomize(r RS spec)
    |	_$(Const("True",_)) => []
    |	_$(Const("False",_)) => []
    |   _ => [r];

(* Simple formula rewrites *)

val refl_T = prover "(x=x) = True";
val T_eq_F = prover "(True=False) = False";
val T_imp_P = prover "(True --> P) = P";
val F_imp_P = prover "(False --> P) = True";
val P_and_T = prover "(P & True) = P";
val F_eq_T = prover "(False=True) = False";
val T_and_P = prover "(True & P) = P";
val not_T = prover "(~True) = False";
val not_P = prover "(~P) = (P = False)";


val if_True = prove_goal thy "(True => x | y) = x"
    (fn _=>[stac Cond_def 1,  fast_tac (HOL_cs addIs [select_equality]) 1]);

val if_False = prove_goal thy "(False => x | y) = y"
    (fn _=>[stac Cond_def 1,  fast_tac (HOL_cs addIs [select_equality]) 1]);

val if_P = prove_goal thy "P ==> (P => x | y) = x"
    (fn [prem] => [ stac (prem RS eqTrueI) 1, rtac if_True 1 ]);

val if_not_P = prove_goal thy "~P ==> (P => x | y) = y"
    (fn [prem] => [ stac (prem RS not_P_imp_P_iff_F) 1, rtac if_False 1 ]);

val expand_if = prove_goal thy "P(Q => x|y) = ((Q --> P(x)) & (~Q --> P(y)))"
    (fn _=> [ (res_inst_tac [("Q","Q")] (excluded_middle RS disjE) 1),
	 rtac (if_P RS ssubst) 2,
	 rtac(if_not_P RS ssubst) 1,
	 REPEAT(fast_tac HOL_cs 1) ]);


structure HOL_SimpData =
  struct
  val refl_thms = [refl]
  val trans_thms = [trans]
  val red1 = eq_mp
  val red2 = sym RS eq_mp
  fun mk_rew_rules r = map (basify o mk_eq) (atomize r)
  fun auto_tac i = !auto_tac_ref i
  val case_splits = [(expand_if,"Cond")]
  val norm_thms = [(I_def RS ap_thm RS sym,I_def RS ap_thm)]
  val subst_thms = [subst]
  fun dest_red(_ $ (red$lhs$rhs)) = (red,lhs,rhs)
    | dest_red _ = error("dest_red")
  end;

structure HOL_Simp = SimpFun(HOL_SimpData);
open HOL_Simp;

val simp_thms = [if_True,if_False] @ map prover
 [ "(x=x) = True",
   "(~True) = False", "(~False) = True",
   "(True=P) = P", "(P=True) = P",
   "(True --> P) = P", "(False --> P) = True", 
   "(P --> True) = True", "(P --> P) = True",
   "(P & True) = P", "(True & P) = P", 
   "(P & False) = False", "(False & P) = False", "(P & P) = P",
   "(P | True) = True", "(True | P) = True", 
   "(P | False) = P", "(False | P) = P", "(P | P) = P",
   "(!x.P ) = P",
   "(P|Q --> R) = ((P-->R)&(Q-->R))" ];

val imp_cong = impI RSN
    (2, prove_goal thy "(P=P')--> (P'--> (Q=Q'))--> ((P-->Q) = (P'-->Q'))"
	(fn _=> [fast_tac HOL_cs 1]) RS mp RS mp);

val congs = imp_cong :: abs ::
	mk_congs thy ["op =","not","op &","op |","op o","Cond"] @
	map (apl(abs,op RS)) (mk_congs thy ["Eps","All","Ex","Ex1"]);

val HOL_ss = empty_ss addcongs congs addrews simp_thms;


(*simplify all subgoals WRT extra rewrite rules*)
fun all_simp_tac ss rews = ALLGOALS (ASM_SIMP_TAC (ss addrews rews));
