(*  Title: 	HOL/nat
    Author: 	Tobias Nipkow, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Definition of types ind and nat.

Type nat is defined as a class Nat over type ind.
*)

structure Nat =
struct

val sext = 
  Sext{mixfix=[Delimfix("0", "nat", "0"),
	       Infixl("<","[nat,nat] => bool",50)],
       print_translation=[], 
       parse_translation=[]}

val thy = extend_theory Trancl.thy "Nat"
([], [],
 [
  (["ind"], ([],"term")),
  (["nat"], ([],"term"))
 ],
 [
  (["Zero_Rep"],	"ind"),
  (["Suc_Rep"],		"ind => ind"),
  (["Nat"],		"ind class"),
  (["Rep_Nat"],		"nat => ind"),
  (["Abs_Nat"],		"ind => nat"),
  (["Suc"],		"nat => nat"),
  (["nat_case"],    	"[nat, 'a, nat=>'a] =>'a"),
  (["pred_nat"],	"(nat*nat) class"),
  (["nat_rec"],		"[nat, 'a, [nat, 'a]=>'a] => 'a")
 ],
 Some(sext))
 [  (*the axiom of infinity in 2 parts*)
  ("One_One_Suc_Rep",  		"One_One(Suc_Rep)"),
  ("Suc_Rep_not_Zero_Rep",	"~(Suc_Rep(x) = Zero_Rep)"),
  ("Nat_def",		"Nat == lfp(%X. {Zero_Rep} Un (Suc_Rep``X))"),
    (*faking a type definition...*)
  ("Rep_Nat", 		"Rep_Nat(n): Nat"),
  ("Rep_Nat_inverse", 	"Abs_Nat(Rep_Nat(n)) = n"),
  ("Abs_Nat_inverse", 	"i: Nat ==> Rep_Nat(Abs_Nat(i)) = i"),
    (*defining the abstract constants*)
  ("Zero_def",  	"0 == Abs_Nat(Zero_Rep)"),
  ("Suc_def",  		"Suc == (%n. Abs_Nat(Suc_Rep(Rep_Nat(n))))"),
     (*nat operations and recursion*)
  ("nat_case_def",	"nat_case == (%n a f. @z.  (n=0 --> z=a)  \
\                                          & (!x. n=Suc(x) --> z=f(x)))"),
  ("pred_nat_def", 	"pred_nat == {p. ? n. p = <n, Suc(n)>}" ),

  ("less_def", "m<n == <m,n>:trancl(pred_nat)"),

  ("nat_rec_def",	
   "nat_rec(n,c,d) == wfrec(trancl(pred_nat), \
\                        %rec l. nat_case(l, c, %m. d(m,rec(m))), \
\                        n)" )
 ]
end;

local val ax = get_axiom Nat.thy
in
val One_One_Suc_Rep= ax"One_One_Suc_Rep";
val Suc_Rep_not_Zero_Rep= ax"Suc_Rep_not_Zero_Rep";
val Nat_def  = ax"Nat_def";
val Rep_Nat= ax"Rep_Nat";
val Rep_Nat_inverse = ax"Rep_Nat_inverse";
val Abs_Nat_inverse = ax"Abs_Nat_inverse";
val Zero_def    = ax"Zero_def";
val Suc_def     = ax"Suc_def";
val nat_case_def = ax"nat_case_def";
val pred_nat_def = ax"pred_nat_def";
val less_def 	= ax "less_def";
val nat_rec_def = ax"nat_rec_def";
end;


goal Nat.thy "mono(%X. {Zero_Rep} Un (Suc_Rep``X))";
by (REPEAT (ares_tac [monoI, subset_refl, image_mono, Un_mono] 1));
val Nat_fun_mono = result();

val Nat_unfold = Nat_fun_mono RS (Nat_def RS Tarski_def_theorem);

(* Zero is a natural number -- this also justifies the type definition*)
goal Nat.thy "Zero_Rep: Nat";
by (rtac (Nat_unfold RS ssubst) 1);
br (singletonI RS UnI1) 1;
val Zero_RepI = result();

val prems = goal Nat.thy "i: Nat ==> Suc_Rep(i) : Nat";
by (rtac (Nat_unfold RS ssubst) 1);
br (imageI RS UnI2) 1;
brs prems 1;
val Suc_RepI = result();

(*** Induction ***)

val major::prems = goalw Nat.thy [Nat_def]
    "[| i: Nat;  P(Zero_Rep);   \
\       !!j. [| j: Nat; P(j) |] ==> P(Suc_Rep(j)) |]  ==> P(i)";
br (Nat_fun_mono RS (major RS general_induction)) 1;
by (fast_tac (class_cs addIs prems) 1);
val Nat_induct = result();

val prems = goalw Nat.thy [Zero_def,Suc_def]
    "[| P(0);   \
\       !!k. [| P(k) |] ==> P(Suc(k)) |]  ==> P(n)";
by (rtac (Rep_Nat_inverse RS subst) 1);   (*types force good instantiation*)
br (Rep_Nat RS Nat_induct) 1;
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [Abs_Nat_inverse RS subst] 1));
val nat_induct = result();

(*Toby's preferred style of induction rule*)
val prems = goal Nat.thy "[| P(0);  (!n. P(n) --> P(Suc(n))) |] ==> (!n.P(n))";
by (cut_facts_tac prems 1);
br (nat_induct RS allI) 1;
ba 1;
by (fast_tac class_cs 1);
val nat_ind = result();

(*Perform induction on n. *)
fun nat_ind_tac a i = 
    EVERY [res_inst_tac [("n",a)] nat_induct i,
	   rename_last_tac a ["1"] (i+1)];

(*A special form of induction for reasoning about m<n and m-n*)
val prems = goal Nat.thy
    "[| !!x. P(x,0);  \
\       !!y. P(0,Suc(y));  \
\       !!x y. [| P(x,y) |] ==> P(Suc(x),Suc(y))  \
\    |] ==> P(m,n)";
by (res_inst_tac [("x","m")] spec 1);
by (nat_ind_tac "n" 1);
br allI 2;
by (nat_ind_tac "x" 2);
by (REPEAT (ares_tac (prems@[allI]) 1 ORELSE etac spec 1));
val diff_induct = result();

(*** Isomorphisms: Abs_Nat and Rep_Nat ***)

(*We can't take these properties as axioms, or take Abs_Nat==Inv(Rep_Nat),
  since we assume the isomorphism equations will one day be given by Isabelle*)

goal Nat.thy "One_One(Rep_Nat)";
br One_One_inverseI 1;
br Rep_Nat_inverse 1;
val One_One_Rep_Nat = result();

goal Nat.thy "One_One_on(Abs_Nat,Nat)";
br One_One_on_inverseI 1;
be Abs_Nat_inverse 1;
val One_One_on_Abs_Nat = result();

(*** Distinctness of constructors ***)

goalw Nat.thy [Zero_def,Suc_def] "~ (Suc(m) = 0)";
br (One_One_on_Abs_Nat RS One_One_on_contraD) 1;
br Suc_Rep_not_Zero_Rep 1;
by (REPEAT (resolve_tac [Rep_Nat, Suc_RepI, Zero_RepI] 1));
val Suc_not_Zero = result();

val Zero_not_Suc = standard (Suc_not_Zero RS neg_sym);

val Suc_neq_Zero = standard (Suc_not_Zero RS notE);
val Zero_neq_Suc = sym RS Suc_neq_Zero;

(** Injectiveness of Suc **)

goalw Nat.thy [Suc_def] "One_One(Suc)";
br One_OneI 1;
bd (One_One_on_Abs_Nat RS One_One_onD) 1;
by (REPEAT (resolve_tac [Rep_Nat, Suc_RepI] 1));
bd (One_One_Suc_Rep RS One_OneD) 1;
be (One_One_Rep_Nat RS One_OneD) 1;
val One_One_Suc = result();

val Suc_inject = One_One_Suc RS One_OneD;;

goal Nat.thy "(Suc(m)=Suc(n)) = (m=n)";
by (EVERY1 [rtac iffI, etac Suc_inject, etac ap_term]); 
val Suc_Suc_eq = result();

goal Nat.thy "~(n=Suc(n))";
by (nat_ind_tac "n" 1);
by (SIMP_TAC (HOL_ss addrews [Zero_not_Suc]) 1);
by (ASM_SIMP_TAC (HOL_ss addrews [Suc_Suc_eq]) 1);
val n_not_Suc_n = result();

(*** nat_case -- the selection operator for nat ***)

goalw Nat.thy [nat_case_def] "nat_case(0, a, f) = a";
by (fast_tac (class_cs addIs [select_equality] addEs [Zero_neq_Suc]) 1);
val nat_case_0_conv = result();

goalw Nat.thy [nat_case_def] "nat_case(Suc(k), a, f) = f(k)";
by (fast_tac (class_cs addIs [select_equality] 
	               addEs [make_elim Suc_inject, Suc_neq_Zero]) 1);
val nat_case_Suc_conv = result();

(** Introduction rules for 'pred_nat' **)

goalw Nat.thy [pred_nat_def] "<n, Suc(n)> : pred_nat";
by (fast_tac class_cs 1);
val pred_natI = result();

val major::prems = goalw Nat.thy [pred_nat_def]
    "[| p : pred_nat;  !!x n. [| p = <n, Suc(n)> |] ==> R \
\    |] ==> R";
br (major RS CollectE) 1;
by (REPEAT (eresolve_tac ([asm_rl,exE]@prems) 1));
val pred_natE = result();

goalw Nat.thy [wf_def] "wf(pred_nat)";
by (strip_tac 1);
by (nat_ind_tac "x" 1);
by (fast_tac (HOL_cs addSEs [mp, pred_natE, Pair_inject, 
			     make_elim Suc_inject]) 2);
by (fast_tac (HOL_cs addSEs [mp, pred_natE, Pair_inject, Zero_neq_Suc]) 1);
val wf_pred_nat = result();


(*** nat_rec -- by wf recursion on pred_nat ***)

val wfrec_nat_unfold = 
    trans_trancl RS (wf_pred_nat RS wf_trancl RS wfrec_def_conv);

val nat_rec_unfold = standard (nat_rec_def RS wfrec_nat_unfold);

(** conversion rules **)

goal Nat.thy "nat_rec(0,c,h) = c";
by (rtac (nat_rec_unfold RS trans) 1);
br nat_case_0_conv 1;
val nat_rec_0_conv = result();

val prems = goal Nat.thy "nat_rec(Suc(n), c, h) = h(n, nat_rec(n,c,h))";
by (rtac (nat_rec_unfold RS trans) 1);
by (rtac (nat_case_Suc_conv RS trans) 1);
by (SIMP_TAC (HOL_ss 
      addcongs (mk_typed_congs Nat.thy [("h", "[nat, ?'a]=> ?'a")])
      addrews [trancl_I1,pred_natI]) 1);
val nat_rec_Suc_conv = result();

(*The following rules avoid explosions in proofs.  NOTE USE OF == *)
val rew::prems = goal Nat.thy
    "[| !!n. f(n) == nat_rec(n,c,h) |] ==> f(0) = c";
bw rew;
br nat_rec_0_conv 1;
val nat_rec_def_0_conv = result();

val rew::prems = goal Nat.thy
    "[| !!n. f(n) == nat_rec(n,c,h) |] ==> f(Suc(n)) = h(n,f(n))";
bw rew;
br nat_rec_Suc_conv 1;
val nat_rec_def_Suc_conv = result();


(*** Basic properties of "less than" ***)

(** Introduction properties **)

val prems = goalw Nat.thy [less_def] "[| i<j;  j<k |] ==> i<k";
br (trans_trancl RS transD) 1;
brs prems 1;
brs prems 1;
val less_trans = result();

goalw Nat.thy [less_def] "n < Suc(n)";
br (pred_natI RS trancl_I1) 1;
val lessI = result();

(* i<j ==> i<Suc(j) *)
val less_SucI = lessI RSN (2, less_trans);

goal Nat.thy "0 < Suc(n)";
by (nat_ind_tac "n" 1);
br lessI 1;
be less_trans 1;
br lessI 1;
val zero_less_Suc = result();

(** Elimination properties **)

goalw Nat.thy [less_def] "n<m --> ~m<n";
br (wf_pred_nat RS wf_trancl RS wf_anti_sym RS notI RS impI) 1;
ba 1;
ba 1;
val less_not_sym = result();

(* [| n<m; m<n |] ==> R *)
val less_anti_sym = standard (less_not_sym RS mp RS notE);


goalw Nat.thy [less_def] "~ (n<n)";
br notI 1;
be (wf_pred_nat RS wf_trancl RS wf_anti_refl) 1;
val less_not_refl = result();

(* n<n ==> R *)
val less_anti_refl = standard (less_not_refl RS notE);


val major::prems = goalw Nat.thy [less_def]
    "[| i<k;  k=Suc(i) ==> P;  !!j. [| i<j;  k=Suc(j) |] ==> P \
\    |] ==> P";
br (major RS tranclE) 1;
by (fast_tac (HOL_cs addSEs (prems@[pred_natE, Pair_inject])) 1);
by (fast_tac (HOL_cs addSEs (prems@[pred_natE, Pair_inject])) 1);
val lessE = result();

goal Nat.thy "~ (n<0)";
br notI 1;
be lessE 1;
by (etac Zero_neq_Suc 1);
by (etac Zero_neq_Suc 1);
val not_less0 = result();

(* ~ n<0 *)
val less_zeroE = standard (not_less0 RS notE);

val [major,less,eq] = goal Nat.thy
    "[| m < Suc(n);  m<n ==> P;  m=n ==> P |] ==> P";
br (major RS lessE) 1;
br eq 1;
by (fast_tac (HOL_cs addSEs [make_elim Suc_inject]) 1);
br less 1;
by (fast_tac (HOL_cs addSEs [make_elim Suc_inject]) 1);
val less_SucE = result();

goal Nat.thy "(m < Suc(n)) = (m < n | m = n)";
by (fast_tac (HOL_cs addSIs [lessI]
		     addEs  [less_trans, less_SucE]) 1);
val less_Suc_eq = result();


(** Inductive properties **)

val prems = goal Nat.thy "Suc(m) < n ==> m<n";
by (subgoal_tac "Suc(m) < n --> m < n" 1);
by (fast_tac (HOL_cs addIs prems) 1);
by (nat_ind_tac "n" 1);
br impI 1;
be less_zeroE 1;
by (fast_tac (HOL_cs addSIs [lessI RS less_SucI]
	 	     addSEs [make_elim Suc_inject]
		     addEs  [less_trans, lessE]) 1);
val Suc_lessD = result();

val [major] = goal Nat.thy "Suc(m) < Suc(n) ==> m<n";
br (major RS lessE) 1;
by (REPEAT (rtac lessI 1
     ORELSE eresolve_tac [make_elim Suc_inject, ssubst, Suc_lessD] 1));
val Suc_less_SucD = result();

val prems = goal Nat.thy "m<n ==> Suc(m) < Suc(n)";
by (subgoal_tac "m<n --> Suc(m) < Suc(n)" 1);
by (fast_tac (HOL_cs addIs prems) 1);
by (nat_ind_tac "n" 1);
br impI 1;
be less_zeroE 1;
by (fast_tac (HOL_cs addSIs [lessI]
	 	     addSEs [make_elim Suc_inject]
		     addEs  [less_trans, lessE]) 1);
val Suc_mono = result();

goal Nat.thy "(Suc(m) < Suc(n)) = (m<n)";
by (EVERY1 [rtac iffI, etac Suc_less_SucD, etac Suc_mono]);
val Suc_less_eq = result();

val [major] = goal Nat.thy "Suc(n)<n ==> P";
br (major RS Suc_lessD RS less_anti_refl) 1;
val not_Suc_n_less_n = result();

goal Nat.thy "m<n | m=n | n<m";
by (nat_ind_tac "m" 1);
by (nat_ind_tac "n" 1);
fr (refl RS disjI1 RS disjI2);
fr (zero_less_Suc RS disjI1);
by (fast_tac (HOL_cs addIs [lessI, Suc_mono, less_SucI] addEs [lessE]) 1);
val less_trichotomy = result();

(*Can be used with less_Suc_eq to get n=m | n<m *)
goal Nat.thy "(~ m < n) = (n < Suc(m))";
by (res_inst_tac [("m","m"),("n","n")] diff_induct 1);
by (SIMP_TAC (HOL_ss addrews [not_less0,zero_less_Suc]) 1);
by (SIMP_TAC (HOL_ss addrews [not_less0,zero_less_Suc,Suc_less_eq]) 1);
by (ASM_SIMP_TAC (HOL_ss addrews [Suc_less_eq]) 1);
val not_less_eq = result();

val prems = goalw Nat.thy [less_def]
    "[| !!n. [| ! m. m<n --> P(m) |] ==> P(n) |]  ==>  P(n)";
by (wf_ind_tac "n" [wf_pred_nat RS wf_trancl] 1);
bes prems 1;
val less_induct = result();

