(*  Title: 	HOL/class
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Class theory for higher-order logic.  A class is simply a predicate.
*)

structure Class =
struct
val const_decs = 
 [
  (["Collect"],		"['a=>bool] => 'a class"),
  (["Ball"], 		"['a class, 'a=>bool]=>bool"),  (*bounded quantifiers*)
  (["Bex"], 		"['a class, 'a=>bool]=>bool"),
  (["Compl"], 		"('a class) => 'a class"),
  (["Union", "Inter"],	"(('a class)class) => 'a class"),
  (["UNION", "INTER"],	"['a class, 'a => 'b class] => 'b class"),
  (["mono","anti_mono"], "['a class => 'b class] => bool"),
  (["One_One","Onto"], 	"('a => 'b) => bool"),
  (["One_One_on"], 	"['a => 'b, 'a class] => bool"),
  (["Onto_Subset"], 	"['a => 'b, 'b class] =>  bool"),
  (["range"], 		"('a => 'b) => 'b class")
 ];

(** Collection  { x. P[x] } = Collect(%(x)P[x]) **)
fun coll_tr[Free(x,T), Px] =
    Const("Collect", dummyT) $ absfree(x,T,Px);

fun coll_tr'[Abs(x,T,P)] =
    let val (x',Px) = variant_abs(x,T,P)
    in Const("@Coll",dummyT) $ Free(x',T) $ Px end;

(** Bounded quantifiers have the form %x:A.P(x) **)
fun qnt_tr q [Free(x,T), A, P] = Const(q,dummyT) $ A $ absfree(x,T,P);

fun qnt_tr' q [A, Abs(x,T,P)] =
    let val (x',P') = variant_abs(x,T,P)
    in Const(q,dummyT) $ Free(x',T) $ A $ P' end;

val mixfix =
 [Delimfix("(1{_./ _})", "[ID,bool]=>'a class", "@Coll"),   (*collection*)
  Delimfix("{_}", "'a=> 'a class", "singleton"),
  Infixl("``",	"['a=>'b,'a class] => ('b class)",	90),
  Infixl("Int", "['a class,'a class]=> 'a class", 	70),
  Infixl("Un",	"['a class,'a class]=> 'a class", 	65),
  Infixl(":",	"['a,'a class]=>bool", 			50), (*membership*)
  Infixl("<=",	"['a class,'a class]=>bool", 		50),
      (*Big intersection/union*)
  Mixfix("(INT _:_./ _)", "[ID,'a class,'b class]=> 'b class", 
	 "@INTER", [], 10),
  Mixfix("(UN _:_./ _)", "[ID,'a class,'b class]=> 'b class", 
	 "@UNION",  [], 10),
      (*Bounded quantifiers*)
  Mixfix("(ALL _:_./ _)", "[ID,'a class,bool]=>bool", "@Ball", [], 10),
  Mixfix("(EX _:_./ _)",  "[ID,'a class,bool]=>bool", "@Bex",  [], 10)
];

val sext = Sext{
  mixfix=mixfix,
  parse_translation=
    [("@Coll",  coll_tr),
     ("@INTER", qnt_tr "INTER"),
     ("@UNION", qnt_tr "UNION"),
     ("@Ball",  qnt_tr "Ball"),
     ("@Bex",   qnt_tr "Bex") ],
  print_translation=
    [("Collect",        coll_tr'),
     ("INTER",  qnt_tr' "@INTER"),
     ("UNION",  qnt_tr' "@UNION"),
     ("Ball",   qnt_tr' "@Ball"),
     ("Bex",    qnt_tr' "@Bex") ]};

val thy = extend_theory HOL_Rule.thy "Class"
 ([], [], 
  [(["class"],(["term"],"term"))],
  const_decs, Some(sext))
[
  ("CollectI",		"[| P(a) |] ==> a : {x.P(x)}" ),
  ("CollectD",		"[| a : {x.P(x)} |] ==> P(a)" ),
  ("class_ext", 	"[| !!x. (x:A) = (x:B) |] ==> A = B" ),

  ("Ball_def",		"Ball(A,P) == ! x. x:A --> P(x)" ),
  ("Bex_def",		"Bex(A,P) == ? x. x:A & P(x)" ),
  ("subset_def", 	"A <= B  == ALL x:A. x:B" ),
  ("Un_def",		"A Un B   == {x.x:A | x:B}" ),
  ("Int_def",		"A Int B  == {x.x:A & x:B}" ),
  ("Compl_def",		"Compl(A) == {x. ~x:A}" ),
  ("Inter_def",		"Inter(S) == {x. ALL A:S. x:A}" ),
  ("Union_def",		"Union(S) == {x. EX A:S. x:A}" ),
  ("INTER_def",		"INTER(A,B) == {y. ALL x:A. y: B(x)}" ),
  ("UNION_def",		"UNION(A,B) == {y. EX x:A. y: B(x)}" ),
  ("mono_def",		"mono(f)  == (!A B. A <= B --> f(A) <= f(B))" ),
  ("anti_mono_def", 	"anti_mono(f) == (!A B. A <= B --> f(B) <= f(A))" ),
  ("image_def",  	"f``A == {y. EX x:A. y=f(x)}" ),
  ("singleton_def",   	"{a} == {x.x=a}" ),
  ("range_def",		"range(f) == {y. ? x. y=f(x)}" ),

  ("One_One_def",	"One_One(f) == ! x y. f(x)=f(y) --> x=y" ),

  ("One_One_on_def",	
   "One_One_on(f,A) == !x y. x:A --> y:A --> f(x)=f(y) --> x=y" ),

  ("Onto_def",		"Onto(f) == ! y. ? x. y=f(x)" ),
  ("Onto_Subset_def", 	"Onto_Subset(f,A) == ! y. y:A = (? x.y=f(x))" )
 ];

end;

local val ax = get_axiom Class.thy
in
val CollectI    = ax"CollectI";
val CollectD    = ax"CollectD";
val class_ext   = ax"class_ext";
val Ball_def    = ax"Ball_def";
val Bex_def     = ax"Bex_def";
val subset_def  = ax"subset_def";
val Un_def      = ax"Un_def";
val Int_def     = ax"Int_def";
val Compl_def   = ax"Compl_def";
val Inter_def   = ax"Inter_def";
val Union_def   = ax"Union_def";
val INTER_def   = ax"INTER_def";
val UNION_def   = ax"UNION_def";
val mono_def    = ax"mono_def";
val anti_mono_def = ax"anti_mono_def";
val image_def   = ax"image_def";
val singleton_def    = ax"singleton_def";
val range_def   = ax"range_def";
val One_One_def = ax"One_One_def";
val One_One_on_def      = ax"One_One_on_def";
val Onto_def    = ax"Onto_def";
val Onto_Subset_def     = ax"Onto_Subset_def";
end;

val [prem] = goal Class.thy
    "[| !!x. P(x)=Q(x) |] ==> {x. P(x)} = {x. Q(x)}";
fr (iffI RS class_ext);
by (ALLGOALS (rtac CollectI));
by (ALLGOALS (dtac CollectD));
by (REPEAT (resolve_tac [CollectI, prem RS subst, prem RS ssubst] 1
         THEN assume_tac 1));
val Collect_cong = result();

val CollectE = make_elim CollectD;

(*** Bounded quantifiers ***)

val prems = goalw Class.thy [Ball_def]
    "[| !!x. x:A ==> P(x) |] ==> ALL x:A. P(x)";
by (REPEAT (ares_tac (prems @ [allI,impI]) 1));
val ballI = result();

val [major,minor] = goalw Class.thy [Ball_def]
    "[| ALL x:A. P(x);  x:A |] ==> P(x)";
br (minor RS (major RS spec RS mp)) 1;
val bspec = result();

val major::prems = goalw Class.thy [Ball_def]
    "[| ALL x:A. P(x);  P(x) ==> Q;  ~ x:A ==> Q |] ==> Q";
br (major RS spec RS impCE) 1;
by (REPEAT (eresolve_tac prems 1));
val ballE = result();

(*Takes assumptions ALL x:A.P(x) and a:A; creates assumption P(a)*)
fun ball_tac i = etac ballE i THEN contr_tac (i+1);

val prems = goalw Class.thy [Bex_def]
    "[| P(x);  x:A |] ==> EX x:A. P(x)";
by (REPEAT (ares_tac (prems @ [exI,conjI]) 1));
val bexI = result();

val bexCI = prove_goal Class.thy 
   "[| ALL x:A. ~P(x) ==> P(a);  a:A |] ==> EX x:A.P(x)"
 (fn prems=>
  [ (rtac classical 1),
    (REPEAT (ares_tac (prems@[bexI,ballI,notI,notE]) 1))  ]);

val major::prems = goalw Class.thy [Bex_def]
    "[| EX x:A. P(x);  !!x. [| x:A; P(x) |] ==> Q  |] ==> Q";
br (major RS exE) 1;
by (REPEAT (eresolve_tac (prems @ [asm_rl,conjE]) 1));
val bexE = result();

(*Trival rewrite rule;   (ALL x:A.P)=P holds only if A is nonempty!*)
val prems = goal Class.thy
    "(ALL x:A. True) = True";
by (REPEAT (ares_tac [TrueI,ballI,iffI] 1));
val ball_rew = result();

(** Congruence rules **)

val prems = goal Class.thy
    "[| A=A';  !!x. x:A' ==> P(x) = P'(x) |] ==> \
\    (ALL x:A. P(x)) = (ALL x:A'. P'(x))";
brs (prems RL [ssubst]) 1;
by (REPEAT (ares_tac [ballI,iffI] 1
     ORELSE eresolve_tac ([make_elim bspec, mp] @ (prems RL [iffE])) 1));
val ball_cong = result();

val iffD1 = sym RS eq_mp
and iffD2 = eq_mp;

val prems = goal Class.thy
    "[| A=A';  !!x. x:A' ==> P(x) = P'(x) |] ==> \
\    (EX x:A. P(x)) = (EX x:A'. P'(x))";
brs (prems RL [ssubst]) 1;
by (REPEAT (eresolve_tac [bexE] 1
     ORELSE ares_tac ([bexI,iffI] @ (prems RL [iffD1,iffD2])) 1));
val bex_cong = result();

(*** Rules for subsets ***)

val prems = goalw Class.thy [subset_def] "(!!x.x:A ==> x:B) ==> A <= B";
by (REPEAT (ares_tac (prems @ [ballI]) 1));
val subsetI = result();

(*Rule in Modus Ponens style*)
val major::prems = goalw Class.thy [subset_def] "[| A <= B;  c:A |] ==> c:B";
br (major RS bspec) 1;
brs prems 1;
val subsetD = result();

(*Classical elimination rule*)
val major::prems = goalw Class.thy [subset_def] 
    "[| A <= B;  ~(c:A) ==> P;  c:B ==> P |] ==> P";
br (major RS ballE) 1;
by (REPEAT (eresolve_tac prems 1));
val subsetCE = result();

(*Takes assumptions A<=B; c:A and creates the assumption c:B *)
fun set_mp_tac i = etac subsetCE i  THEN  mp_tac i;

val subset_refl = prove_goal Class.thy "A <= A"
 (fn _=> [ (REPEAT (ares_tac [subsetI] 1)) ]);

val prems = goal Class.thy "[| A<=B;  B<=C |] ==> A<=C";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [subsetI] 1 ORELSE set_mp_tac 1));
val subset_trans = result();


(*** Rules for equality ***)

(*Anti-symmetry of the subset relation*)
val prems = goal Class.thy "[| A <= B;  B <= A |] ==> A = B";
br (iffI RS class_ext) 1;
by (REPEAT (ares_tac (prems RL [subsetD]) 1));
val subset_antisym = result();
val equalityI = subset_antisym;

(* Equality rules from ZF set theory -- are they appropriate here? *)
val prems = goal Class.thy "A = B ==> A<=B";
brs (prems RL [subst]) 1;
br subset_refl 1;
val equalityD1 = result();

val prems = goal Class.thy "A = B ==> B<=A";
brs (prems RL [subst]) 1;
br subset_refl 1;
val equalityD2 = result();

val prems = goal Class.thy
    "[| A = B;  [| A<=B; B<=A |] ==> P |]  ==>  P";
by (resolve_tac prems 1);
by (REPEAT (resolve_tac (prems RL [equalityD1,equalityD2]) 1));
val equalityE = result();

(*Lemma for creating induction formulae -- for "pattern matching" on p
  To make the induction hypotheses usable, apply "spec" or "bspec" to
  put universal quantifiers over the free variables in p. *)
val prems = goal Class.thy 
    "[| p:A;  !!z. z:A ==> p=z --> R |] ==> R";
by (rtac mp 1);
by (REPEAT (resolve_tac (refl::prems) 1));
val setup_induction = result();

(*** Singleton sets ***)

goalw Class.thy [singleton_def] "a : {a}";
br CollectI 1;
br refl 1;
val singletonI = result();

val [major] = goalw Class.thy [singleton_def] "b : {a} ==> b=a"; 
by (rtac (major RS CollectD) 1);
val singletonD = result();

val singletonE = make_elim singletonD;

(*** Image of a set under a function ***)

val prems = goalw Class.thy [image_def] "[| x:A |] ==> f(x) : f``A";
by (REPEAT (resolve_tac (prems @ [CollectI,bexI,refl]) 1));
val imageI = result();

val major::prems = goalw Class.thy [image_def]
    "[| b : f``A;  !!x.[| b=f(x);  x:A |] ==> P |] ==> P"; 
by (rtac (major RS CollectD RS bexE) 1);
by (REPEAT (ares_tac prems 1));
val imageE = result();

(*** Range of a function ***)

goalw Class.thy [range_def] "f(x) : range(f)";
by (EVERY1 [rtac CollectI, rtac exI, rtac refl]);
val rangeI = result();

val [major,minor] = goalw Class.thy [range_def]
    "[| b : range(f);  !!x.[| b=f(x) |] ==> P |] ==> P"; 
by (rtac (major RS CollectD RS exE) 1);
by (etac minor 1);
val rangeE = result();

(*** Rules for Unions ***)

(*The order of the premises presupposes that C is rigid; A may be flexible*)
val prems = goalw Class.thy [Union_def]
    "[| X:C;  A:X |] ==> A : Union(C)";
by (REPEAT (resolve_tac (prems @ [bexI,CollectI]) 1));
val UnionI = result();

val major::prems = goalw Class.thy [Union_def]
    "[| A : Union(C);  !!X.[| A:X;  X:C |] ==> R |] ==> R";
by (rtac (major RS CollectD RS bexE) 1);
by (REPEAT (ares_tac prems 1));
val UnionE = result();

(*** Rules for Inter ***)

val prems = goalw Class.thy [Inter_def]
    "[| !!X. X:C ==> A:X |] ==> A : Inter(C)";
by (REPEAT (ares_tac ([CollectI,ballI] @ prems) 1));
val InterI = result();

(*A "destruct" rule -- every X in C contains A as an element, but
  A:X can hold when X:C does not!  This rule is analogous to "spec". *)
val major::prems = goalw Class.thy [Inter_def]
    "[| A : Inter(C);  X:C |] ==> A:X";
by (rtac (major RS CollectD RS bspec) 1);
brs prems 1;
val InterD = result();

(*"Classical" elimination rule -- does not require proving X:C *)
val major::prems = goalw Class.thy [Inter_def]
    "[| A : Inter(C);  A:X ==> R;  ~ X:C ==> R |] ==> R";
by (rtac (major RS CollectD RS ballE) 1);
by (REPEAT (eresolve_tac prems 1));
val InterE = result();

(*** Unions of families ***)

(*The order of the premises presupposes that A is rigid; b may be flexible*)
val prems = goalw Class.thy [UNION_def]
    "[| a:A;  b: B(a) |] ==> b: (UN x:A. B(x))";
by (REPEAT (resolve_tac (prems @ [bexI,CollectI]) 1));
val UN_I = result();

val major::prems = goalw Class.thy [UNION_def]
    "[| b : (UN x:A. B(x));  !!x.[| x:A;  b: B(x) |] ==> R |] ==> R";
by (rtac (major RS CollectD RS bexE) 1);
by (REPEAT (ares_tac prems 1));
val UN_E = result();


(*** Intersections of families -- INTER x:A. B(x) is Inter(B)``A ) *)

val prems = goalw Class.thy [INTER_def]
    "(!!x. x:A ==> b: B(x)) ==> b : (INT x:A. B(x))";
by (REPEAT (ares_tac ([CollectI,ballI] @ prems) 1));
val INT_I = result();

val major::prems = goalw Class.thy [INTER_def]
    "[| b : (INT x:A. B(x));  a:A |] ==> b: B(a)";
by (rtac (major RS CollectD RS bspec) 1);
brs prems 1;
val INT_D = result();

(*"Classical" elimination rule -- does not require proving X:C *)
val major::prems = goalw Class.thy [INTER_def]
    "[| b : (INT x:A. B(x));  b: B(a) ==> R;  ~ a:A ==> R |] ==> R";
by (rtac (major RS CollectD RS ballE) 1);
by (REPEAT (eresolve_tac prems 1));
val INT_E = result();

(*** Rules for binary union -- Un ***)

val prems = goalw Class.thy [Un_def] "c:A ==> c : A Un B";
by (REPEAT (resolve_tac (prems @ [CollectI,disjI1]) 1));
val UnI1 = result();

val prems = goalw Class.thy [Un_def] "c:B ==> c : A Un B";
by (REPEAT (resolve_tac (prems @ [CollectI,disjI2]) 1));
val UnI2 = result();

(*Classical introduction rule: no commitment to A vs B*)
val UnCI = prove_goal Class.thy "(~c:B ==> c:A) ==> c : A Un B"
 (fn prems=>
  [ (rtac classical 1),
    (REPEAT (ares_tac (prems@[UnI1,notI]) 1)),
    (REPEAT (ares_tac (prems@[UnI2,notE]) 1)) ]);

val major::prems = goalw Class.thy [Un_def]
    "[| c : A Un B;  c:A ==> P;  c:B ==> P |] ==> P";
by (rtac (major RS CollectD RS disjE) 1);
by (REPEAT (eresolve_tac prems 1));
val UnE = result();


(*** Rules for small intersection -- Int ***)

val prems = goalw Class.thy [Int_def]
    "[| c:A;  c:B |] ==> c : A Int B";
by (REPEAT (resolve_tac (prems @ [CollectI,conjI]) 1));
val IntI = result();

val [major] = goalw Class.thy [Int_def] "c : A Int B ==> c:A";
by (rtac (major RS CollectD RS conjunct1) 1);
val IntD1 = result();

val [major] = goalw Class.thy [Int_def] "c : A Int B ==> c:B";
by (rtac (major RS CollectD RS conjunct2) 1);
val IntD2 = result();

val [major,minor] = goal Class.thy
    "[| c : A Int B;  [| c:A; c:B |] ==> P |] ==> P";
by (rtac minor 1);
br (major RS IntD1) 1;
br (major RS IntD2) 1;
val IntE = result();


(*** Rules for set complement -- Compl ***)

val prems = goalw Class.thy [Compl_def]
    "[| c:A ==> False |] ==> c : Compl(A)";
by (REPEAT (ares_tac (prems @ [CollectI,notI]) 1));
val ComplI = result();

(*Better with Classical prover, since negated assumptions behave like
  formulae on the right side of the notional turnstile...*)
val major::prems = goalw Class.thy [Compl_def]
    "[| c : Compl(A) |] ==> ~c:A";
by (rtac (major RS CollectD) 1);
val NComplD = result();

val NComplE = make_elim NComplD;


(*** Monotonicity ***)

val [prem] = goalw Class.thy [mono_def]
    "[| !!A B. A <= B ==> f(A) <= f(B) |] ==> mono(f)";
by (REPEAT (ares_tac [allI, impI, prem] 1));
val monoI = result();

val [major,minor] = goalw Class.thy [mono_def]
    "[| mono(f);  A <= B |] ==> f(A) <= f(B)";
br (major RS spec RS spec RS mp) 1;
br minor 1;
val monoD = result();
