(*  Title: 	ZF/lemmas
    Author: 	Martin D Coen, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Basic introduction and elimination rules for Zermelo-Fraenkel Set Theory 
*)

(*Tactic for proving an "expansive" introduction rule, assuming the negated
  conclusion in all subgoals;  intr is the ordinary intro rule. *)
fun prove_hazXI_tac prems intr =
    rtac classical 1 THEN
    rtac intr 1 THEN
    REPEAT (eresolve_tac (asm_rl::prems) 1);


val basic_ZF_congs = mk_congs set_thy 
    ["op `", "op ``", "op Int", "op Un", "op -", "op <=", "op <!", "op :", 
     "Pow", "Union", "Inter", "fst", "snd", "succ", "Pair", "Upair", "cons",
     "domain", "range", "restrict"];

fun prove_cong_tac prems i =
    REPEAT (ares_tac (prems@[refl]@FOL_congs@basic_ZF_congs) i);

(*** Bounded universal quantifier ***)

val prems = goalw set_thy [Ball_def]
    "[| !!x. x:A ==> P(x) |] ==> ALL x:A. P(x)";
by (REPEAT (ares_tac (prems @ [allI,impI]) 1));
val ballI = result();

val major::prems = goalw set_thy [Ball_def]
    "[| ALL x:A. P(x);  x: A |] ==> P(x)";
br (major RS spec RS mp) 1;
brs prems 1;
val bspec = result();

val major::prems = goalw set_thy [Ball_def]
    "[| ALL x:A. P(x);  P(x) ==> Q;  ~ x:A ==> Q |] ==> Q";
br (major RS allE) 1;
by (REPEAT (eresolve_tac (prems@[asm_rl,impCE]) 1));
val ballE = result();

(*Takes assumptions ALL x:A.P(x) and a:A; creates assumption P(a)*)
fun ball_tac i = etac ballE i THEN contr_tac (i+1);

(*Trival rewrite rule;   (ALL x:A.P)<->P holds only if A is nonempty!*)
val prems = goal set_thy "(ALL x:A. True) <-> True";
by (REPEAT (ares_tac [TrueI,ballI,iffI] 1));
val ball_rew = result();

(*Congruence rule for rewriting*)
val prems = goalw set_thy [Ball_def]
    "[| A=A';  !!x. x:A' ==> P(x) <-> P'(x) \
\    |] ==> (ALL x:A. P(x)) <-> (ALL x:A'. P'(x))";
by (prove_cong_tac prems 1);
val ball_cong = result();

(*** Bounded existential quantifier ***)

val prems = goalw set_thy [Bex_def] "[| P(x);  x: A |] ==> EX x:A. P(x)";
by (REPEAT (ares_tac (prems @ [exI,conjI]) 1));
val bexI = result();

(*Not of the general form for such rules; ~EX has become ALL~ *)
val bexXI = prove_goal set_thy 
   "[| ALL x:A. ~P(x) ==> P(a);  a: A |] ==> EX x:A.P(x)"
 (fn prems=>
  [ (rtac classical 1),
    (REPEAT (ares_tac (prems@[bexI,ballI,notI,notE]) 1))  ]);

val major::prems = goalw set_thy [Bex_def]
    "[| EX x:A. P(x);  !!x. [| x:A; P(x) |] ==> Q \
\    |] ==> Q";
br (major RS exE) 1;
by (REPEAT (eresolve_tac (prems @ [asm_rl,conjE]) 1));
val bexE = result();

(*We do not even have (EX x:A. True) <-> True unless A is nonempty!!*)

val prems = goalw set_thy [Bex_def]
    "[| A=A';  !!x. x:A' ==> P(x) <-> P'(x) \
\    |] ==> (EX x:A. P(x)) <-> (EX x:A'. P'(x))";
by (prove_cong_tac prems 1);
val bex_cong = result();

(*** Rules for subsets ***)

val prems = goalw set_thy [subset_def] "(!!x.x:A ==> x:B) ==> A <= B";
by (REPEAT (ares_tac (prems @ [ballI]) 1));
val subsetI = result();

(*Rule in Modus Ponens style [was called subsetE] *)
val major::prems = goalw set_thy [subset_def] "[| A <= B;  c:A |] ==> c:B";
br (major RS bspec) 1;
brs prems 1;
val subsetD = result();

(*Classical elimination rule*)
val major::prems = goalw set_thy [subset_def]
    "[| A <= B;  ~(c:A) ==> P;  c:B ==> P |] ==> P";
br (major RS ballE) 1;
by (REPEAT (eresolve_tac prems 1));
val subsetCE = result();

(*Takes assumptions A<=B; c:A and creates the assumption c:B *)
fun set_mp_tac i = etac subsetCE i  THEN  mp_tac i;

val subset_refl = prove_goal set_thy "A <= A"
 (K [ (rtac subsetI 1), atac 1 ]);

val prems = goal set_thy "[| A<=B;  B<=C |] ==> A<=C";
by (REPEAT (ares_tac ([subsetI]@(prems RL [subsetD])) 1));
val subset_trans = result();


(*** Rules for equality ***)

(*Anti-symmetry of the subset relation*)
val prems = goal set_thy "[| A <= B;  B <= A |] ==> A = B";
by (REPEAT (resolve_tac (prems@[conjI, extension RS iffD1]) 1));
val equalityI = result();

val prems = goal set_thy "A = B ==> A<=B";
by (rtac (extension RS iffD2 RS conjunct1) 1);
by (resolve_tac prems 1);
val equalityD1 = result();

val prems = goal set_thy "A = B ==> B<=A";
by (rtac (extension RS iffD2 RS conjunct2) 1);
by (resolve_tac prems 1);
val equalityD2 = result();

val prems = goal set_thy "[| A = B;  [| A<=B; B<=A |] ==> P |]  ==>  P";
by (DEPTH_SOLVE (resolve_tac (prems@[equalityD1,equalityD2]) 1));
val equalityE = result();

(*Lemma for creating induction formulae -- for "pattern matching" on p
  To make the induction hypotheses usable, apply "spec" or "bspec" to
  put universal quantifiers over the free variables in p. 
  Would it be better to do subgoal_tac "ALL z. p = f(z) --> R(z)" ??*)
val prems = goal set_thy "[| p: A;  !!z. z: A ==> p=z --> R |] ==> R";
by (rtac mp 1);
by (REPEAT (resolve_tac (refl::prems) 1));
val setup_induction = result();


(*** Rules for the empty set ***)

val [major] = goal set_thy "a:0 ==> P";
by (resolve_tac [major RS (empty_set RS notE)] 1);
val emptyE = result();

goal set_thy "0 <= A";
by (REPEAT (ares_tac [equalityI,subsetI,emptyE] 1));
val empty_subsetI = result();

val [major] = goal set_thy "A<=0 ==> A=0";
by (rtac (empty_subsetI RS (major RS equalityI)) 1);
val empty_subsetD = result();

val prems = goal set_thy "[| !!y. y:A ==> False |] ==> A=0";
by (REPEAT (ares_tac (prems@[empty_subsetI,subsetI,equalityI]) 1 
     ORELSE eresolve_tac (prems RL [FalseE]) 1));
val equals0I = result();

val [major,minor] = goal set_thy "[| A=0;  a:A |] ==> P";
by (rtac (minor RS (major RS equalityD1 RS subsetD RS emptyE)) 1);
val equals0D = result();

val prems = goal set_thy "a:A ==> ~ A=0";
by (rtac notI 1);
by (etac equals0D 1);
by (resolve_tac prems 1);
val nonemptyI = result();

val prems = goal set_thy "[| ~ A=0;  !!y. y:A ==> P |] ==> P";
by (cut_facts_tac prems 1);
by (REPEAT (swap_res_tac ([equals0I]@prems) 1));
val nonemptyE = result();


(*** Rules for PrimReplace -- the primitive form of replacement ***)

val prems = goal set_thy
    "[| a: A;  P(a,b);  \
\       !!x y z.[| x:A; P(x,y); P(x,z) |] ==> y=z \
\    |] ==> b : PrimReplace(A,P)";
by (resolve_tac [replacement RS iffE] 1);
by (etac impE 2 THEN assume_tac 3);
by (REPEAT (ares_tac ([bexI,conjI] @ prems) 1));
val PrimReplaceI = result();

val prems = goal set_thy
    "[| b : PrimReplace(A,P);  \
\       !!x. [| x: A;  P(x,b) |] ==> R; \
\       !!x y z.[| x:A; P(x,y); P(x,z) |] ==> y=z \
\    |] ==> R";
by (rtac bexE 1);
by (resolve_tac [replacement RS iffE] 1);
by (etac impE 2 THEN resolve_tac prems 2);
by (REPEAT (ares_tac prems 1));
val PrimReplaceE = result();


(*** Rules for Replace -- the derived form of replacement ***)

val prems = goal set_thy
    "[| EX! z. P(a,z);  P(a,b);  P(a,c) |] ==> b = c";
by (cut_facts_tac prems 1);
by (etac ex1E 1);
by (rtac trans 1);
by (rtac sym 2);
by (REPEAT (eresolve_tac [asm_rl,mp] 1 ORELSE rtac allE 1));
val ex1_functional = result();

(*Introduction; there must be a unique y such that P(x,y), namely y=b. *)
val prems = goalw set_thy [Replace_def]
    "[| x: A;  P(x,b);  !!y. P(x,y) ==> y=b |] ==> \
\    b : {y. x:A, P(x,y)}";
by (REPEAT (ares_tac (prems@[PrimReplaceI,conjI,ex1I]) 1));
by (REPEAT (eresolve_tac [asm_rl,conjE,ex1_functional] 1));
val ReplaceI = result();

(*Useful??*)
val prems = goal set_thy
    "[| ~ (b: Replace(A,P)) ==> a: A;    \
\       ~ (b: Replace(A,P)) ==> P(a,b);  \
\       !!y. P(a,y) ==> y=b              \
\    |] ==> b : Replace(A,P)";
by (prove_hazXI_tac prems ReplaceI);
val ReplaceXI = result();

(*Elimination; may asssume there is a unique y such that P(x,y), namely y=b. *)
val major::prems = goalw set_thy [Replace_def]
    "[| b : {y. x:A, P(x,y)};  \
\       !!x. [| x: A;  P(x,b);  ALL y. P(x,y)-->y=b |] ==> R \
\    |] ==> R";
br (major RS PrimReplaceE) 1;
by (REPEAT (ares_tac [allI,impI] 1
     ORELSE eresolve_tac (prems@[conjE,ex1_functional]) 1));
val ReplaceE = result();

val prems = goal set_thy
    "[| A=B;  !!x y. x:B ==> P(x,y) <-> Q(x,y) |] ==> \
\    {y. x:A, P(x,y)} = {y. x:B, Q(x,y)}";
val substprems = prems RL [subst, ssubst];
val iffprems = prems RL [iffD1,iffD2];
br equalityI 1;
by (REPEAT (eresolve_tac (substprems@[asm_rl, ReplaceE, spec RS mp]) 1
     ORELSE resolve_tac [subsetI, ReplaceI] 1
     ORELSE (resolve_tac iffprems 1 THEN assume_tac 2)));
val Replace_cong = result();

(*** Rules for RepFun ***)

val prems = goalw set_thy [RepFun_def] "[| a : A |] ==> f(a) : RepFun(A,f)";
by (REPEAT (ares_tac (prems @ [ReplaceI,refl]) 1));
val RepFunI = result();

val major::prems = goalw set_thy [RepFun_def]
    "[| b : RepFun(A, %x.f(x));  \
\       !!x.[| x:A;  b=f(x) |] ==> P |] ==> \
\    P";
br (major RS ReplaceE) 1;
by (REPEAT (ares_tac prems 1));
val RepFunE = result();

val prems = goalw set_thy [RepFun_def]
    "[| A=B;  !!x. x:B ==> f(x)=g(x) |] ==> \
\    RepFun(A, %x.f(x)) = RepFun(B, %x.g(x))";
by (prove_cong_tac (prems@[Replace_cong]) 1);
val RepFun_cong = result();

(*** Rules for Collect -- forming a subset by separation ***)

(* The separation axiom is derivable from the axiom of replacement *)
val prems = goalw set_thy [Collect_def] "x : Collect(A,P) <-> x:A & P(x)";
by (REPEAT (eresolve_tac [conjE, subst, ReplaceE] 1 
     ORELSE ares_tac [refl,iffI,conjI,ReplaceI] 1));
val separation = result();

val prems = goal set_thy "[| a:A;  P(a) |] ==> a : {x:A. P(x)}";
by (resolve_tac [separation RS iffE] 1);
by (etac impE 1 THEN assume_tac 2);
by (REPEAT (ares_tac (prems @ [conjI]) 1));
val CollectI = result();

val prems = goal set_thy "[| a : {x:A. P(x)};  [| a:A; P(a) |] ==> R |] ==> R";
by (resolve_tac [separation RS iffE] 1);
by (resolve_tac prems 1);
by (REPEAT (resolve_tac prems 1 
     ORELSE (etac impE 1 THEN eresolve_tac [conjunct1,conjunct2] 2)));
val CollectE = result();

val [major] = goal set_thy "a : {x:A. P(x)} ==> a:A";
by (rtac (major RS CollectE) 1);
by (assume_tac 1);
val CollectD1 = result();

val [major] = goal set_thy "a : {x:A. P(x)} ==> P(a)";
by (rtac (major RS CollectE) 1);
by (assume_tac 1);
val CollectD2 = result();

val prems = goalw set_thy [Collect_def] 
    "[| A=B;  !!x. x:B ==> P(x) <-> Q(x) |] ==> \
\    {x:A. P(x)} = {x:B. Q(x)}";
by (prove_cong_tac (prems@[Replace_cong]) 1);
val Collect_cong = result();

(*** Rules for Unions ***)

(*The order of the premises presupposes that C is rigid; A may be flexible*)
val prems = goal set_thy "[| B: C;  A: B |] ==> A: Union(C)";
by (resolve_tac [union_iff RS iffD1] 1);
by (REPEAT (resolve_tac (prems @ [bexI]) 1));
val UnionI = result();

val prems = goal set_thy
    "[| ~ (A: Union(C)) ==> B: C;  \
\       ~ (A: Union(C)) ==> A: B |]  ==>  A: Union(C)";
by (prove_hazXI_tac prems UnionI);
val UnionXI = result();

val prems = goal set_thy
    "[| A : Union(C);  !!B.[| A: B;  B: C |] ==> R |] ==> R";
by (resolve_tac [union_iff RS iffD2 RS bexE] 1);
by (REPEAT (ares_tac prems 1));
val UnionE = result();

(*** Rules for Inter ***)

(* Intersection is well-behaved only if the family is non-empty! *)
val prems = goalw set_thy [Inter_def]
    "[| !!x. x: C ==> A: x;  c:C |] ==> A : Inter(C)";
by (DEPTH_SOLVE (ares_tac ([CollectI,UnionI,ballI] @ prems) 1));
val InterI = result();

(*A "destruct" rule -- every B in C contains A as an element, but
  A:B can hold when B:C does not!  This rule is analogous to "spec". *)
val [major,minor] = goalw set_thy [Inter_def]
    "[| A : Inter(C);  B : C |] ==> A : B";
br (major RS CollectD2 RS bspec) 1;
br minor 1;
val InterD = result();

(*"Classical" elimination rule -- does not require exhibiting B:C *)
val major::prems = goalw set_thy [Inter_def]
    "[| A : Inter(C);  A:B ==> R;  ~ B:C ==> R |] ==> R";
br (major RS CollectD2 RS ballE) 1;
by (REPEAT (eresolve_tac prems 1));
val InterE = result();

(*** Rules for Unions of families ***)
(* UN x:A. B(x) abbreviates Union(RepFun(A, %x.B(x))) *)

(*The order of the premises presupposes that A is rigid; b may be flexible*)
val prems = goal set_thy "[| a: A;  b: B(a) |] ==> b: (UN x:A. B(x))";
by (REPEAT (resolve_tac (prems@[UnionI,RepFunI]) 1));
val UN_I = result();

val major::prems = goal set_thy
    "[| b : (UN x:A. B(x));  !!x.[| x: A;  b: B(x) |] ==> R |] ==> R";
by (rtac (major RS UnionE) 1);
by (REPEAT (eresolve_tac (prems@[asm_rl, RepFunE, subst]) 1));
val UN_E = result();

(*** Rules for Intersections of families ***)
(* INT x:A. B(x) abbreviates Inter(RepFun(A, %x.B(x))) *)

val prems = goal set_thy
    "[| !!x. x: A ==> b: B(x);  a: A |] ==> b: (INT x:A. B(x))";
by (REPEAT (ares_tac (prems@[InterI,RepFunI]) 1
     ORELSE eresolve_tac [RepFunE,ssubst] 1));
val INT_I = result();

val [major,minor] = goal set_thy
    "[| b : (INT x:A. B(x));  a: A |] ==> b : B(a)";
by (rtac (major RS InterD) 1);
by (rtac (minor RS RepFunI) 1);
val INT_E = result();

(*** Rules for Powersets ***)

val [prem] = goal set_thy "A <= B ==> A : Pow(B)";
by (rtac (prem RS (power_set RS iffD1)) 1);
val PowI = result();

val [major] = goal set_thy "A : Pow(B)  ==>  A<=B";
by (rtac (major RS (power_set RS iffD2)) 1);
val PowD = result();


val lemmas_cs = FOL_cs
  addSIs [ballI, InterI, CollectI, PowI, subsetI]
  addIs [bexI, UnionI, ReplaceI, RepFunI]
  addXIs [bexXI, UnionXI, ReplaceXI]
(*many of these are not safe at all! including emptyE*)
  addSEs [bexE, make_elim PowD, UnionE, ReplaceE, RepFunE,
	  CollectE, emptyE]
  addEs [ballE, InterD, make_elim InterD, subsetD, subsetCE ];

