(*  Title: 	ZF/ex/simult
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Primitives for simultaneous recursive type definitions: Outl and Outr
  includes worked example of trees & forests

Would prefer to use the following, if univ(A) were closed under them:
  ("Outl_def",  "Outl(s) == {x. z: s, z = Inl(x)}" ),
  ("Outr_def",  "Outr(s) == {y. z: s, z = Inr(y)}" ),
*)

writeln"File ZF/ex/simult";

structure Simult =
struct

val const_decs = 
 [ 
   (["Outl","Outr"],	"[i,i] => i"),
   (["tree","forest","TF"],	"i=>i"),
   (["sub_TF"],	"i => i"),
   (["TF_rec","tree_rec","forest_rec"], 
                "[i, i, [i,i,i]=>i, i, [i,i,i,i]=>i] => i")
 ];

val thy = extend_theory List.thy "forest"
     ([], [], [], const_decs, None)
 [ 
    (*operator for selecting out the various types*)
  ("Outl_def",	"Outl(A,s) == {x: univ(A). EX z: s. z = Inl(x)}" ),
  ("Outr_def",	"Outr(A,s) == {y: univ(A). EX z: s. z = Inr(y)}" ),

  ("TF_def",
   "TF(A) == lfp(lam s:Pow(univ(A)). A * Outr(A,s) \
\                                   + ({0} Un Outl(A,s)*Outr(A,s)))"),

  ("tree_def",	"tree(A) == Outl(A,TF(A))" ),
  ("forest_def", "forest(A) == Outr(A,TF(A))" ),

  ("sub_TF_def",	
   "sub_TF(A) == (UN ts: forest(A). UN x: A. {<Inr(ts), Inl(<x,ts>)>}) \
\             Un (UN t: tree(A). UN ts: forest(A). \
\                        {<Inl(t), Inr(<t,ts>)>, <Inr(ts), Inr(<t,ts>)>})" ),

  ("TF_rec_def",	
   "TF_rec(A,M,f,c,g) == wfrec(trancl(sub_TF(A)),  M,  \
\      %M rec. when(M, %ats. split(ats, %a ts. f(a, ts, rec`Inr(ts))), \
\                      %l. list_case(l, c, \
\                             %t ts. g(t, ts, rec`Inl(t), rec`Inr(ts)))))" ),

  ("tree_rec_def", "tree_rec(A, t, f, c, g) == TF_rec(A, Inl(t), f, c, g)" ),

  ("forest_rec_def",
   "forest_rec(A, ts, f, c, g) == TF_rec(A, Inr(ts), f, c, g)" )
 ];
end;

local val ax = get_axiom Simult.thy
in 
val Outl_def    = ax"Outl_def";
val Outr_def    = ax"Outr_def";
val TF_def      = ax"TF_def";
val tree_def    = ax"tree_def";
val forest_def  = ax"forest_def";
val sub_TF_def  = ax"sub_TF_def";
val TF_rec_def  = ax"TF_rec_def";
val tree_rec_def        = ax"tree_rec_def";
val forest_rec_def      = ax"forest_rec_def";
end;


(*** General rules for Outl and Outr ***)

val prems = goalw Simult.thy [Outl_def]
    "[| Inl(a) : s;  a: univ(A) |] ==> a : Outl(A,s)";
by (REPEAT (resolve_tac (prems@[refl,bexI,CollectI]) 1));
val OutlI = result();

val major::prems = goalw Simult.thy [Outl_def]
    "[| a : Outl(A,s);  [| Inl(a) : s;  a: univ(A) |] ==> P  \
\    |] ==> P";
by (rtac (major RS CollectE) 1);
by (etac bexE 1);
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [bexE, subst] 1));
val OutlE = result();

val prems = goalw Simult.thy [Outr_def]
    "[| Inr(a) : s;  a: univ(A) |] ==> a : Outr(A,s)";
by (REPEAT (resolve_tac (prems@[refl,bexI,CollectI]) 1));
val OutrI = result();

val major::prems = goalw Simult.thy [Outr_def]
    "[| a : Outr(A,s);  [| Inr(a) : s;  a: univ(A) |] ==> P  \
\    |] ==> P";
by (rtac (major RS CollectE) 1);
by (etac bexE 1);
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [bexE, subst] 1));
val OutrE = result();

goal Simult.thy "Outl(A,s) <= univ(A)";
by (rtac subsetI 1);
by (etac OutlE 1);
by (assume_tac 1);
val Outl_subset_univ = result();

goal Simult.thy "Outr(A,s) <= univ(A)";
by (rtac subsetI 1);
by (etac OutrE 1);
by (assume_tac 1);
val Outr_subset_univ = result();

val [prem] = goal Simult.thy "B<=C ==> Outl(A,B) <= Outl(A,C)";
by (REPEAT (eresolve_tac [OutlE, prem RS subsetD] 1
     ORELSE ares_tac [subsetI,OutlI] 1));
val Outl_mono = result();

val [prem] = goal Simult.thy "B<=C ==> Outr(A,B) <= Outr(A,C)";
by (REPEAT (eresolve_tac [OutrE, prem RS subsetD] 1
     ORELSE ares_tac [subsetI,OutrI] 1));
val Outr_mono = result();


(*** Monotonicity and unfolding of the function ***)

goal Simult.thy 
    "(lam s:Pow(univ(A)). A*Outr(A,s) + ({0} Un Outl(A,s)*Outr(A,s))) \
\      : mono(Pow(univ(A)), Pow(univ(A)))";
by (rtac lam_mono_Powtype 1);
by (REPEAT (ares_tac [subset_refl, zero_in_univ,
                      singleton_subsetI, A_subset_univ,
                      Un_mono, plus_mono, product_mono, Outl_mono, Outr_mono,
		      Un_least, Outl_subset_univ, Outr_subset_univ,
		      plus_subset_univ, product_subset_univ] 1));
val TFfun_mono = result();

goalw Simult.thy [TF_def]
    "TF(A) = A*Outr(A,TF(A)) + ({0} Un Outl(A,TF(A))*Outr(A,TF(A)))";
by (rtac (TFfun_mono RS lam_Tarski_theorem) 1);
val TF_unfold = result();

(** Introduction rule for "tree" **)

val prems = goalw Simult.thy [tree_def,forest_def]
    "[| a:A;  ts: forest(A) |] ==> <a,ts> : tree(A)";
by (cut_facts_tac prems 1);
by (rtac (TF_unfold RS ssubst) 1);
by (rtac (SigmaI RS plus_InlI RS OutlI) 1);
by (REPEAT (etac OutrE 1
     ORELSE ares_tac (prems@[pair_in_univ,A_in_univ]) 1));
val treeI = result();

(** Introduction rules for "forest" **)
 
goalw Simult.thy [forest_def] "0 : forest(A)";
by (rtac (TF_unfold RS ssubst) 1);
by (rtac (plus_InrI RS OutrI) 1);
by (rtac (singletonI RS UnI1) 1);
by (rtac zero_in_univ 1);
val forest_0I = result();

val prems = goalw Simult.thy [tree_def,forest_def]
    "[| t: tree(A);  ts: forest(A) |] ==> <t,ts> : forest(A)";
by (cut_facts_tac prems 1);
by (rtac (TF_unfold RS ssubst) 1);
by (rtac (plus_InrI RS OutrI) 1);
by (rtac (SigmaI RS UnI2) 1);
by (REPEAT (ares_tac (prems@[pair_in_univ]) 1
     ORELSE eresolve_tac [OutlE,OutrE] 1));
val forest_PairI = result();

(** Elimination -- structural induction on TF **)

val prems = goalw Simult.thy [tree_def,forest_def]
    "[| z: TF(A);  \
\       !!x ts. [| x: A;  ts: forest(A);  R(Inr(ts)) |] ==> R(Inl(<x,ts>)); \
\       R(Inr(0));        \
\       !!t ts. [| t:  tree(A);  ts: forest(A);  R(Inl(t));  R(Inr(ts)) \
\               |] ==> R(Inr(<t,ts>))    \
\    |] ==> R(z)";
by (rtac lam_general_induction 1);
by (rtac TFfun_mono 3);
by (fold_tac [TF_def]);
by (REPEAT (ares_tac ([OutlI,OutrI]@prems) 1
     ORELSE eresolve_tac [plusE, OutlE, OutrE, 
			  UnE, singletonE, SigmaE, CollectE, ssubst] 1));
val TF_induct = result();

(*This lemma replaces a use of subgoal_tac to prove tree_forest_induct*)
val [prem] = goalw Simult.thy [tree_def,forest_def]
    "[| !!z. z: TF(A) ==> \
\            (ALL x. z=Inl(x) --> P(x)) & (ALL y. z=Inr(y) --> Q(y))    \
\    |] ==> (ALL z:tree(A). P(z)) & (ALL z:forest(A). Q(z))";
by (REPEAT (eresolve_tac [spec RS mp, OutlE, OutrE, prem RS conjE] 1
     ORELSE ares_tac [refl,ballI,conjI] 1));
val TF_induct_lemma = result();

val plus_cs = ZF_cs addEs [make_elim Inl_inject, make_elim Inr_inject, 
			   Inl_neq_Inr, sym RS Inl_neq_Inr];

val prems = goal Simult.thy
    "[| !!x ts. [| x: A;  ts: forest(A);  Q(ts) |] ==> P(<x,ts>);     \
\	Q(0);        \
\       !!t ts. [| t: tree(A);  ts: forest(A);  P(t);  Q(ts) \
\               |] ==> Q(<t,ts>)    \
\    |] ==> (ALL t:tree(A). P(t)) & (ALL ts:forest(A). Q(ts))";
by (rtac TF_induct_lemma 1);
by (etac TF_induct 1);
by (REPEAT (fast_tac (plus_cs addIs prems) 1));
val tree_forest_induct = result();


(** Introduction rules for "TF" **)

val [major] = goalw Simult.thy [tree_def]
    "[| t: tree(A) |] ==> Inl(t) : TF(A)";
by (rtac (major RS OutlE) 1);
by (assume_tac 1);
val TF_InlI = result();

val [major] = goalw Simult.thy [forest_def]
    "[| ts: forest(A) |] ==> Inr(ts) : TF(A)";
by (rtac (major RS OutrE) 1);
by (assume_tac 1);
val TF_InrI = result();

(*** The sub_TF relation ***)

goalw Simult.thy [sub_TF_def] "sub_TF(A) <= TF(A) * TF(A)";
by (fast_tac (ZF_cs addIs [TF_InlI,TF_InrI,treeI,forest_PairI]) 1);
val sub_TF_type = result();

(** Introduction rules for 'sub_TF' **)

val prems = goalw Simult.thy [sub_TF_def] 
    "[| ts: forest(A);  x: A |] ==> <Inr(ts), Inl(<x,ts>)> : sub_TF(A)";
by (fast_tac (ZF_cs addIs prems) 1);
val sub_TF_I1 = result();

val prems = goalw Simult.thy [sub_TF_def] 
    "[| t: tree(A);  ts: forest(A) |] ==> <Inl(t), Inr(<t,ts>)> : sub_TF(A)";
by (fast_tac (ZF_cs addIs prems) 1);
val sub_TF_I2 = result();

val prems = goalw Simult.thy [sub_TF_def] 
    "[| t: tree(A);  ts: forest(A) |] ==> <Inr(ts), Inr(<t,ts>)> : sub_TF(A)";
by (fast_tac (ZF_cs addIs prems) 1);
val sub_TF_I3 = result();

val major::prems = goalw Simult.thy [sub_TF_def]
 "[| p : sub_TF(A);  \
\    !!ts x.[| ts: forest(A);  x: A;  p = <Inr(ts), Inl(<x,ts>)> |] ==> R; \
\    !!t ts.[| t: tree(A); ts: forest(A); p = <Inl(t), Inr(<t,ts>)> |] ==> R; \
\    !!t ts.[| t: tree(A); ts: forest(A); p = <Inr(ts), Inr(<t,ts>)> |] ==> R \
\  |] ==> R";
by (cut_facts_tac [major] 1);
by (DEPTH_SOLVE (eresolve_tac [UnE,UN_E,emptyE,consE] 1
          ORELSE ares_tac prems 1));
val sub_TF_E = result();

goal Simult.thy "wf(sub_TF(A))";
by (rtac (sub_TF_type RS field_rel_subset RS wfI2) 1);
by (rtac subsetI 1);
by (etac TF_induct 1);
by (fast_tac (plus_cs addIs [treeI RS TF_InlI]
	              addSEs [bspec RS mp, sub_TF_E]) 1);
by (fast_tac (plus_cs addIs [forest_0I RS TF_InrI]
	              addSEs [bspec RS mp, sub_TF_E]) 1);
by (fast_tac (plus_cs addIs [forest_PairI RS TF_InrI]
	              addSEs [bspec RS mp, sub_TF_E]) 1);
val wf_sub_TF = result();


(*** TF_rec -- by wf recursion on sub_TF ***)

(*Used only to verify TF_rec*)
val TF_rec_ss = ZF_ss 
      addcongs (mk_typed_congs Simult.thy 
		     [("f", "[i,i,i]=>i"), ("g", "[i,i,i,i]=>i")])
      addrews [trans_trancl, wf_sub_TF RS wf_trancl, 
	       TF_rec_def RS wfrec_def_conv,
	       when_Inl_conv, when_Inr_conv, 
	       list_case_0_conv, list_case_Pair_conv];

(** conversion rules **)

val prems = goalw Simult.thy [tree_rec_def,forest_rec_def]
    "[| a: A;  ts: forest(A) |] ==> \
\    tree_rec(A, <a,ts>, f, c, g) = f(a, ts, forest_rec(A,ts,f,c,g))";
by (SIMP_TAC (TF_rec_ss addrews 
		([under_iff,r_sub_trancl,sub_TF_I1]@prems)) 1);
val tree_rec_conv = result();

goalw Simult.thy [forest_rec_def] "forest_rec(A, 0, f, c, g) = c";
by (SIMP_TAC TF_rec_ss 1);
val forest_rec_0_conv = result();

val prems = goalw Simult.thy [tree_rec_def,forest_rec_def]
    "[| t: tree(A);  ts: forest(A) |] ==> \
\    forest_rec(A, <t,ts>, f, c, g) = \
\      g(t, ts, tree_rec(A, t, f, c, g), forest_rec(A, ts, f, c, g))";
by (SIMP_TAC (TF_rec_ss addrews 
		([under_iff,r_sub_trancl,sub_TF_I2,sub_TF_I3]@prems)) 1);
val forest_rec_Pair_conv = result();

val TF_ss = ZF_ss addrews
  [tree_rec_conv, forest_rec_0_conv, forest_rec_Pair_conv,
   treeI, forest_0I, forest_PairI];

(*Type checking*)
val prems = goal Simult.thy
    "[| !!x ts r. [| x: A;  ts: forest(A);  r: D(ts) \
\               |] ==> f(x,ts,r): C(<x,ts>);     \
\	c : D(0);        \
\       !!t ts r1 r2. [| t: tree(A);  ts: forest(A);  r1: C(t);  r2: D(ts) \
\               |] ==> g(t,ts,r1,r2): D(<t,ts>)    \
\    |] ==> (ALL t:tree(A). tree_rec(A, t, f, c, g) : C(t)) \
\         & (ALL ts:forest(A). forest_rec(A,ts,f,c,g) : D(ts))";
fr tree_forest_induct;
by (SIMP_TAC (TF_ss addrews prems) 1);
by (SIMP_TAC (TF_ss addrews prems) 1);
by (SIMP_TAC (TF_ss addrews prems) 1);
val tree_forest_rec_type = result();
