(*  Title: 	ZF/perm
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

The theory underlying permutation groups
  -- Composition of relations, the identity relation, converse of a relation
  -- Injections, surjections, bijections
  -- Lemmas for the Schroeder-Bernstein Theorem
*)

writeln"File ZF/ex/perm";

val perm_mixfix =                    (*composition*)
    [ Infixr("O","[i,i]=>i",60) ];

val perm_sext = Sext{mixfix		= perm_mixfix,
                     parse_translation	= [],
                     print_translation	= []};

val perm_const_decs =
    [ (["id","converse"],	"i=>i"),
      (["inj","surj","bij"],	"[i,i]=>i") ];

val perm_thy = extend_theory set_thy "perm"
     ([], [], [], perm_const_decs, Some perm_sext)
  [ 
    ("comp_def",	(*composition of relations/functions*)
	"r O s == {xz : domain(s)*range(r) . \
\                  EX x y z. xz=<x,z> & <x,y>:s & <y,z>:r}"),
    ("id_def",		(*the identity function for A*)
	"id(A) == (lam x:A. x)"),
    ("converse_def",	(*the converse of relation r (inverse of function)*)
	"converse(r) == {yx. xy:r, EX x y. xy=<x,y> & yx=<y,x>}"),
    ("inj_def",		(*one-to-one functions from A to B*)
	"inj(A,B) == \
\            { f: A->B. ALL w:A. ALL x:A. f`w=f`x --> w=x}"),
    ("surj_def",	(*onto functions from A to B*)
	"surj(A,B) == { f: A->B . ALL y:B. EX x:A. f`x=y}"),
    ("bij_def",		(*one-to-one and onto functions*)
	"bij(A,B) == inj(A,B) Int surj(A,B)")
  ];

local val ax = get_axiom perm_thy in 
val comp_def = ax"comp_def";
val id_def = ax"id_def";
val converse_def = ax"converse_def";
val inj_def = ax"inj_def";
val surj_def = ax"surj_def";
val bij_def = ax"bij_def";
end;


(** Surjective function space **)

val prems = goal perm_thy
    "[| f: A->B;  \
\       !!y. y: B ==> EX x:A. f`x=y |] ==> \
\    f: surj(A,B)";
by (rewtac surj_def);
by (REPEAT (ares_tac (prems@[CollectI,conjI,ballI]) 1));
val surjI = result();

val prems = goal perm_thy
    "[| f: surj(A,B);  \
\       [| f: A->B;  ALL y:B. EX x:A. f`x=y |] ==> P |] ==> \
\    P";
by (cut_facts_tac prems 1);
by (rewtac surj_def);
by (REPEAT (ares_tac prems 1 ORELSE eresolve_tac [CollectE,conjE] 1));
val surjE = result();

val prems = goal perm_thy
    "f: surj(A,B) ==> f: A->B";
br surjE 1;
brs prems 1;
ba 1;
val surj_is_fun = result();

val prems = goal perm_thy
    "f : Pi(A,B) ==> f: surj(A,range(f))";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [surjI] 1
     ORELSE eresolve_tac [rangeE,range_of_fun] 1));
by (REPEAT (ares_tac [bexI] 1
     ORELSE eresolve_tac [apply_equality,Pair_mem_PiE] 1));
val fun_is_surj = result();

val prems = goal perm_thy
    "f: surj(A,B) ==> range(f)=B";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [equalityI,subsetI] 1
     ORELSE set_mp_tac 1
     ORELSE eresolve_tac [surjE,rangeE,Pair_mem_PiE] 1));
by (EVERY1 [ball_tac, etac bexE, etac subst, 
	    rtac rangeI, etac apply_Pair, assume_tac]);
val surj_range = result();


(** Injective function space **)

val prems = goal perm_thy
    "[| f: A->B;  \
\       !!w x. [| w: A;  x: A;  f`w=f`x |] ==> w=x \
\    |] ==> f: inj(A,B)";
by (rewtac inj_def);
by (REPEAT (ares_tac (prems@[CollectI,conjI,impI,ballI]) 1));
val injI = result();

val prems = goal perm_thy
    "[| f: inj(A,B);  \
\       [| f: A->B;  ALL w:A. ALL x:A. f`w=f`x --> w=x |] ==> P \
\    |] ==> P";
by (cut_facts_tac prems 1);
by (rewtac inj_def);
by (REPEAT (ares_tac prems 1 ORELSE eresolve_tac [CollectE,conjE] 1));
val injE = result();

val prems = goal perm_thy
    "f: inj(A,B) ==> f: A->B";
br injE 1;
brs prems 1;
ba 1;
val inj_is_fun = result();

val prems = goal perm_thy
    "[| <a,b>:f;  <c,b>:f;  f: inj(A,B) |] ==> a=c";
by (cut_facts_tac prems 1);
by (etac injE 1);
by (REPEAT (ares_tac [refl] 1
     ORELSE eresolve_tac [ballE, mp, apply_equality RS ssubst] 1));
(*put back things that were discarded...*)
by (cut_facts_tac prems 1);
by (REPEAT (contr_tac 1 ORELSE eresolve_tac [asm_rl,Pair_mem_PiE] 1));
val inj_equality = result();

(** Bijections -- simple lemmas but no intro/elim rules -- use unfolding **)

val prems = goal perm_thy
    "f: bij(A,B) ==> f: A->B";
by (cut_facts_tac prems 1);
by (rewtac bij_def);
be IntE 1;
be inj_is_fun 1;
val bij_is_fun = result();

(** Identity function **)

val prems = goal perm_thy
    "a:A ==> <a,a>:id(A)";
by (rewtac id_def);
by (rtac lamI 1);
by (resolve_tac prems 1);
val idI = result();

val prems = goal perm_thy
    "[| c : id(A);  \
\       !!x. [| x:A;  c=<x,x> |] ==> P \
\    |] ==> P";
by (cut_facts_tac prems 1);
by (rewtac id_def);
by (REPEAT (etac lamE 1 ORELSE ares_tac prems 1));
val idE = result();

val prems = goal perm_thy
    "<a,b> : id(A) ==> a=b";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [idE,Pair_inject,sym,ssubst] 1));
val id_Pair = result();

goal perm_thy
    "id(A) : A->A";
by (rewtac id_def);
by (rtac lam_type 1);
by (assume_tac 1);
val id_type = result();

val prems = goal perm_thy
    "a: A ==> id(A)`a = a";
by (rewtac id_def);
by (rtac beta_conv 1);
by (resolve_tac prems 1);
val id_conv = result();

goal perm_thy
    "id(A): inj(A,A)";
by (REPEAT (ares_tac [injI,id_type] 1));
by (rtac trans 1);
by (rtac sym 1);
by (REPEAT (eresolve_tac [id_conv,ssubst] 1));
val id_inj = result();

goal perm_thy
    "id(A): surj(A,A)";
by (REPEAT (ares_tac [surjI,bexI,impI,id_type,id_conv] 1));
val id_surj = result();

goal perm_thy
    "id(A): bij(A,A)";
by (rewtac bij_def);
by (REPEAT (ares_tac [IntI,id_inj,id_surj] 1));
val id_bij = result();


(** Converse of a relation **)

val prems = goal perm_thy
    "<a,b>:r ==> <b,a>:converse(r)";
by (rewtac converse_def);
by (REPEAT (ares_tac ([ReplaceI,exI,conjI,refl] @ prems) 1
     ORELSE eresolve_tac [exE,conjE,Pair_inject,ssubst] 1));
val converseI = result();

val prems = goal perm_thy
    "[| yx : converse(r);  \
\       !!x y. [| yx=<y,x>;  <x,y>:r |] ==> P \
\    |] ==> P";
by (cut_facts_tac prems 1);
by (rewtac converse_def);
by (REPEAT (eresolve_tac [ReplaceE, exE, conjE] 1
     ORELSE ares_tac prems 1));
by (eresolve_tac [subst] 1);
by (assume_tac 1);
val converseE = result();

val prems = goal perm_thy
    "<a,b> : converse(r) ==> <b,a> : r";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl,converseE,Pair_inject,ssubst] 1));
val converseD = result();

val prems = goal perm_thy
    "r<=A*B ==> converse(converse(r)) = r";
by (REPEAT (ares_tac [equalityI,subsetI] 1
     ORELSE eresolve_tac [converseE,Pair_inject,ssubst] 1));
by (cut_facts_tac prems 1);
by (set_mp_tac 1);
by (etac SigmaE 1);
by (hyp_subst_tac 1);
by (REPEAT (ares_tac [converseI] 1));
val converse_of_converse = result();

val prems = goal perm_thy
    "r<=A*B ==> converse(r)<=B*A";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [subsetI,SigmaI] 1
     ORELSE set_mp_tac 1
     ORELSE eresolve_tac [converseE,SigmaE2,ssubst] 1));
val converse_rel = result();

val prems = goal perm_thy
    "f: inj(A,B) ==> converse(f) : range(f)->A";
by (EVERY1 [cut_facts_tac prems,
	    etac injE,
	    etac PiE,
	    rtac PiI,
	    rtac converse_rel,
	    rtac subset_trans,
	    etac domain_times_range,
	    rtac Sigma_mono,
	    etac domain_subset,
	    rtac subset_refl,
	    etac rangeE,
	    rtac ex1I,
	    etac converseI,
	    etac (make_elim converseD),
	    etac inj_equality,
	    REPEAT o (ares_tac prems)]);
val inj_converse_fun = result();

val prems = goal perm_thy
    "f: inj(A,B) ==> converse(f): surj(range(f), A)";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac (reslist(prems,1,inj_converse_fun)@
                      [surjI,bexI,apply_equality,converseI,rangeI]) 1
     ORELSE eresolve_tac [injE,apply_Pair] 1));
val inj_converse_surj = result();

(*The premises are equivalent to saying that f is injective...*) 
val prems = goal perm_thy
    "[| f: A->B;  converse(f): C->A;  a: A |] ==> converse(f)`(f`a) = a";
by (REPEAT (ares_tac (prems@[apply_equality,converseI,apply_Pair]) 1
     ORELSE etac apply_Pair 1));
val left_inverse_lemma = result();

val prems = goal perm_thy
    "[| f: inj(A,B);  a: A |] ==> converse(f)`(f`a) = a";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [left_inverse_lemma,inj_converse_fun] 1
     ORELSE etac injE 1));
val left_inverse = result();

val prems = goal perm_thy
    "[| f: A->B;  converse(f): C->A;  b: C |] ==> f`(converse(f)`b) = b";
by (rtac apply_equality 1);
by (resolve_tac [(apply_equality RS ssubst)] 1);
by (REPEAT (resolve_tac (prems@[apply_Pair, apply_Pair RS converseD]) 1));
val right_inverse_lemma = result();

val prems = goal perm_thy
    "[| f: inj(A,B);  b: range(f) |] ==> f`(converse(f)`b) = b";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [right_inverse_lemma,inj_converse_fun] 1
     ORELSE etac injE 1));
val right_inverse = result();

val prems = goal perm_thy
    "f: inj(A,B) ==> converse(f): inj(range(f), A)";
by (cut_facts_tac prems 1);
by (rtac injI 1);
by (etac inj_converse_fun 1);
(*apply f to both sides...*)
by (eresolve_tac [(subst_context RS box_equals)] 1);
(*...and simplify using right_inverse*)
by (REPEAT (eresolve_tac [asm_rl, right_inverse] 1));
val inj_converse_inj = result();

val prems = goal perm_thy
    "f: bij(A,B) ==> converse(f): bij(B,A)";
by (cut_facts_tac prems 1);
by (rewtac bij_def);
by (etac IntE 1);
by (eresolve_tac [(surj_range RS subst)] 1);
by (rtac IntI 1);
by (etac inj_converse_inj 1);
by (etac inj_converse_surj 1);
val bij_converse_bij = result();


(** Composition of two relations **)

val prems = goal perm_thy
    "[| <a,b>:s; <b,c>:r |] ==> <a,c> : r O s";
by (cut_facts_tac prems 1);
by (rewtac comp_def);
by (REPEAT (ares_tac [SigmaI,CollectI,exI,conjI,refl,domainI,rangeI] 1
     ORELSE eresolve_tac [CollectE,exE,conjE,Pair_inject,ssubst] 1));
val compI = result();

val prems = goal perm_thy
    "[| xz : r O s;  \
\       !!x y z. [| xz=<x,z>;  <x,y>:s;  <y,z>:r |] ==> P \
\    |] ==> P";
by (cut_facts_tac prems 1);
by (rewtac comp_def);
by (REPEAT (eresolve_tac [CollectE, exE, conjE] 1
     ORELSE ares_tac prems 1));
val compE = result();

val prems = goal perm_thy
    "[| <a,c> : r O s;  \
\       !!y. [| <a,y>:s;  <y,c>:r |] ==> P \
\    |] ==> P";
by (rtac compE 1);
by (REPEAT (ares_tac prems 1
     ORELSE eresolve_tac [Pair_inject,ssubst] 1));
val compEpair = result();

(*a typing theorem: composition preserves relations*)
val prems = goal perm_thy
    "[| s<=A*B;  r<=B*C |] ==> (r O s) <= A*C";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [subsetI,SigmaI] 1
     ORELSE set_mp_tac 1
     ORELSE eresolve_tac [compE,SigmaE2,ssubst] 1));
val comp_rel = result();

(*associative law for composition*)
goal perm_thy
    "(r O s) O t = r O (s O t)";
by (REPEAT (ares_tac [equalityI,subsetI,compI] 1
     ORELSE eresolve_tac [compEpair,compE,ssubst] 1));
val comp_assoc = result();

(*left identity of composition -- 1st inclusion*)
goal perm_thy
    "id(A) O r <= r";
by (rtac subsetI 1);
by (etac compE 1);
by (etac ssubst 1);
by (eresolve_tac [(id_Pair RS subst)] 1);
by (assume_tac 1);
val left_comp_id1 = result();

(*left identity of composition -- 2nd inclusion*)
val prems = goal perm_thy
    "[| r<=A*B; B<=C |] ==> r <= id(C) O r";
by (cut_facts_tac prems 1);
by (rtac subsetI 1);
by (etac (subsetD RS SigmaE) 1);
by (assume_tac 1);
by (hyp_subst_tac 1);
by (REPEAT (ares_tac [compI,idI] 1
     ORELSE eresolve_tac [subsetD] 1));
val left_comp_id2 = result();

(*left identity of composition*)
val prems = goal perm_thy
    "r<=A*B ==> id(B) O r = r";
by (REPEAT (ares_tac (prems@[equalityI,left_comp_id1,left_comp_id2,
			     subset_refl]) 1));
val left_comp_id = result();


(*right identity of composition -- 1st inclusion*)
goal perm_thy
    "r O id(A) <= r";
by (rtac subsetI 1);
by (etac compE 1);
by (etac ssubst 1);
by (eresolve_tac [(id_Pair RS ssubst)] 1);
by (assume_tac 1);
val right_comp_id1 = result();

(*right identity of composition -- 2nd inclusion*)
val prems = goal perm_thy
    "[| r<=A*B; A<=C |] ==> r <= r O id(C)";
by (cut_facts_tac prems 1);
by (rtac subsetI 1);
by (etac (subsetD RS SigmaE) 1);
by (assume_tac 1);
by (hyp_subst_tac 1);
by (REPEAT (ares_tac [compI,idI] 1
     ORELSE eresolve_tac [subsetD] 1));
val right_comp_id2 = result();

(*right identity of composition*)
val prems = goal perm_thy
    "r<=A*B ==> r O id(A) = r";
by (REPEAT (ares_tac (prems@[equalityI,right_comp_id1,right_comp_id2,
			     subset_refl]) 1));
val right_comp_id = result();


(** Composition preserves functions, injections, and surjections **)

val prems = goal perm_thy
    "[| g: A->B;  f: B->C |] ==> (f O g) : A->C";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [PiI,comp_rel,ex1I,compI] 1
     ORELSE eresolve_tac [fun_is_rel,apply_Pair,apply_type,compE,
			  Pair_inject, ssubst] 1));
by (REPEAT (ares_tac [refl] 1
     ORELSE eresolve_tac [apply_equality RS ssubst] 1));
val comp_func = result();

val prems = goal perm_thy
    "[| g: A->B;  f: B->C;  a:A |] ==> (f O g)`a = f`(g`a)";
by (REPEAT (resolve_tac (prems@[comp_func,apply_equality,compI,
	                        apply_Pair,apply_type]) 1));
val comp_func_apply = result();

val prems = goal perm_thy
    "[| g: inj(A,B);  f: inj(B,C) |] ==> (f O g) : inj(A,C)";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl, injE, bspec RS bspec RS mp, box_equals] 1
     ORELSE resolve_tac [injI,comp_func,apply_type,comp_func_apply] 1));
val comp_inj = result();

val prems = goal perm_thy
    "[| g: surj(A,B);  f: surj(B,C) |] ==> (f O g) : surj(A,C)";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl, surjE, bexE] 1
     ORELSE ball_tac 1
     ORELSE resolve_tac [surjI, comp_func] 1));
by (REPEAT (ares_tac [bexI, comp_func_apply RS trans] 1
     ORELSE (etac ssubst 1 THEN assume_tac 1)));
val comp_surj = result();

val prems = goal perm_thy
    "[| g: bij(A,B);  f: bij(B,C) |] ==> (f O g) : bij(A,C)";
by (cut_facts_tac prems 1);
by (rewtac bij_def);
by (REPEAT (eresolve_tac [asm_rl, IntE] 1
     ORELSE resolve_tac [IntI, comp_inj, comp_surj] 1));
val comp_bij = result();


(** inverses of composition **)

(*half of the equality*)
val prems = goal perm_thy
    "f: A->B ==> id(A) <= converse(f) O f";
by (REPEAT (ares_tac (prems@[subsetI,compI,apply_Pair,converseI]) 1
     ORELSE eresolve_tac [idE,ssubst] 1));
val left_comp_inverse1 = result();

val prems = goal perm_thy
    "f: inj(A,B) ==> converse(f) O f = id(A)";
by (cut_facts_tac prems 1);
by (rtac equalityI 1);
by (rtac subsetI 1);
by (etac compE 1);
by (etac ssubst 1);
by (eresolve_tac [(make_elim converseD)] 1);
by (eresolve_tac [(inj_equality RS ssubst)] 1);
by (REPEAT (ares_tac (prems@[idI,left_comp_inverse1]) 1
     ORELSE eresolve_tac [injE, Pair_mem_PiE] 1));
val left_comp_inverse = result();

(*half of the equality*)
val prems = goal perm_thy
    "f: A->B ==> f O converse(f) <= id(B)";
by (rtac subsetI 1);
by (etac compE 1);
by (etac ssubst 1);
by (eresolve_tac [(make_elim converseD)] 1);
by (eresolve_tac [(apply_equality2 RS ssubst)] 1);
by (REPEAT (ares_tac (prems@[idI]) 1
     ORELSE etac Pair_mem_PiE 1));
val right_comp_inverse1 = result();

val prems = goal perm_thy
    "f: surj(A,B) ==> f O converse(f) = id(B)";
by (cut_facts_tac prems 1);
by (etac surjE 1);
by (rtac equalityI 1);
by (etac right_comp_inverse1 1);
by (rtac subsetI 1);
by (etac idE 1);
by (etac ssubst 1);
by (REPEAT (eresolve_tac [asm_rl, apply_Pair, bspec RS bexE, subst] 1
     ORELSE resolve_tac [compI,converseI] 1));
val right_comp_inverse = result();

val prems = goal perm_thy
    "[| converse(f): B->A;  f: A->B |] ==> f : bij(A,B)";
by (rewtac bij_def);
by (rtac IntI 1);
by (rtac injI 1);
by (resolve_tac prems 1);
(*apply converse(f) to both sides; simplify using left_inverse_lemma*)
by (eresolve_tac [(subst_context RS box_equals)] 1);
by (REPEAT (ares_tac (prems@[left_inverse_lemma]) 1));
(*surjective case is even simpler...*)
by (REPEAT (ares_tac (prems@[surjI,bexI,right_inverse_lemma,apply_type]) 1));
val invertible_imp_bijective = result();

(** Unions of functions -- cf similar theorems on func.ML **)

goal perm_thy
    "converse(r Un s) = converse(r) Un converse(s)";
by (REPEAT (eresolve_tac [UnE,converseE,ssubst] 1
     ORELSE swap_res_tac [equalityI,subsetI,UnCI,converseI] 1));
val converse_of_Un = result();

val prems = goal perm_thy
    "[| f: surj(A,B);  g: surj(C,D);  A Int C = 0 |] ==> \
\    (f Un g) : surj(A Un C, B Un D)";
by (cut_facts_tac prems 1);
by (REPEAT (eresolve_tac [asm_rl, surjE, bexE, UnE, UnI1, UnI2,
			  fun_disjoint_apply1, fun_disjoint_apply2] 1
     ORELSE ball_tac 1
     ORELSE resolve_tac [surjI, fun_disjoint_Un, bexI] 1
     ORELSE (rtac trans 1 THEN atac 2)));
val surj_disjoint_Un = result();

(*A simple, high-level proof; the version for injections follows from it,
  using  f:inj(A,B)<->f:bij(A,range(f))  *)
val prems = goal perm_thy
    "[| f: bij(A,B);  g: bij(C,D);  A Int C = 0;  B Int D = 0 |] ==> \
\    (f Un g) : bij(A Un C, B Un D)";
br invertible_imp_bijective 1;
br (converse_of_Un RS ssubst) 1;
by (REPEAT (resolve_tac (prems@[fun_disjoint_Un, bij_is_fun, 
				bij_converse_bij]) 1));
val bij_disjoint_Un = result();


(** Restrictions as injections, surjections, and bijections *)

val prems = goal perm_thy
    "[| f: Pi(A,B);  C<=A |] ==> restrict(f,C): surj(C, f``C)";
by (cut_facts_tac prems 1);
by (REPEAT (ares_tac [surjI,restrict_type,imageI,apply_Pair] 1
     ORELSE eresolve_tac [subsetD,imageE] 1));
by (REPEAT (ares_tac [bexI, restrict_conv RS trans] 1
     ORELSE eresolve_tac [apply_equality] 1));
val restrict_surj = result();

val prems = goal perm_thy
    "[| f: inj(A,B);  C<=A |] ==> restrict(f,C): bij(C, f``C)";
by (cut_facts_tac prems 1);
be injE 1;
by (rewtac bij_def);
by (REPEAT (ares_tac [IntI,injI,surj_is_fun,restrict_surj] 1
     ORELSE eresolve_tac [bspec RS bspec RS mp, subsetD, 
			  box_equals, restrict_conv] 1));
val restrict_bij = result();
