(*  Title: 	ZF/ex/list
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1991  University of Cambridge

Lists in Zermelo-Fraenkel Set Theory 

Could type labels be removed by defining is_tail over the elements of l?
Or use V(rank(l))...  
*)

writeln"File ZF/ex/list";

structure List =
struct
val const_decs = 
 [ 
  (["list"],		"i=>i"),
  (["list_case"],	"[i, i, [i,i]=>i] => i"),
  (["pred_list"],	"i => i"),
  (["list_rec"],	"[i, i, i, [i,i,i]=>i] => i")
 ];

val thy = extend_theory Univ.thy "list"
     ([], [], [], const_decs, None)
 [ 
  ("list_def", "list(A) == lfp(lam r: Pow(univ(A)). {0} Un A*r)" ),

  ("list_case_def",	
   "list_case(l,c,d) == THE z. l=0 & z=c | (EX x y. l = <x,y> & z=d(x,y))" ),

  ("pred_list_def",	
   "pred_list(A) == UN l: list(A). UN x: A. {<l, <x,l>>}" ),

  ("list_rec_def",	
   "list_rec(A,l,c,d) == wfrec(trancl(pred_list(A)),  l, \
\                          %l rec. list_case(l, c, %x xs. d(x,xs,rec`xs)))" )
 ];
end;

local val ax = get_axiom List.thy
in 
val list_def = ax"list_def";
val list_case_def = ax"list_case_def";
val pred_list_def = ax"pred_list_def";
val list_rec_def = ax"list_rec_def";
end;


(** Monotonicity and unfolding of the function **)

goal List.thy
    "(lam r: Pow(univ(A)). {0} Un A*r) : mono(Pow(univ(A)), Pow(univ(A)))";
by (rtac lam_mono_Powtype 1);
by (REPEAT (ares_tac [subset_refl, zero_in_univ, singleton_subsetI, 
		      A_subset_univ, Un_mono, product_mono, 
		      Un_least, product_subset_univ] 1));
val listfun_mono = result();

goalw List.thy [list_def] "list(A) = {0} Un (A * list(A))";
by (rtac (listfun_mono RS lam_Tarski_theorem) 1);
val list_unfold = result();

(** Type checking of 0 (as nil) and <-,-> (as cons) **)

goal List.thy "0 : list(A)";
by (rtac (list_unfold RS ssubst) 1);
by (rtac (singletonI RS UnI1) 1);
val list_0I = result();

val prems = goal List.thy "[| a: A;  l: list(A) |] ==> <a,l> : list(A)";
by (rtac (list_unfold RS ssubst) 1);
by (REPEAT (resolve_tac (prems @ [SigmaI RS UnI2]) 1));
val list_PairI = result();

(** Structural induction on lists **)

val prems = goalw List.thy [list_def]
    "[| l: list(A);  \
\       P(0);        \
\       !!x y. [| x: A;  y: list(A);  P(y) |] ==> P(<x,y>) \
\    |] ==> P(l)";
by (rtac lam_general_induction 1);
by (REPEAT (ares_tac (listfun_mono::prems) 1
     ORELSE eresolve_tac [UnE, singletonE, SigmaE, CollectE, ssubst] 1));
val list_induct = result();

(*Perform induction on l, then prove the major premise using prems. *)
fun list_ind_tac a prems i = 
    EVERY [res_inst_tac [("l",a)] list_induct i,
	   rename_last_tac a ["1"] (i+2),
	   ares_tac prems i];

val [prem] = goal List.thy
    "l: list(A) ==> l=0 | (EX x:A. EX xs: list(A). l=<x,xs>)";
by (rtac (prem RS list_induct) 1);
by (DEPTH_SOLVE (ares_tac [refl, disjI1, disjI2, bexI] 1));
val list_cases = result();

(** list_case **)

goalw List.thy [list_case_def] "list_case(0,c,d) = c";
by (fast_tac (ZF_cs addIs [the_equality]) 1);
val list_case_0_conv = result();

goalw List.thy [list_case_def] "list_case(<a,l>, c, d) = d(a,l)";
by (fast_tac (ZF_cs addIs [the_equality]) 1);
val list_case_Pair_conv = result();

val major::prems = goal List.thy
    "[| l: list(A);    \
\       c: C(0);       \
\       !!x y. [| x: A;  y: list(A) |] ==> d(x,y): C(<x,y>)  \
\    |] ==> list_case(l,c,d) : C(l)";
by (rtac (major RS list_induct) 1);
by (rtac (list_case_Pair_conv RS ssubst) 2);
by (rtac (list_case_0_conv RS ssubst) 1);
by (REPEAT (ares_tac prems 1));
val list_case_type = result();


(*** The pred_list relation ***)

goalw List.thy [pred_list_def] "pred_list(A) <= list(A) * list(A)";
by (fast_tac (ZF_cs addIs [list_PairI]) 1);
val pred_list_type = result();

(** Introduction rules for 'pred_list' **)

val prems = goalw List.thy [pred_list_def] 
    "[| l: list(A);  a: A |] ==> <l, <a,l>> : pred_list(A)";
by (REPEAT (ares_tac (prems@[UN_I,singletonI]) 1));
val pred_listI = result();

val major::prems = goalw List.thy [pred_list_def]
    "[| p : pred_list(A);  !!x l. [| x:A;  l:list(A);  p = <l,<x,l>> |] ==> R \
\    |] ==> R";
by (cut_facts_tac [major] 1);
by (REPEAT (eresolve_tac [UN_E,singletonE] 1 ORELSE ares_tac prems 1));
val pred_listE = result();

val major::prems = goal List.thy
    "[| <l',l> : pred_list(A);  !!x. [| x:A;  l = <x,l'> |] ==> R \
\    |] ==> R";
by (rtac (major RS pred_listE) 1);
by (eresolve_tac prems 1);
by (fast_tac ZF_cs 1);
val pred_listE2 = result();

goal List.thy "wf(pred_list(A))";
by (rtac (pred_list_type RS field_rel_subset RS wfI2) 1);
by (rtac subsetI 1);
by (etac list_induct 1);
by (fast_tac (ZF_cs addIs [list_0I,list_PairI]
	            addSEs [bspec RS mp, pred_listE2]) 2);
by (fast_tac (ZF_cs addIs [list_0I,list_PairI]
	            addSEs [bspec RS mp, pred_listE2]) 1);
val wf_pred_list = result();


(*alternative proof using wfI is a bit harder...*)
choplev 0;
by (rtac (pred_list_type RS field_rel_subset RS wfI) 1);
by (res_inst_tac [("P","u:Z")] mp 1);
by (eres_inst_tac [("l","u")] (subsetD RS list_induct) 1);
by (assume_tac 1);
by (assume_tac 3);
by (fast_tac (ZF_cs addIs [list_PairI] addSEs [pred_listE2]) 2);
by (fast_tac (ZF_cs addIs [list_PairI] 
		    addSEs [sym RS Pair_neq_0, pred_listE2]) 1);
result();

(*** list_rec -- by wf recursion on pred_list ***)

(*Used just to verify list_rec*)
val list_rec_ss = ZF_ss 
      addcongs (mk_typed_congs List.thy [("h", "[i,i,i]=>i")])
      addrews [trans_trancl, wf_pred_list RS wf_trancl, 
	       list_rec_def RS wfrec_def_conv,
	       list_case_0_conv,list_case_Pair_conv];

(** conversion rules **)

goal List.thy "list_rec(A,0,c,h) = c";
by (SIMP_TAC list_rec_ss 1);
val list_rec_0_conv = result();

val prems = goal List.thy
    "[| a: A;  l: list(A) |] ==> \
\   list_rec(A, <a,l>, c, h) = h(a, l, list_rec(A,l,c,h))";
by (SIMP_TAC (list_rec_ss addrews 
		([under_iff,r_sub_trancl,pred_listI]@prems)) 1);
val list_rec_Pair_conv = result();

val list_ss = ZF_ss addrews
  [list_case_0_conv,list_case_Pair_conv,
   list_rec_0_conv,list_rec_Pair_conv,
   list_0I, list_PairI];

(*Type checking.  This proof is vastly simpler than using wfrec_type*)
val prems = goal List.thy
     "[| l: list(A);    \
\        c: C(0);       \
\        !!x y r. [| x: A;  y: list(A);  r: C(y) |] ==> h(x,y,r): C(<x,y>)  \
\     |] ==> list_rec(A,l,c,h) : C(l)";
by (list_ind_tac "l" prems 1);
by (SIMP_TAC (list_ss addrews prems) 1);
by (SIMP_TAC (list_ss addrews prems) 1);
val list_rec_type = result();

