(*  Title: 	HOL/ex/arith
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Examples about elementary arithmetic: addition, multiplication, etc.
Tests definitions and simplifier.
*)

fun typechk_tac asms =
  let val tac = compat_resolve_tac (asms @ all_type_rls) 3
  in  REPEAT_SOME (rigid_assume_tac  ORELSE'  tac)  end;


(** Addition *)

val add_type = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> a #+ b : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (typechk_tac asms) ]);


(*computation for add: 0 and Successor cases*)

val add_conv0 = prove_goal arith_thy 
    "b:nat ==> 0 #+ b = b : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac asms) ]);

val add_conv_Succ = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> Succ(a) #+ b = Succ(a #+ b) : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac asms) ]); 


(** Multiplication *)

val mult_type = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> a #* b : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (typechk_tac([add_type]@asms))  ]);


(*computation for mult: 0 and Successor cases*)

val mult_conv0 = prove_goal arith_thy 
    "b:nat ==> 0 #* b = 0 : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac asms)  ]);


val mult_conv_Succ = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> Succ(a) #* b = b #+ (a #* b) : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac asms)  ]);


(** Difference *)

val diff_type = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> a - b : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (typechk_tac asms) ]);



(*computation for difference: 0 and Successor cases*)

val diff_conv0 = prove_goal arith_thy 
    "a:nat ==> a - 0 = a : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac asms) ]);


(*Note: rec(a, 0, %z w.z) is pred(a). *)

val diff_0_eq_0 = prove_goal arith_thy 
    "b:nat ==> 0 - b = 0 : nat"
 (fn asms=>
  [ (nat_induct_tac "b" 1),
    (rewrite_goals_tac arith_defs),
    (hyp_rew_tac asms) ]);


(*Essential to simplify FIRST!!  (Else we get a critical pair)
  Succ(a) - Succ(b) rewrites to   pred(Succ(a) - b)  *)
val diff_Succ_Succ = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> Succ(a) - Succ(b) = a - b : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (hyp_rew_tac asms),
    (nat_induct_tac "b" 1),
    (hyp_rew_tac asms) ]);



(*** Simplification *)

val arith_type_rls = [add_type, mult_type, diff_type];

val arith_congr_rls = make_congr_rules arith_type_rls;

val all_congr_rls = arith_congr_rls@congr_rls;

val arith_conv_rls =
  [add_conv0, add_conv_Succ,
   mult_conv0, mult_conv_Succ,
   diff_conv0, diff_0_eq_0, diff_Succ_Succ];


print_depth 1;

structure Arith_simp_data: SIMP_DATA =
struct
  structure Pure = Pure
  val refl= refl
  val sym= sym
  val trans= trans
  val refl_red= refl_red
  val trans_red= trans_red
  val red_if_equal= red_if_equal
  val default_rls = arith_conv_rls @ conv_rls
  val routine_tac = routine_tac (arith_type_rls @ all_type_rls)
end;

structure Arith_simp = SimpFun(Arith_simp_data);

fun arith_rew_tac asms = make_rew_tac
    (Arith_simp.norm_tac(all_congr_rls, asms));

fun hyp_arith_rew_tac asms = make_rew_tac
    (Arith_simp.cond_norm_tac(prove_cond_tac, all_congr_rls, asms));

print_depth 8;



(**********
  Addition
 **********)


(*Associative law for addition*)
val add_assoc = prove_goal arith_thy 
    "[| a:nat;  b:nat;  c:nat |] ==> (a #+ b) #+ c = a #+ (b #+ c) : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
    (hyp_arith_rew_tac asms)  (*7 secs*)  ]);



(*Commutative law for addition.  Can be proved using three inductions.
  Must simplify after first induction!!  Orientation of rewrites is delicate*)  
val add_commute = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> a #+ b = b #+ a : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
    (hyp_arith_rew_tac asms),
    (nat_induct_tac "b" 2),
    (resolve_tac [sym] 1),
    (nat_induct_tac "b" 1),
    (hyp_arith_rew_tac asms)   ]);



(****************
  Multiplication
 ****************)


(*Commutative law for multiplication
val mult_commute = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> a #* b = b #* a : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
    (hyp_arith_rew_tac asms),
    (nat_induct_tac "b" 2),
    (resolve_tac [sym] 1),
    (nat_induct_tac "b" 1),
    (hyp_arith_rew_tac asms)   ]);   NEEDS COMMUTATIVE MATCHING
***************)

(*right annihilation in product*)
val mult_right0 = prove_goal arith_thy 
    "a:nat ==> a #* 0 = 0 : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
    (hyp_arith_rew_tac asms)  ]);



(*right Successor law for multiplication*)
val mult_right_Succ = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> a #* Succ(b) = a #+ (a #* b) : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
(*swap round the associative law of addition*)
    (hyp_arith_rew_tac (asms @ [ sym RES add_assoc ])),  
(*8 secs, leaves a goal involving a commutative law*)
    (REPEAT (assume_tac 1  ORELSE  
            resolve_tac
             (asms@[add_commute]@
	       all_congr_rls@[refl])   1))  ]);


(*Commutative law for multiplication*)
val mult_commute = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> a #* b = b #* a : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
    (hyp_arith_rew_tac (asms @ [mult_right0, mult_right_Succ])) ]);


(*addition distributes over multiplication*)
val add_mult_dist = prove_goal arith_thy 
    "[| a:nat;  b:nat;  c:nat |] ==> (a #+ b) #* c = (a #* c) #+ (b #* c) : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
(*swap round the associative law of addition*)
    (hyp_arith_rew_tac (asms @ [ sym RES add_assoc ])) (*11 secs*) ]);



(*Associative law for multiplication*)
val mult_assoc = prove_goal arith_thy 
    "[| a:nat;  b:nat;  c:nat |] ==> (a #* b) #* c = a #* (b #* c) : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
    (hyp_arith_rew_tac (asms @ [add_mult_dist]))  (*7 secs*) ]);



(************
  Difference
 ************

Difference on natural numbers, without negative numbers
  a - b = 0  iff  a<=b    a - b = Succ(c) iff a>b
*)


val diff_self_eq_0 = prove_goal arith_thy 
    "a:nat ==> a - a = 0 : nat"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
    (hyp_arith_rew_tac asms) ]);


val add_right0 = trans RES add_commute RESN (3, [add_conv0]);


(*Addition is the inverse of subtraction: if b<=x then b#+(x-b) = x.
  An example of induction over a quantified formula. *)

(*Permits rewriting with a quantified, implicative inductive hypothesis.*)
val prove_cond2_tac = eresolve_tac [asm_rl, spec, spec RS mp];

val asms =
goal arith_thy "b:nat ==> ALL x:nat.(b-x = 0 : nat)-->(b #+ (x-b) = x : nat)";
by (nat_induct_tac "b" 1);
by (resolve_tac [all_intr] 3);  
(*case analysis on x in   (Succ(u) <= x) --> (Succ(u)#+(x-Succ(u)) = x) *)
by (nat_induct_tac "kb" 3 THEN assume_tac 3); 

(*Prepare for simplification of formulae -- the antecedent Succ(u)<=x *)
(****by (resolve_tac [replace_iff] 4);
by (resolve_tac [replace_iff] 3);
***)

by (REPEAT (resolve_tac [all_intr,imp_intr] 2));
by (arith_rew_tac asms); 
(*10 secs: solves first 0 goal, simplifies others.  Two sugbgoals remain.
  Both follow by rewriting, (2) using quantified induction hyp*)
by (REPEAT_SOME (resolve_tac [imp_intr]));  
by (make_rew_tac
    (Arith_simp.cond_norm_tac(prove_cond2_tac, all_congr_rls,
	 asms@[add_right0])));  (*8 secs*)
val add_inverse_diff_lemma = result();



(*Version of above with premise   b-a=0   i.e.    a >= b.
  Using Prod_elim does not work -- for ?B(?a) is ambiguous.
  Instead, add_inverse_diff_lemma states the desired induction scheme;
    the use of RES below instantiates Vars in Prod_elim automatically. *)
val asms =
goal arith_thy "[| a:nat;  b:nat;  b-a = 0 : nat |] ==> b #+ (a-b) = a : nat";
by (resolve_tac [add_inverse_diff_lemma RS spec RS mp] 1);
by (REPEAT (resolve_tac asms 1));
val add_inverse_diff = result();



(********************
  Absolute difference
 ********************)


(*typing of absolute difference: short and long versions*)

val absdiff_type = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> a |-| b : nat"
 (fn asms=>
  [ (rewrite_goals_tac arith_defs),
    (typechk_tac asms) ]);


val absdiff_self_eq_0 = prove_goal arith_thy 
    "a:nat ==> a |-| a = 0 : nat"
 (fn asms=>
  [ (rewrite_goals_tac [absdiff_def]),
    (arith_rew_tac (asms@[diff_self_eq_0])) ]);


val absdiff_conv0 = prove_goal arith_thy 
    "a:nat ==> 0 |-| a = a : nat"
 (fn asms=>
  [ (rewrite_goals_tac [absdiff_def]),
    (hyp_arith_rew_tac asms) ]);


val absdiff_Succ_Succ = prove_goal arith_thy 
    "[| a:nat;  b:nat |] ==> Succ(a) |-| Succ(b) = a |-| b : nat"
 (fn asms=>
  [ (rewrite_goals_tac [absdiff_def]),
    (hyp_arith_rew_tac asms) ]);


(*Note how easy using commutative laws can be?  ...not always... *)
val asms =
goal arith_thy "[| a:nat;  b:nat |] ==> a |-| b = b |-| a : nat";
by (rewrite_goals_tac [absdiff_def]);
by (resolve_tac [add_commute] 1);
by (typechk_tac ([diff_type]@asms));
val absdiff_commute = result();


(*If a+b=0 then a=0.   Surprisingly tedious*)
val asms =
goal arith_thy "[| a:nat;  b:nat |] ==> (a#+b = 0 : nat) --> (a = 0 : nat)";
by (nat_induct_tac "a" 1);
by (arith_rew_tac asms);
by (REPEAT_SOME (resolve_tac [refl, imp_intr]));  
by (eresolve_tac [sym RS nat_distinct] 2);
by (typechk_tac ([add_type] @asms));
val add_eq0_lemma = result();



(*Version of above with the premise  a+b=0.
  Again, resolution instantiates variables in Prod_elim *)
val asms =
goal arith_thy "[| a:nat;  b:nat;  a #+ b = 0 : nat |] ==> a = 0 : nat";
by (resolve_tac [add_eq0_lemma RS mp] 1);
by (ALLGOALS (resolve_tac asms));
val add_eq0 = result();



(*Here is a lemma to infer a-b=0 and b-a=0 from a|-|b=0, below. *)
val asms =
goal arith_thy
 "[| a:nat;  b:nat;  a|-|b = 0 : nat |] ==> (a-b = 0 : nat) & (b-a = 0 : nat)";
by (resolve_tac [conj_intr] 1);
by (resolve_tac [add_eq0] 2);
by (resolve_tac [add_eq0] 1);
by (resolve_tac [add_commute RS trans] 6);
by (typechk_tac (diff_type:: map (rewrite_rule [absdiff_def]) asms));
val absdiff_eq0_lem = result();


(*if  a |-| b = 0  then  a = b  
  proof: a-b=0 and b-a=0, so b = a+(b-a) = a+0 = a*)
val asms =
goal arith_thy "[| a|-|b = 0 : nat;  a:nat;  b:nat |] ==> a = b : nat";
by (resolve_tac [absdiff_eq0_lem RS conj_elim] 1);
by (REPEAT_SOME (resolve_tac asms));
by (resolve_tac [trans RES sym RES add_inverse_diff ] 1);
by (assume_tac 3);
by (hyp_arith_rew_tac (asms@[add_right0]));
val absdiff_eq0 = result();



(***********************
  Remainder and Quotient
 ***********************)


(*remainder (//) *)

val mod_type = prove_goal arith_thy
    "[| a:nat;  b:nat |] ==> a//b : nat"
 (fn asms=>
  [ (rewrite_goals_tac [mod_def]),
    (typechk_tac (absdiff_type::asms)) ]);
 

(*computation for //: 0 and Successor cases*)

val mod_conv0 = prove_goal arith_thy "b:nat ==> 0//b = 0 : nat"
 (fn asms=>
  [ (rewrite_goals_tac [mod_def]),
    (rew_tac(absdiff_type::asms)) ]);


val mod_conv_Succ = prove_goal arith_thy "[| a:nat;  b:nat |] ==> \
\    Succ(a)//b = \
\    rec(Succ(a//b) |-| b, 0, %x y.Succ(a//b)) : nat"
 (fn asms=>
  [ (rewrite_goals_tac [mod_def]),
    (rew_tac(absdiff_type::asms)) ]);


val quo_type = prove_goal arith_thy
    "[| a:nat;  b:nat |] ==> a / b : nat"
 (fn asms=>
  [ (rewrite_goals_tac [quo_def]),
    (typechk_tac ([absdiff_type,mod_type]@asms)) ]);


val quo_type_rls = [mod_type, quo_type, absdiff_type];


(*computation for quotient: 0 and Successor cases*)

val quo_conv0 = prove_goal arith_thy "b:nat ==> 0 / b = 0 : nat"
 (fn asms=>
  [ (rewrite_goals_tac [quo_def]),
    (rew_tac([mod_type, absdiff_type] @ asms)) ]);


val quo_conv_Succ =
prove_goal arith_thy "[| a:nat;  b:nat |] ==> \
\    Succ(a) / b = rec(Succ(a)//b, Succ(a / b), %x y. a / b) : nat"
 (fn asms=>
  [ (rewrite_goals_tac [quo_def]),
    (rew_tac([mod_type]@asms)) ]);


(*Version of above with same condition as the // one*)
val quo_conv_Succ2 = prove_goal arith_thy
    "[| a:nat;  b:nat |] ==> Succ(a) / b = \
\	  rec(Succ(a//b) |-| b, Succ(a / b), %x y. a / b) : nat"
 (fn asms=>
  [ (resolve_tac [ trans RES quo_conv_Succ ] 1),
    (rew_tac(quo_type_rls @ asms @ [mod_conv_Succ])),
    (nat_induct_tac "Succ(a//b)|-|b" 1),
    (rew_tac ([mod_type, quo_type, absdiff_type] @asms)) ]);


(*for case analysis on whether a number is 0 or a Successor*)
val nat_cases = prove_goal arith_thy
    "a:nat ==> (a=0 : nat) | (EX x:nat. a = Succ(x) : nat)"
 (fn asms=>
  [ (nat_induct_tac "a" 1),
    (REPEAT (resolve_tac [refl] 1  ORELSE 
	     Pc.step_tac (asms@[Zero_type,Succ_type]) 1)) ]);



(*Main Result.  Holds when b is 0 since   a//0 = a     and    a/0 = 0  *)
val asms =
goal arith_thy "[| a:nat;  b:nat |] ==> a//b  #+  (a/b) #* b = a : nat";
by (nat_induct_tac "a" 1);
by (arith_rew_tac (quo_type_rls @ asms @
	[mod_conv0,mod_conv_Succ, quo_conv0, quo_conv_Succ2])); 
(*5 secs*)
(*case analysis on   Succ(u//b)|-|b  *)
by (resolve_tac [disj_elim] 1);
by (res_inst_tac [("a", "Succ(ka // b) |-| b", Aterm)] nat_cases 1);
by (eresolve_tac [exists_elim] 3);
by (hyp_arith_rew_tac (asms @ quo_type_rls @
	[mod_conv0,mod_conv_Succ, quo_conv0, quo_conv_Succ2])); 
(*36 secs*)
(*Replace one occurence of  b  by Succ(ka//b).  Clumsy!*)
by (resolve_tac [trans] 1);
by (resolve_tac arith_congr_rls 1);
by (eresolve_tac [absdiff_eq0 RS sym] 1);
by (resolve_tac [refl] 3);
by (hyp_arith_rew_tac (asms @ quo_type_rls)); 
(*4 secs*)
val mod_quo_equality = result();


writeln"Reached end of file.";
