(*  Title: 	FOL/cla-lemmas
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Classical First-Order Logic
*)

signature CLA_LEMMAS = 
sig
  structure Int_Prover: INT_PROVER (*needed by ProverFun*)
  local open Int_Prover.Int_Rule.Pure
  in
  val chain_tac : int -> tactic
  val cla_thy : theory
  val classical : thm
  val contr_tac : int -> tactic
  val disj_cintr : thm
  val ex_middle : thm
  val exists_cintr : thm
  val iff_celim : thm
  val imp_celim : thm
  val joinrules : thm list * thm list -> (bool * thm) list
  val size_of_thm : thm -> int
  val swap : thm
end;
end;


functor Cla_LemmasFun(Int_Prover: INT_PROVER) : CLA_LEMMAS = 
struct
structure Int_Prover = Int_Prover;
structure Int_Rule = Int_Prover.Int_Rule;

local open  Int_Prover  Int_Rule  Int_Rule.Pure  Int_Rule.FOL_Syntax
in
val cla_thy = extend_theory Int_Rule.thy  "Classical" ([], [])
  [ ("classical",  "(~P ==> P) ==> P") ];

val classical = get_axiom cla_thy "classical";


(** Classical intro rules for | and EX *)

val disj_cintr = prove_goal cla_thy 
   "(~Q ==> P) ==> P|Q"
 (fn prems=>
  [ (resolve_tac [classical] 1),
    (REPEAT (ares_tac (prems@[disj_intr1,not_intr]) 1)),
    (REPEAT (ares_tac (prems@[disj_intr2,contr]) 1)) ]);


val exists_cintr = prove_goal cla_thy 
   "(ALL x. ~P(x) ==> P(a)) ==> EX x.P(x)"
 (fn prems=>
  [ (resolve_tac [classical] 1),
    (REPEAT (ares_tac (prems@[exists_intr,all_intr,not_intr,contr]) 1))  ]);


val ex_middle = prove_goal cla_thy "~P | P"
 (fn _=> [ (REPEAT (ares_tac [disj_cintr] 1)) ]);


(*** Special elimination rules *)


(*Classical implies (-->) elimination. *)
val imp_celim = prove_goal cla_thy 
    "[| P-->Q; ~P ==> R; Q ==> R |] ==> R"
 (fn prems=>
  [ (resolve_tac [disj_elim RES ex_middle] 1),
    (REPEAT (DEPTH_SOLVE_1 (ares_tac (prems @ [ mp RESN (1,prems) ]) 1))) ]);


(*** Tactics for implication and contradiction ***)

(*Solve goal that assumes both P and ~P 
  Does not use imp_elim -- would be redundant in complete prover *)
val contr_tac = eresolve_tac [not_elim]  THEN'  assume_tac;

(*Given assumption P-->Q, reduces subgoal Q to P [deletes the implication!] *)
fun chain_tac i =
    eresolve_tac [imp_elim] i  THEN  
    (assume_tac (i+1)  ORELSE  contr_tac (i+1));


(*Classical <-> elimination.  Proof substitutes P=Q in 
    ~P ==> ~Q    and    P ==> Q  *)
val iff_celim = prove_goal cla_thy 
    "[| P<->Q;  [| P; Q |] ==> R;  [| ~P; ~Q |] ==> R |] ==> R"
 (fn prems =>
  [ (resolve_tac [conj_elim] 1),
    (REPEAT (DEPTH_SOLVE_1 
	(eresolve_tac [imp_celim] 1  ORELSE 
	 mp_tac 1  ORELSE 
	 ares_tac (map (rewrite_rule [iff_def]) prems) 1))) ]);


(*Should be used as swap since ~P becomes redundant*)
val swap = prove_goal cla_thy 
   "~P ==> (~Q ==> P) ==> Q"
 (fn prems=>
  [ (resolve_tac [classical] 1),
    (resolve_tac [prems RSN (1, contr RS False_elim)] 1),
    (*Looks strange but it works!*)
    (REPEAT (ares_tac prems 1)) ]);



(*** Operations used in theorem provers 
     Better one copy here than multiple copies in Prover *)

val size_of_thm = size_of_term o #prop o rep_thm;

(*For use with biresolve_tac.  Combines intrs with swap to catch negated
  assumptions.  Also pairs elims with true. *)
fun joinrules (intrs,elims) =  
  (map (pair true) (elims @ reslist(intrs,2,swap))) @
  (map (pair false) intrs);

end;
end;

