(*  Title: 	tactic
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Tactics 
*)

signature TACTIC =
sig
  structure Tactical: TACTICAL and Stringtree: STRINGTREE
  local open Tactical Tactical.Thm
  in
    val ares_tac: thm list -> int -> tactic   
    val assume_tac: int -> tactic   
    val biresolve_tac: (bool*thm)list -> int -> tactic
    val compat_resolve_tac: thm list -> int -> int -> tactic   
    val compat_thms: 'a Stringtree.tree * term -> 'a list   
    val compose_inst_tac: (string*string*typ)list -> (bool*thm*int) -> int -> tactic   
    val compose_tac: (bool * thm * int) -> int -> tactic 
    val cut_facts_tac: thm list -> int -> tactic
    val eq_assume_tac: int -> tactic   
    val eresolve_tac: thm list -> int -> tactic
    val eres_inst_tac: (string*string*typ)list -> thm -> int -> tactic   
    val filter_thms: (term*term->bool) -> int*term*thm list -> thm list
    val filt_resolve_tac: thm list -> int -> int -> tactic
    val forwards_tac: thm -> int -> tactic   
    val head_string: term -> string list
    val insert_thm: thm * thm Stringtree.tree -> thm Stringtree.tree
    val lessb: (bool * thm) * (bool * thm) -> bool
    val lift_inst_rule: thm * int * (string*string*typ)list * thm -> thm
    val lift_inst_tac: (string*string*typ)list -> thm -> int -> tactic   
    val metacut_tac: thm -> int -> tactic   
    val PRIMITIVE: (thm -> thm) -> tactic  
    val resolve_tac: thm list -> int -> tactic
    val res_inst_tac: (string*string*typ)list -> thm -> int -> tactic   
    val smash_all_ff_tac: tactic
    val subgoals_of_brl: bool * thm -> int
    val subgoal_tac: string -> int -> tactic
    val trace_tac: int -> tactic
  end
end;


functor TacticFun (structure Logic: LOGIC and Tactical: TACTICAL and
	           Stringtree: STRINGTREE) : TACTIC = 
struct
structure Tactical = Tactical;
structure Thm = Tactical.Thm;
structure Stringtree = Stringtree;
structure Sequence = Thm.Sequence;
structure Sign = Thm.Sign;
local open Tactical Tactical.Thm
in

(*Discover what goal is chosen:  SOMEGOAL(tac THEN' trace_tac) *)
fun trace_tac i = Tactic (fn state => 
   (prs("Subgoal " ^ string_of_int i ^ " selected\n"); 
    Sequence.single state));


(*** Basic tactics ***)


(*Makes a tactic whose action on a state is the mapping thmfun.*)
fun PRIMITIVE thmfun = Tactic (fn state => Sequence.single (thmfun state));


(*** The following fail if the goal number is out of range:
     thus (REPEAT (resolve_tac rules i)) stops once subgoal i disappears. *)


(*The composition rule/state: no lifting or var renaming.
  The arg = (bires_flg, orule, m) ;  see bicompose for explanation.*)
fun compose_tac arg i = Tactic (fn state =>
    bicompose(state, i)arg  handle THM _ => Sequence.null);

(*Solve subgoal i by assumption*)
fun assume_tac i = Tactic (fn state => assumption(state,i)
	handle THM _ => Sequence.null);

(*Solve subgoal i by assumption, using no unification*)
fun eq_assume_tac i = Tactic
    (fn state => Sequence.single (eq_assumption(state,i))
	handle THM _ => Sequence.null);

(*Resolution: the simple case, works for introduction rules*)
fun resolve_tac rules i = Tactic (fn state => resolution(state,i,rules)
	handle THM _ => Sequence.null);

(*Attack subgoal i by resolution, using flags to indicate elimination rules*)
fun biresolve_tac brules i = Tactic (fn state=> biresolution(state,i,brules)
	handle THM _ => Sequence.null);

(*Resolution with elimination rules only*)
fun eresolve_tac rules = biresolve_tac (map (pair true) rules);


(*Lift and instantiate a rule wrt the given state and subgoal number *)
fun lift_inst_rule (state, i, sinsts, rule) =
  let val {maxidx,sign,...} = rep_thm state;
      fun cfun f = (Sign.cterm_of sign) o f o Sign.term_of;
      val (_, _, Bi, _) = dest_state(state,i);
      val params = rev (#2 (Logic.strip_assums Bi));   (*params of subgoal i*)
      fun liftvar (Var ((a,j), T)) = Var((a, j+maxidx+1), map snd params--->T)
	| liftvar t = raise TERM_ERROR("lift_inst: Var expected", [t]);
      fun liftterm t = list_abs_free (params, t);
      (*Lifts instantiation pair over params*)
      fun liftpair (cv,ct) = (cfun liftvar cv, cfun liftterm ct);
      val insts = Sign.read_insts (#sign(rep_thm state)) sinsts
  in  instantiate (map liftpair insts)  (lift_rule (state,i) rule)
  end;


(*** Resolve after lifting and instantation; may refer to parameters of the
     subgoal.  Fails if "i" is out of range.  WARNING: Variables in the given
     terms will be incremented!  CAN THIS BE PREVENTED?? ***)

(*compose version: arguments are as for bicompose.*)
fun compose_inst_tac sinsts (bires_flg, rule, nsubgoal) i =
  STATE (fn state => 
    compose_tac (bires_flg, lift_inst_rule (state, i, sinsts, rule),
		 nsubgoal) i
    handle TERM_ERROR _ => no_tac
   	 | THM _ => no_tac);

(*Resolve version*)
fun res_inst_tac sinsts rule i =
    compose_inst_tac sinsts (false, rule, length(prems_of rule)) i;

(*eresolve (elimination) version*)
fun eres_inst_tac sinsts rule i =
    compose_inst_tac sinsts (true, rule, length(prems_of rule)) i;

(*older name: for compatibility*)
val lift_inst_tac = res_inst_tac;



(*Used by forwards_tac and metacut_tac*)
fun bires_cut_tac arg i =
    resolve_tac [cut_rl] i  THEN  biresolve_tac arg (i+1) ;

(*Forwards reasoning. 
  Rule has the form A==>B, infers forwards from assumption A.*)
fun forwards_tac rule = bires_cut_tac [(true,rule)];

(*The conclusion of the rule gets assumed in subgoal i,
  while subgoal i+1,... are the premises of the rule.*)
fun metacut_tac rule = bires_cut_tac [(false,rule)];

(*Recognizes theorems that are not rules, but simple propositions*)
fun is_fact th =
    case prems_of th of
	[] => true  |  _::_ => false;

(*"Cut" all facts from theorem list into the goal as assumptions. *)
fun cut_facts_tac ths i =
    MAP_EVERY (fn th => metacut_tac th i) (filter is_fact ths);

(*Introduce the given proposition as a lemma and subgoal*)
fun subgoal_tac sprop = res_inst_tac [("psi", sprop, Aprop)] cut_rl;


(*Use an assumption or some rules ... A popular combination!*)
fun ares_tac rules = assume_tac  ORELSE'  resolve_tac rules;


(*Smash all flex-flex disagreement pairs*)
val smash_all_ff_tac =
    Tactic(fn state => unify_instantiate (tpairs_of state) state);


(*** Indexing and filtering of theorems ***)

(*string of symbols following first argument of combinations
  symbols are constants, parameters, bound vars
  example:  rec(succ(x),...,...)  has   [ "rec", "succ", "*param*" ]  *)
fun head_string t : string list =
  let val (head,args) = strip_comb t;
      fun tail_string [] = []
	| tail_string(arg::_) = head_string arg
  in  case head of
      Const(name,_) =>     name :: tail_string args
    | Free (name,_) => "*"^name :: tail_string args
    | Bound bno => "B."^chr(bno+ord"0") :: tail_string args
    | _ => []
  end;


(*insert a thm in a thm tree using its head_string*)
fun insert_thm (th,rtr) =
  Stringtree.insert((head_string (concl_of th), th), rtr);

(*returns the list of theorems in the tree that are compatible with prem*)
fun compat_thms (rtr,prem) =
   Stringtree.compat(rtr, head_string (Logic.strip_assums_concl prem));


(*Resolve subgoal i using the tree of rules, unless too flexible --
   which means: >maxr are compatible.      *)
fun rtr_resolve_tac rtr maxr = SUBGOAL(fn (prem,i) =>
    let val rls = compat_thms (rtr,prem)
    in  if length rls > maxr  then no_tac  else resolve_tac rls i
    end);

(*For efficiency, bind its result to a tactic.  It builds a stringtree.*)
fun compat_resolve_tac rules = 
  rtr_resolve_tac (itlist_right insert_thm (rules, Stringtree.null));
 

(*Returns the list of potentially resolvable theorems for the goal "prem",
	using the predicate  could(subgoal,concl).
  Resulting list is no longer than "limit"*)
fun filter_thms could (limit, prem, ths) =
  let val pb = Logic.strip_assums_concl prem;   (*delete assumptions*)
      fun filtr (limit, []) = []
	| filtr (limit, th::ths) =
	    if limit=0 then  []
	    else if could(pb, concl_of th)  then th :: filtr(limit-1, ths)
	    else filtr(limit,ths)
  in  filtr(limit,ths)  end;


(*Resolve subgoal i using the rules, unless >maxr are compatible. *)
fun filt_resolve_tac rules maxr = SUBGOAL(fn (prem,i) =>
  let val rls = filter_thms could_unify (maxr+1, prem, rules)
  in  if length rls > maxr  then  no_tac  else resolve_tac rls i
  end);


(*** For Natural Deduction using (bires_flg, rule) pairs ***)

(*The number of new subgoals produced by the brule*)
fun subgoals_of_brl (true,rule) = length (prems_of rule) - 1
  | subgoals_of_brl (false,rule) = length (prems_of rule);

(*Less-than test: for sorting to minimize number of new subgoals*)
fun lessb (brl1,brl2) = subgoals_of_brl brl1 < subgoals_of_brl brl2;

end;
end;

