(*  Title: 	FOL/cla-prover
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Functor for constructing theorem provers from sets of rules

Rules must be classified as intr, elim, safe, hazardous, ...
*)

signature RULE_TAB =
sig
  structure Pure: PURE
  val pairs: (Pure.thm * string) list
end;

signature PROVER =
sig
  structure Pure: PURE
  local open Pure
  in
  val best_tac : thm list -> tactic
  val comp_step_tac : thm list -> tactic
  val comp_tac : thm list -> tactic
  val determ_prop_tac : int -> tactic
  val expand_tac : int -> tactic
  val fast_tac : thm list -> int -> tactic
  val getrules : string -> thm list
  val haz_brls : (bool * thm) list
  val haz_cintr_rls : thm list
  val haz_elim_rls : thm list
  val haz_intr_rls : thm list
  val onestep_tac : int -> tactic
  val safe0_brls : (bool * thm) list
  val safep_brls : (bool * thm) list
  val safe_brls : (bool * thm) list
  val safe_elim_rls : thm list
  val safe_intr_rls : thm list
  val step_tac : thm list -> int -> tactic
end;
end;


functor ProverFun (structure Cla_Lemmas: CLA_LEMMAS and Rule_tab: RULE_TAB
	sharing Cla_Lemmas.Int_Prover.Int_Rule.Pure = Rule_tab.Pure)
    : PROVER = 
struct
structure Int_Prover = Cla_Lemmas.Int_Prover;
structure Int_Rule = Int_Prover.Int_Rule;
structure Pure = Int_Rule.Pure;

local open Pure Int_Rule Int_Prover Cla_Lemmas 
in
	
val getrules = keyfilter Rule_tab.pairs;

(*Safe rules*)
val safe_intr_rls = getrules "safe_intr";

val safe_elim_rls = getrules "safe_elim";


(*Hazardous (unsafe) rules: require delay or lead to infinite regress*)
val haz_intr_rls = getrules "haz_intr";
val haz_cintr_rls = getrules "haz_cintr";
val haz_elim_rls = getrules "haz_elim";


val safe_brls = sort lessb 
   ((true,asm_rl) :: joinrules(safe_intr_rls, safe_elim_rls));

(*Note that all_elim precedes exists_intr, like in LK ????? *)
val haz_brls = sort lessb (joinrules(haz_intr_rls, haz_elim_rls));

(*0 subgoals vs 1 or more*)
val (safe0_brls, safep_brls) =
    partition (apl(0,op=) o subgoals_of_brl) safe_brls;



(** Complete (?) but slow prover *)

(*Either simple step (which might solve goal completely!) or a hazardous rule. 
  Should expand all quantifiers, not just pick one!  Redundant alternatives! *)
val expand_tac = 
    biresolve_tac safe0_brls  APPEND'  contr_tac  APPEND'  
    (resolve_tac haz_elim_rls  THEN'  assume_tac)  APPEND'  
    biresolve_tac (joinrules(haz_cintr_rls,[]));

(*NOT COMPLETE BECAUSE contr_tac MAY INSTANTIATE VARIABLES****
  ***NEED A RIGID VERSION OF contr_tac TO USE IN expand_tac ***
  ***GENERAL VERSION TO BE CALLED IN comp_step_tac ***)


(*Attack subgoal i using propositional reasoning*)
val determ_prop_tac = DETERM o REPEAT1 o
	(eq_assume_tac ORELSE' eq_mp_tac ORELSE' biresolve_tac safep_brls);


(*Single step for the "complete" prover.*)
fun comp_step_tac thms =
    REPEAT (FIRSTGOAL determ_prop_tac)   THEN
    COND (has_fewer_prems 1) all_tac
	 (FIRSTGOAL expand_tac  ORELSE
	  FIRSTGOAL (assume_tac  APPEND'  resolve_tac thms));


(*Resolution with the safe rules is presumed deterministic,
  except for the axioms (like reflexivity and assumption!!) *)
val onestep_tac =
    biresolve_tac safe0_brls  ORELSE'  mp_tac  ORELSE' 
    (DETERM o biresolve_tac safep_brls);


(*Single step for the incomplete prover. 
  FAILS unless it makes reasonable progress.
  Can use thms as introduction rules (via joinrules) *)
fun step_tac thms = 
    let val btac = biresolve_tac (joinrules(thms, []) @ haz_brls);
	fun tac i = REPEAT1 (onestep_tac i)  ORELSE  btac i;
    in  tac  end;


(*Incomplete but fast.  Fails unless it solves one goal!
  Can use thms as introduction rules *)
fun fast_tac thms i = DEPTH_SOLVE_1 (step_tac thms i);

(*Incomplete but fast.  Fails unless it solves ALL goals!
  Can use thms as introduction rules *)
fun best_tac thms = BEST_FIRST (has_fewer_prems 1, size_of_thm) 
	(step_tac thms 1  ORELSE  assume_tac 1);


(*[Less in] complete for first order logic. Fails unless it solves ALL goals!*)
fun comp_tac thms = 
    BEST_FIRST (has_fewer_prems 1, size_of_thm) (comp_step_tac thms);

end; 
end;
