(*  Title: 	thm
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

  the abstract types "theory" and "thm"

  need inference rule for == introduction (A==>B and B==>A imply A==B) ??
  is unify_instantiate secure??
*)

signature THM = 
sig
  structure Envir : ENVIR
  structure Sequence : SEQUENCE
  structure Sign : SIGN
  type theory
  type thm
  exception THM of string * int * thm list
  exception THEORY of string * theory list
  val abstract_rule: string -> Sign.cterm -> thm -> thm
  val asm_rl: thm
  val assume: Sign.cterm -> thm
  val assumption: thm * int -> thm Sequence.seq   
  val beta_conversion: Sign.cterm -> thm   
  val bicompose: thm * int -> bool * thm * int -> thm Sequence.seq   
  val biresolution: thm * int * (bool*thm)list -> thm Sequence.seq   
  val combination: thm -> thm -> thm   
  val concl_of: thm -> term   
  val cut_rl: thm
  val dest_state: thm * int -> (term*term)list * term list * term * term
  val env_instantiate: Envir.env -> thm -> thm
  val enrich_theory: theory -> string -> string list * (string list * typ)list * Sign.Syntax.syntax -> (string*string)list -> theory
  val eq_assumption: thm * int -> thm   
  val equal_props: thm -> thm -> thm   
  val extend_theory: theory -> string -> string list * (string list * typ)list -> (string*string)list -> theory
  val extensional: thm -> thm   
  val forall_elim: Sign.cterm -> thm -> thm
  val forall_intr: Sign.cterm -> thm -> thm
  val get_axiom: theory -> string -> thm
  val implies_elim: thm -> thm -> thm
  val implies_intr: Sign.cterm -> thm -> thm
  val implies_intr_hyps: thm -> thm
  val instantiate: (Sign.cterm*Sign.cterm) list -> thm -> thm
  val kind_of_theory: theory -> string   
  val lift_rule: (thm * int) -> thm -> thm
  val merge_thy: theory * theory -> theory   
  val prems_of: thm -> term list
  val pure_thy: theory
  val reflexive: Sign.cterm -> thm 
  val rep_thm: thm ->  {prop: term, hyps: term list, maxidx: int, sign: Sign.sg}
  val resolution: thm * int * thm list -> thm Sequence.seq   
  val sign_of: theory -> Sign.sg   
  val symmetric: thm -> thm   
  val tpairs_of: thm -> (term*term)list
  val transitive: thm -> thm -> thm   
  val trivial: Sign.cterm -> thm
  val unify_instantiate: (term*term) list -> thm -> thm Sequence.seq
end;



functor ThmFun (structure Logic: LOGIC and Unify: UNIFY and Sign: SIGN) : THM = 
struct
structure Sequence = Unify.Sequence;
structure Envir = Unify.Envir;
structure Sign = Sign;
structure Syntax = Sign.Syntax;
structure Symtab = Syntax.Symtab;


(*Meta-theorems*)
datatype thm = Thm of
    {sign: Sign.sg,  maxidx: int,  hyps: term list,  prop: term};

fun rep_thm (Thm x) = x;

(*Errors involving theorems*)
exception THM of string * int * thm list;


(*maps object-rule to tpairs *)
fun tpairs_of th =
  let val {prop,...} = rep_thm th;
      val (tpairs,horn) = Logic.strip_equals prop 
  in  tpairs  end;

(*maps object-rule to premises *)
fun prems_of th =
  let val {prop,...} = rep_thm th;
      val (tpairs,horn) = Logic.strip_equals prop 
  in  Logic.strip_imp_prems horn  end;

(*maps object-rule to conclusion *)
fun concl_of th =
  let val {prop,...} = rep_thm th;
      val (tpairs,horn) = Logic.strip_equals prop 
  in  Logic.strip_imp_concl horn   end;



(*Theories.  There is one pure theory.
  A theory can be extended.  Two theories can be merged.*)
datatype theory =
    Pure of {sign: Sign.sg}
  | Extend of {sign: Sign.sg,  axioms: thm Symtab.table,  thy: theory}
  | Merge of {sign: Sign.sg,  thy1: theory,  thy2: theory};

(*Errors involving theories*)
exception THEORY of string * theory list;

fun kind_of_theory (Pure _) = "Pure"
  | kind_of_theory (Extend _) = "Extend"
  | kind_of_theory (Merge _) = "Union";

fun sign_of (Pure {sign}) = sign
  | sign_of (Extend {sign,...}) = sign
  | sign_of (Merge {sign,...}) = sign;


(*Merge theories of two theorems.  Raise exception if incompatible.
  Prefers (via Sign.merge) the signature of th1.  *)
fun merge_theories(th1,th2) =
  let val Thm{sign=sign1,...} = th1 and Thm{sign=sign2,...} = th2
  in  Sign.merge (sign1,sign2)  end
  handle TERM_ERROR _ => raise THM("incompatible signatures", 0, [th1,th2]);



(*Primitive rules*)
(*****************)


(* discharge all assumptions t from ts *)
fun disch (t,ts) = filter (fn u => not (t aconv u)) ts;

(*The assumption rule A|-A in a theory  *)
fun assume ct : thm = 
  let val {sign, t=prop, T, maxidx} = Sign.rep_cterm ct
  in  if T<>Aprop then  
	raise THM("assume: not a proposition", 0, [])
      else if maxidx <> ~1 then
	raise THM("assume: proposition contains Vars", maxidx, [])
      else Thm{sign = sign, maxidx = ~1, hyps = [prop], prop = prop}
  end;


(* Implication introduction  
	      A |- B
	      -------
	      A ==> B    *)
fun implies_intr cA (thB as Thm{sign,maxidx,hyps,prop}) : thm =
  let val {sign=signA, t=A, T, maxidx=maxidxA} = Sign.rep_cterm cA
  in  if T<>Aprop then
	raise THM("implies_intr: not a proposition", 0, [thB])
      else Thm{sign= Sign.merge (sign,signA),  maxidx= max[maxidxA, maxidx], 
	     hyps= disch(A,hyps),  prop= implies$A$prop}
      handle TERM_ERROR _ =>
        raise THM("implies_intr: incompatible signatures", 0, [thB])
  end;



(* Implication elimination
	A ==> B       A
	---------------
		B      *)
fun implies_elim thAB thA : thm =
    let val Thm{maxidx=maxA, hyps=hypsA, prop=propA,...} = thA
	and Thm{sign, maxidx, hyps, prop,...} = thAB;
	fun err(a) = raise THM("implies_elim: "^a, 0, [thAB,thA])
    in  case prop of
	    imp$A$B => 
		if imp=implies andalso  A aconv propA
		then  Thm{sign= merge_theories(thAB,thA),
			  maxidx= max[maxA,maxidx], 
			  hyps= hypsA@hyps,  (*dups not suppressed*)
			  prop= B}
		else err("major premise")
	  | _ => err("major premise")
    end;
      

(* Forall introduction.  The Free or Var x must not be free in the hypotheses.
     A
   ------
   !(x)A       *)
fun forall_intr cx (th as Thm{sign,maxidx,hyps,prop}) =
  let val x = Sign.term_of cx;
      fun result(a,T) = Thm{sign= sign, maxidx= maxidx, hyps= hyps,
	                    prop= all(T) $ Abs(a, T, abstract_over (x,prop))}
  in  case x of
	Free(a,T) => 
	  if exists (apl(x, Logic.occs)) hyps 
	  then  raise THM("forall_intr: variable free in hyps", 0, [th])
	  else  result(a,T)
      | Var((a,_),T) => result(a,T)
      | _ => raise THM("forall_intr: not a variable", 0, [th])
  end;


(* Forall elimination
	      !(x)A
	     --------
	      A[t/x]     *)
fun forall_elim ct (th as Thm{sign,maxidx,hyps,prop}) : thm =
  let val {sign=signt, t, T, maxidx=maxt} = Sign.rep_cterm ct
  in  case prop of
	  Const("all",(qary-->_)-->_) $ A =>
	    if T<>qary then
		raise THM("forall_elim: wrong typ", 0, [th])
	    else Thm{sign= Sign.merge(sign,signt), 
		     maxidx= max[maxidx, maxt],
		     hyps= hyps,  prop= betapply(A,t)}
	| _ => raise THM("forall_elim: not quantified", 0, [th])
  end
  handle TERM_ERROR _ =>
	 raise THM("forall_elim: incompatible signatures", 0, [th]);



(*Equality
  *********)

(*The reflexivity rule: maps  t   to the theorem   t==t   *)
fun reflexive ct = 
  let val {sign, t, T, maxidx} = Sign.rep_cterm ct
  in  Thm{sign= sign, hyps= [], maxidx= maxidx, prop= Logic.mk_equals(t,t)}
  end;



(*The symmetry rule
    t==u
    ----
    u==t
*)
fun symmetric (th as Thm{sign,hyps,prop,maxidx}) =
  case prop of
      (eq as Const("==",_)) $ t $ u =>
	  Thm{sign=sign, hyps=hyps, maxidx=maxidx, prop= eq$u$t} 
    | _ => raise THM("symmetric", 0, [th]);


(*The transitive rule
    t1==u    u==t2
    ------------
        t1==t2      *)
fun transitive th1 th2 =
  let val Thm{maxidx=max1, hyps=hyps1, prop=prop1,...} = th1
      and Thm{maxidx=max2, hyps=hyps2, prop=prop2,...} = th2;
      fun err(msg) = raise THM("transitive: "^msg, 0, [th1,th2])
  in case (prop1,prop2) of
       ((eq as Const("==",_)) $ t1 $ u, Const("==",_) $ u' $ t2) =>
	  if not (u aconv u') then err"middle term"  else
	      Thm{sign= merge_theories(th1,th2), hyps= hyps1@hyps2, 
		  maxidx= max[max1,max2], prop= eq$t1$t2}
     | _ =>  err"premises"
  end;



(*Beta-conversion: maps (%(x)t)(u) to the theorem  (%(x)t)(u) == t[u/x]   *)
fun beta_conversion ct = 
  let val {sign, t, T, maxidx} = Sign.rep_cterm ct
  in  case t of
	  Abs(_,_,bodt) $ u => 
	    Thm{sign= sign,  hyps= [],  
		maxidx= maxidx_of_term t, 
		prop= Logic.mk_equals(t, subst_bounds([u],bodt))}
	| _ =>  raise THM("beta_conversion: not a redex", 0, [])
  end;



(*The extensionality rule   (proviso: x not free in f, g, or hypotheses)
    f(x) == g(x)
    ------------
       f == g    *)
fun extensional (th as Thm{sign,maxidx,hyps,prop}) =
  case prop of
    (Const("==",_)) $ (f$x) $ (g$y) =>
      let fun err(msg) = raise THM("extensional: "^msg, 0, [th]) 
      in (if x<>y then err"different variables" else
          case y of
		Free _ => 
		  if exists (apl(y, Logic.occs)) (f::g::hyps) 
		  then err"variable free in hyps or functions"    else  ()
	      | Var _ => 
		  if Logic.occs(y,f)  orelse  Logic.occs(y,g) 
		  then err"variable free in functions"   else  ()
	      | _ => err"not a variable");
	  Thm{sign=sign, hyps=hyps, maxidx=maxidx, 
	      prop= Logic.mk_equals(f,g)} 
      end
 | _ =>  raise THM("extensional: premise", 0, [th]);



(*The abstraction rule.  The Free or Var x must not be free in the hypotheses.
  The bound variable will be named "a" (since x will be something like x320)
          t == u
    ----------------
      %(x)t == %(x)u     *)
fun abstract_rule a cx (th as Thm{sign,maxidx,hyps,prop}) =
  let val x = Sign.term_of cx;
      val (t,u) = Logic.dest_equals prop  
	    handle TERM_ERROR _ =>
		raise THM("abstract_rule: premise not an equality", 0, [th])
      fun result T =
            Thm{sign= sign, maxidx= maxidx, hyps= hyps,
	        prop= Logic.mk_equals(Abs(a, T, abstract_over (x,t)),
		  	              Abs(a, T, abstract_over (x,u)))}
  in  case x of
	Free(_,T) => 
	  if exists (apl(x, Logic.occs)) hyps 
	  then  raise THM("abstract_rule: variable free in hyps", 0, [th])
	  else  result T
      | Var(_,T) => result T
      | _ => raise THM("abstract_rule: not a variable", 0, [th])
  end;


(*The combination rule
    f==g    t==u
    ------------
     f(t)==g(u)      *)
fun combination th1 th2 =
  let val Thm{maxidx=max1, hyps=hyps1, prop=prop1,...} = th1
      and Thm{maxidx=max2, hyps=hyps2, prop=prop2,...} = th2
  in  case (prop1,prop2)  of
       (Const("==",_) $ f $ g, Const("==",_) $ t $ u) =>
	      Thm{sign= merge_theories(th1,th2), hyps= hyps1@hyps2, 
		  maxidx= max[max1,max2], prop= Logic.mk_equals(f$t, g$u)}
     | _ =>  raise THM("combination: premises", 0, [th1,th2])
  end;


(*The equal propositions rule
    A==B    A
    ---------
        B          *)
fun equal_props th1 th2 =
  let val Thm{maxidx=max1, hyps=hyps1, prop=prop1,...} = th1
      and Thm{maxidx=max2, hyps=hyps2, prop=prop2,...} = th2;
      fun err(msg) = raise THM("equal_props: "^msg, 0, [th1,th2])
  in  case prop1  of
       Const("==",_) $ A $ B =>
	  if not (prop2 aconv A) then err"not equal"  else
	      Thm{sign= merge_theories(th1,th2), hyps= hyps1@hyps2, 
		  maxidx= max[max1,max2], prop= B}
     | _ =>  err"major premise"
  end;



(*Derived rules*)
(***************)


(*Discharge all hypotheses (need not verify cterms)
  Repeated hypotheses are discharged only once;  itlist cannot do this*)
fun implies_intr_hyps (Thm{sign, maxidx, hyps=A::As, prop}) =
      implies_intr_hyps
	    (Thm{sign=sign,  maxidx=maxidx, 
	         hyps= disch(A,As),  prop= implies$A$prop})
  | implies_intr_hyps th = th;


(*Remove trivial tpairs using implies_elim and reflexivity
	example: t1==t2 ==> u==u ==> P    goes to    t1==t2 ==> P   
  Used to smash flex-flex pairs*)
fun refl_imp_elim (th as Thm{sign,maxidx,hyps,prop}) : thm = 
    let val (tpairs,horn) = Logic.strip_equals prop;
       	val newprop = Logic.list_equals(filter (not o op aconv) tpairs, horn)
    in  Thm{sign= sign, hyps= hyps, 
	    maxidx= maxidx, prop= newprop}
    end;


(*Instantiation of Vars
		      A
	     --------------------
	      A[t1/v1,....,tn/vn]     *)

(*Check: all are Vars and are distinct*)
fun instl_ok ts = case t_findrep ts of
        [] => forall is_var ts
    | _::_ => false;


(*For instantiate: process pair of cterms, merge theories*)
fun add_ctpair ((ct,cu), (sign,tpairs)) =
  let val {sign=signt, t=t, T= T, maxidx= maxt} = Sign.rep_cterm ct
      and {sign=signu, t=u, T= U, maxidx= maxu} = Sign.rep_cterm cu
  in  if T=U  then
	 (Sign.merge(sign, Sign.merge(signt, signu)),
	  (t,u)::tpairs)
      else  raise TYPE("add_ctpair", [T,U], [t,u])
  end;


(*Left-to-right replacements: tpairs = [...,(vi,ti),...].
  Instantiates distinct Vars by terms of same type.
  Normalizes the new theorem! *)
fun instantiate ctpairs  (th as Thm{sign,maxidx,hyps,prop}) = 
  let val (newsign,tpairs) = itlist_right add_ctpair (ctpairs, (sign,[]));
      val newprop = Envir.norm_term (Envir.empty 0) (subst_atomic tpairs prop)
  in  if not (instl_ok (map fst tpairs)) then
        raise THM("instantiate: not distinct Vars", 0, [th])
      else Thm{sign= newsign, hyps= hyps, 
	       maxidx= maxidx_of_term newprop, prop= newprop}
      end
  handle TERM_ERROR _ => raise THM("instantiate: incompat signatures",0,[th])
       | TYPE _ => raise THM("instantiate: types", 0, [th]);


(*Instantiation using environment: COMPARE TYPES!!!?? *)
fun env_instantiate env th : thm =
  let val Thm{sign,hyps,prop,...} = th;
      val cts = map (Sign.cterm_of sign) (map snd (Envir.alist_of env))
	    handle TERM_ERROR _ => 
		raise THM("env_instantiate: term not in signature", 0, [th]);
      val nprop = Envir.norm_term env prop
  in  Thm{sign=sign, hyps=hyps,  prop=nprop,  maxidx=maxidx_of_term nprop}
  end;


(*The trivial implication A==>A in a theory
  Derived from assume and forall rules 
  Note: A can contain Vars, not so for assume!   *)
fun trivial ct : thm = 
  let val {sign, t=A, T, maxidx} = Sign.rep_cterm ct
  in  if T<>Aprop then  
	    raise THM("trivial: not a proposition", 0, [])
      else Thm{sign= sign, maxidx= maxidx, hyps= [], prop= implies$A$A}
  end;


(*Destruct proof state into constraints, other goals, goal(i), rest *)
fun dest_state (state as Thm{prop,...}, i) =
  let val (tpairs,horn) = Logic.strip_equals prop
  in  case  Logic.strip_prems(i, [], horn) of
          (B::rBs, C) => (tpairs, rev rBs, B, C)
        | _ => raise THM("dest_state", i, [state])
  end
  handle TERM_ERROR _ => raise THM("dest_state", i, [state]);



(*Increment variables and parameters of rule as required for
  resolution with goal i of state. *)
fun lift_rule (state, i) orule =
  let val Thm{prop=sprop,maxidx=smax,...} = state;
      val (_,shorn) = Logic.strip_equals(sprop);
      val (Bi::_, _) = Logic.strip_prems(i, [], shorn)
	handle TERM_ERROR _ => raise THM("lift_rule", i, [orule,state]);
      val (lift_abs,lift_all) = Logic.lift_fns(Bi,smax+1);
      val (Thm{sign,maxidx,hyps,prop}) = orule;
      val (tpairs,As,B) = Logic.strip_horn prop
  in  Thm{hyps=hyps, sign= merge_theories(state,orule),
	  maxidx= maxidx+smax+1,
	  prop= Logic.rule_of(map (pairself lift_abs) tpairs,
			      map lift_all As,    lift_all B)}
  end;


(*Solve subgoal Bi of proof state B1...Bn/C by assumption. *)
fun assumption (state, i) =
  let val Thm{sign,maxidx,hyps,prop} = state;
      val (tpairs, Bs, Bi, C) = dest_state(state,i)
      fun newth (env as Envir.Envir{maxidx,asol}, tpairs) =
	  Thm{sign=sign, hyps=hyps, maxidx=maxidx, prop=
	    case Envir.alist_of_olist asol of  (*avoid wasted normalizations*)
	        [] => Logic.rule_of(tpairs, Bs, C)
	      | ((_,m),_) :: _ => (*normalize the new rule fully*)
		      Envir.norm_term env (Logic.rule_of(tpairs, Bs, C))};
      fun addprfs [] = Sequence.null
        | addprfs ((t,u)::apairs) = Sequence.seqof (fn()=> Sequence.pull
             (Sequence.mapp newth
	        (Unify.unifiers(Envir.empty maxidx, (t,u)::tpairs)) 
	        (addprfs apairs)))
  in  addprfs (Logic.assum_pairs Bi)  end;


(*Solve subgoal Bi of proof state B1...Bn/C by assumption. 
  Checks if Bi's conclusion is alpha-convertible to one of its assumptions*)
fun eq_assumption (state, i) =
  let val Thm{sign,maxidx,hyps,prop} = state;
      val (tpairs, Bs, Bi, C) = dest_state(state,i)
  in  if exists (op aconv) (Logic.assum_pairs Bi)
      then Thm{sign=sign, hyps=hyps, maxidx=maxidx, 
	       prop=Logic.rule_of(tpairs, Bs, C)}
      else  raise THM("eq_assumption", 0, [state])
  end;


fun rename t = Logic.rename_params ("ka",t);

(*Composition of object rule (A1...Am/B) with proof state (B1...Bn/C)
  Unifies B with Bi, replacing subgoal i    (1 <= i <= n)  
  If bires_flg then simultaneously proves A1 by assumption.
  nsubgoal is the number of new subgoals (written m above). *)
fun bicompose_aux (state, (stpairs, Bs, Bi, C)) (bires_flg, orule, nsubgoal) =
  let val Thm{maxidx=smax, hyps=shyps, ...} = state
      and Thm{maxidx=rmax, hyps=rhyps, prop=rprop,...} = orule;
      val sign = merge_theories(state,orule);
      fun newth As (env as Envir.Envir{maxidx,asol}, tpairs) =
	  Thm{sign=sign, hyps=rhyps@shyps, maxidx=maxidx, prop=
	    case Envir.alist_of_olist asol of  (*avoid wasted normalizations*)
	        [] => Logic.rule_of(tpairs, Bs @ As, C)
	      | ((_,idx),_) :: _ =>
	          if idx>smax then (*no assignments in state*)
		      Logic.rule_of(map (pairself(Envir.norm_term env)) tpairs,
			            Bs @ map (Envir.norm_term env) As,  C)
	          else (*normalize the new rule fully*)
		      Envir.norm_term env (Logic.rule_of(tpairs, Bs @ As, C))};
      val (rtpairs,rhorn) = Logic.strip_equals(rprop);
      val (rAs,B) = Logic.strip_prems(nsubgoal, [], rhorn)
	handle TERM_ERROR _ => raise THM("bicompose: rule", 0, [orule,state]);
      val env = Envir.empty(max[rmax,smax])
      and dpairs = (B,Bi)::rtpairs@stpairs;
      fun tryasms (_, _, []) = Sequence.null
        | tryasms (As, n, (t,u)::apairs) =
	   (case Sequence.pull(Unify.unifiers(env, (t,u)::dpairs))  of
		None => tryasms (As, n+1, apairs)
	      | cell => Sequence.mapp 
		  (newth (map (rename o Logic.delete_asm n) As)
			handle TERM_ERROR _ =>
			raise THM("bicompose: 1st premise", 0, [orule]))
		  (Sequence.seqof (fn()=> cell))
		  (Sequence.seqof (fn()=> Sequence.pull 
		      (tryasms (As, n+1, apairs)))));
      fun bires [] = raise THM("bicompose: no premises", 0, [orule,state])
        | bires (A1::As) = tryasms (As, 1, Logic.assum_pairs A1)
  in  if bires_flg then bires(rev rAs)  
      else  case Sequence.pull(Unify.unifiers(env, dpairs)) of
		  None => Sequence.null
		| cell => Sequence.maps (newth (map rename (rev rAs))) 
				(Sequence.seqof (fn()=> cell))
  end;


(*Curried so that resolution calls dest_state only once.*)
fun bicompose (state, i) = bicompose_aux (state, dest_state(state,i));


(*Quick test whether rule is resolvable with the subgoal with hyps Hs
  and conclusion B.  If bires_flg then checks 1st premise of rule also*)
fun could_bires (Hs, B, bires_flg, rule) =
    let fun could_reshyp (A1::_) = exists (apl(A1,could_unify)) Hs
	  | could_reshyp [] = false;  (*no premise -- illegal*)
    in  could_unify(concl_of rule, B) andalso 
	(not bires_flg  orelse  could_reshyp (prems_of rule))
    end;


(*Bi-resolution of a state with a list of flag/rule pairs.
  Puts the rule above:  rule/state.  Renames vars in the rules. *)
fun biresolution (state, i, brules) = 
    let val lift = lift_rule(state, i);
	val (stpairs, Bs, Bi, C) = dest_state(state,i)
	val B = Logic.strip_assums_concl Bi;
	val Hs = Logic.strip_assums_hyp Bi;
	val comp = bicompose_aux (state, (stpairs, Bs, Bi, C));
	fun res [] = Sequence.null
	  | res ((bires_flg, rule)::brules) = 
	      if could_bires (Hs, B, bires_flg, rule)
	      then Sequence.seqof (*delay processing remainder til needed*)
	          (fn()=> Some(comp (bires_flg, lift rule, 
			     	     length(prems_of rule)),
			       res brules))
	      else res brules
    in  Sequence.flats (res brules)  end;


(*Old style resolution, the flag is false (no attempt to unify assumptns) *)
fun resolution (state, i, rules) = 
  biresolution (state, i, map (fn rl => (false,rl)) rules);


(*Smash" unifies the list of term pairs leaving no flex-flex pairs.
  Instantiates the theorem and deletes trivial tpairs. *)
fun unify_instantiate [] th = Sequence.single th
  | unify_instantiate tpairs th =
      let val {sign,maxidx,...} = rep_thm th;
	  fun newpr env = refl_imp_elim (env_instantiate env th)
      in Sequence.maps newpr (Unify.smash_unifiers(Envir.empty maxidx, tpairs))
      end;



(* T H E O R I E S *)


val pure_thy = Pure{sign = Sign.pure};


(*Look up the named axiom in the theory*)
fun get_axiom thy axname =
    let fun get (Pure _) = raise Match
	  | get (Extend{axioms,thy,...}) =
	     (case Symtab.lookup(axioms,axname) of
		  Some th => th
		| None => get thy)
 	 | get (Merge{thy1,thy2,...}) = 
		get thy1  handle Match => get thy2
    in  get thy
	handle Match => raise THEORY("get_axiom: No axiom "^axname, [thy])
    end;



(*Converts Frees to Vars: axioms can be written without question marks*)
fun prepare_axiom sign sP =
    Logic.varify (Sign.term_of (Sign.read_cterm sign (sP,Aprop)));


(*Read an axiom for a new theory*)
fun read_ax sign (a, sP) : string*thm =
  let val prop = prepare_axiom sign sP
  in  (a, Thm{sign=sign, hyps=[], maxidx= maxidx_of_term prop, prop= prop}) 
  end
  handle ERROR =>
	error("extend_theory: The above error occurred in axiom " ^ a ^ "\n");


(*Enrichment of a theory with given types and constants.  Uses new syntax.
  Reads the axioms from strings: axpairs have the form (axname, axiom). *)
fun enrich_theory thy thyname (types, const_decs, syn) axpairs =
  let val sign = Sign.extend (sign_of thy) thyname
		(types,const_decs,syn);
      val axioms= Symtab.st_of_alist(map (read_ax sign) axpairs, Symtab.null)
  in  Extend{sign=sign, axioms= axioms, thy = thy}  end;


(*Like enrich_theory but uses old syntax.  Infix decs involve new syntax! *)
fun extend_theory thy thyname (types, const_decs) axpairs =
  let val oldsign = sign_of thy;
      val sign = Sign.extend oldsign thyname
		(types, const_decs, #syn(Sign.rep_sg oldsign));
      val axioms= Symtab.st_of_alist(map (read_ax sign) axpairs, Symtab.null)
  in  Extend{sign=sign, axioms= axioms, thy = thy}  end;


(*The union of two theories*)
fun merge_thy (thy1,thy2) =
    Merge{sign = Sign.merge(sign_of thy1, sign_of thy2),
	  thy1 = thy1, thy2 = thy2};



(*The rule P/P, obtains assumption solving for eresolve_tac*)
val asm_rl = trivial(Sign.read_cterm Sign.pure ("?psi",Aprop));

(*Meta-level cut rule*)
val cut_rl = trivial(Sign.read_cterm Sign.pure ("?psi==>?theta",Aprop));

end;
