(*  Title: 	NJ/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Intuitionistic first-order logic (Natural Deduction). 

Note that ~P abbreviates P-->False;  negation does not exist in its own right.
    (Otherwise we would have to write ~(P) instead of ~P)

Should allow multiple quantification like ALL x y.P
Could define P<->Q == (P-->Q)&(Q-->P) and derive its rules.
Could define True == False-->False as a standard tautology.
*)


signature NJ_SYNTAX =
sig
  structure Syntax : SYNTAX
  val Aexp: typ
  val Aform: typ
  val Forall: term
  val Exists: term
  val True: term
  val const_decs: (string list * typ) list
  val imp: term
  val parser: (string*Syntax.lexsy) list -> term * (string*Syntax.lexsy) list
  val printer: Syntax.lexsy Syntax.Symtab.table -> term -> unit   
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;


functor NJ_SyntaxFun (structure Syntax: SYNTAX and Pretty: PRETTY) 
  : NJ_SYNTAX = 
struct
structure Syntax = Syntax;
structure Symtab = Syntax.Symtab;
local open Syntax
in


(*Types of expressions, formulae*)
val Aexp  = Ground "exp";
val Aform = Ground "form";


(*Constants and their types*)
val const_decs = 
  [ (["True"],			Aform-->Aprop ),
    (["="],			[Aexp,Aexp]--->Aform ),
    (["False"],			Aform ),
    (["&", "|", "-->", "<->"],	[Aform,Aform]--->Aform ),
    (["Forall", "Exists"],	(Aexp-->Aform) --> Aform ) ];


(*Get constants from table*)
val get = Sign.get_const const_decs;

val True	= get "True";
val Forall	= get "Forall"
and Exists	= get "Exists"
and imp   	= get "-->"
and False   	= get "False";


 
(*Parsing of terms and theorems (mutually recursive).
  Allows prefix notation: ~P for P-->False.  Need a mechanism for prefix ops.
  Treatment of ~ is strange since precedence must be passed to parser. *)
fun parser lxs = parse_functl parse 0 lxs
and parse_quant quant lxs =
      apfst (fn ((a,T), P) => quant $ absfree(a,T,P))
	    (((parse_ident thenkey".") thenp parser) lxs)
and parse (("[|",DelimSy) :: lxs) =
      apfst (fn P => True$P) ((parser thenkey"|]") lxs)
  | parse (("ALL",DelimSy) :: lxs) = parse_quant Forall lxs
  | parse (("EXISTS",DelimSy) :: lxs) = parse_quant Exists lxs
  | parse (("~",DelimSy) :: lxs) =
      apfst (fn P => imp$P$False) (parse_functl parse 100 lxs) 
  | parse lxs = parserr("Illegal NJ expression", lxs);



(** Printing **)

(*Special treatment of Exists, Forall *)
fun print1 lextab (outl,outr) =
  let fun pr P = print1 lextab (0,0) P; 
      fun print_quant (pre, (a,P)) =  (*same code in LK*)
	  (Pretty.bg 2;  Pretty.st (pre ^ " " ^ a ^ ".");  Pretty.brk(1,0);  
           pr P;  Pretty.en())
  in  fn
      Const("True",_) $ P => (Pretty.bgs"[| ";  pr P;  Pretty.ens" |]")
    | Const("Forall",_) $ Abs(a,T,P) => 
        enclose_if (outr>0) print_quant ("ALL", variant_abs(a,T,P))
    | Const("Exists",_)  $ Abs(a,T,P) =>
        enclose_if (outr>0) print_quant ("EXISTS", variant_abs(a,T,P))
    | Const("-->",_)  $ P $ Const("False",_) =>
	(Pretty.st"~ ";  print1 lextab (100, outr) P)
    | t => print_functl (print1 lextab, lextab) (outl,outr) t
  end;
  

fun printer lextab : term->unit = print1 lextab (0,0);


(*Infixes and their precedence*)
val infix_decs = 
    [ (["="],		"left", 6),
      (["&"],		"right", 5),
      (["|"],		"right", 4),
      (["-->","<->"],	"right", 3) ];

val syn = Syntax.extend pure
	([ ".", "~", "ALL", "EXISTS"], infix_decs, parser, printer); 

(*read/print functions*)
fun read a = Syntax.read syn "?" a;
fun prin t = (Pretty.init();  Syntax.prin syn t;  Pretty.nl());

end;
end;

