(*  Title: 	LK/set/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Zermelo-Fraenkel Set Theory
*)


signature SET_SYNTAX =
sig
  structure Syntax : SYNTAX
  val const_decs: (string list * typ) list
  val parser: (string*Syntax.lexsy) list -> term * (string*Syntax.lexsy) list
  val printer: Syntax.lexsy Syntax.Symtab.table -> term -> unit
  val set_elems: term -> term list
  val syn: Syntax.syntax
end;

functor Set_SyntaxFun (structure LK_Syntax: LK_SYNTAX and Pretty: PRETTY) 
      : SET_SYNTAX = 
struct
structure Syntax = LK_Syntax.Syntax;
structure Symtab = Syntax.Symtab;
local open Syntax LK_Syntax
in

(*Constants and their types*)
val const_decs = 
  [ (["`", "Int", "Un", "-", "::", "Pair"],	[Aexp,Aexp]--->Aexp ),
    (["Pow","Choose","Union","Inter","succ"],	Aexp-->Aexp),
    (["<=", ":"],	[Aexp,Aexp]--->Aform ),
    (["Collect"],	[Aexp, Aexp-->Aform] ---> Aexp ),
    (["Replace"],	[Aexp-->Aexp, Aexp] ---> Aexp ),
    (["0","INF"],	Aexp ) ];


(*Get constants.*)
val get = Sign.get_const const_decs;

val zero   	= get "0"
and setcons   	= get "::"
and Pair   	= get "Pair"
and Collect   	= get "Collect"
and Replace   	= get "Replace";


(*** Parsing ***)


(** Pair and set notation -- special treatment of Pair ::  **)


(*Construct x::y *)
fun mk_setcons(x,y) = setcons$x$y;


exception SET_ELEMS;

(*Return the elements of a finite set, raise exception if ill-formed.*)
fun set_elems (Const("::",_) $ a $ b) = a :: set_elems b
  | set_elems (Const("0",_)) = []
  | set_elems _ = raise SET_ELEMS;


(*Replacement [ b[x] || x:A ]       = Replace(%(x)b[x], A)
  Collection  [  x   || x:A, P[x] ] = Collect(A, %(x)P[x])
  both	      [ b[x] || x:A, P[x] ] = Replace(%(x)b[x], Collect(A, %(x)P[x]))
*)
fun parse_abs recparse ls = 
    let val (((t, (a,T)), A), ls1) = 
	(recparse     thenkey"||" thenp  
	 parse_ident  thenkey":"  thenp  recparse) ls
    in  case ls1 of
	((",",DelimSy) :: ls2) =>
	        let val (P, ls3) = recparse ls2;
		    val coll = Collect $ A $ absfree(a,T,P);
		    val u = if t = Free(a,T) then coll
		             else  Replace $ absfree(a,T,t) $ coll
		in  (u, ls3)  end
      | _ => (Replace $ absfree(a,T,t) $ A,  ls1)
    end;


fun set_parse1 recparse =
  let fun parse (("{",DelimSy) :: ls) =
	    apfst (fn elems => itlist_right mk_setcons (elems,zero)) 
		  ((parepin (",", recparse) thenkey"}") ls)
        | parse (("<",DelimSy) :: ls) =
	    apfst (fn (a,b) => Pair$a$b) 
		  ((recparse thenkey","  thenp  recparse thenkey">") ls)
        | parse (("[",DelimSy) :: ls) = (parse_abs recparse thenkey"]") ls
        | parse ls = LK_Syntax.parse1 recparse ls
  in  parse_functl parse 0  end;


fun parser ls = set_parse1 parser ls;



(*** Printing ***)


(*A functional for use with LK_Syntax.print1*)
fun set_print1 recprint lex (arg as (outleft,outright)) t =
  let fun prfront (t,a,A) =
	    (Pretty.bgs"[ ";  recprint lex (0,0) t;
             Pretty.brks (" || " ^ a ^ " :") 1;  recprint lex (0,0) A);
      fun prback P = (Pretty.brks "," 1;  recprint lex (0,0) P;  Pretty.ens" ]")
  in case t of
      Const("::",_) $ _ $ _ =>  
	(Pretty.list("{", "}", recprint lex (0,0)) (set_elems t)
	 handle SET_ELEMS => LK_Syntax.print1 recprint lex arg t)
    | Const("Pair",_) $ a $ b =>  
	Pretty.list("<", ">", recprint lex (0,0)) [a,b]
    | Const("Replace",_) $ Abs(a,T,t) $
			  (Const("Collect",_) $ A $ Abs(_,_,P)) => 
  	let val frees = add_term_names(t, add_term_names(P, []));
	    val b = variant frees a;
	    val u = subst_bounds ([Free(b,T)], t);
	    val Q = subst_bounds ([Free(b,T)], P)
	in  prfront (u,b,A);  prback Q  end
    | Const("Collect",_) $ A $ Abs(a,T,P) => 
  	let val (b, Q) = variant_abs(a,T,P)
	in  prfront (Free(b,T), b, A);  prback Q  end
    | Const("Replace",_) $ Abs(a,T,t) $ A => 
  	let val (b, u) = variant_abs(a,T,t)
	in  prfront (u,b,A);  Pretty.ens" ]"  end
    | _ => LK_Syntax.print1 recprint lex arg t
  end;


fun printer lex =
  let fun  pr x = set_print1 pr x
  in  pr lex (0,0)  end;


(*Infixes and their precedence*)
val infix_decs = 
    [ (["`"],		"left",  9),
      (["Int"],   	"right",  8),
      (["Un"],    	"right",  7),
      (["-"],     	"right",  7),
      (["::"],		"right",  7),
      (["<="],		"right",  6),
      ([":"],		"right",  6) ];

val syn = Syntax.extend LK_Syntax.syn
	([ "{", "}", "<", ">", "[", "]", "||"],
		infix_decs, parser, printer); 

(*read/print functions*)
fun read a = Syntax.read syn "?" a;
fun prin t = (Pretty.init();  Syntax.prin syn t;  Pretty.nl());

end;
end;
