(*  Title: 	LK/resolve
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Classical first-order logic tactics
  (thanks also to Philippe de Groote)  *)

signature LK_RESOLVE = 
sig
  structure Tactic : TACTIC
  val all_left_thin: Tactic.Thm.thm
  val could_res: term * term -> bool
  val could_resolve_seq: term * term -> bool
  val cut_tac: string -> int -> Tactic.tactic
  val cut_left_tac: string -> int -> Tactic.tactic
  val cut_right_tac: string -> int -> Tactic.tactic
  val duplicate_left: Tactic.Thm.thm
  val duplicate_left_tac: string -> int -> Tactic.tactic
  val duplicate_right: Tactic.Thm.thm
  val duplicate_right_tac: string -> int -> Tactic.tactic
  val exists_right_thin: Tactic.Thm.thm
  val filseq_resolve_tac: Tactic.Thm.thm list -> int -> int -> Tactic.tactic
  val forms_of_seq: term -> term list
  val has_prems: int -> Tactic.Thm.thm -> bool   
  val iff_left: Tactic.Thm.thm
  val iff_right: Tactic.Thm.thm
  val LK_pack: Tactic.Thm.thm list * Tactic.Thm.thm list
  val pc_tac: int -> Tactic.tactic
  val pjoin: ('a list * 'b list) * ('a list * 'b list) -> 'a list * 'b list
  val repeat_goal_tac: (Tactic.Thm.thm list * Tactic.Thm.thm list) list -> int -> Tactic.tactic
  val reresolve_tac: Tactic.Thm.thm list -> int -> Tactic.tactic   
  val RESOLVE_THEN: Tactic.Thm.thm list -> (int -> Tactic.tactic) -> int -> Tactic.tactic   
  val safe_goal_tac: (Tactic.Thm.thm list * 'a) list -> int -> Tactic.tactic
  val step_tac: (Tactic.Thm.thm list * Tactic.Thm.thm list) list -> int -> Tactic.tactic
  val sym_left: Tactic.Thm.thm
  val thin_left_tac: string -> int -> Tactic.tactic
  val thin_right_tac: string -> int -> Tactic.tactic
  val triv_pack: Tactic.Thm.thm list * 'a list
end;


functor LK_ResolveFun (structure LK_Syntax: LK_SYNTAX 
			     and LK_Rule: LK_RULE 
			     and Conv: CONV
	sharing LK_Syntax.Syntax = LK_Rule.Thm.Sign.Syntax
	    and Conv.Tactic.Thm = LK_Rule.Thm) : LK_RESOLVE = 
struct
structure Tactic = Conv.Tactic;
local  open Conv Tactic Thm LK_Rule LK_Syntax

in 


(*For "cutting in" an arbitrary formula*)
fun cut_tac (sP: string) = res_inst_tac [ ("P",sP,Aform) ] cut;

(*Cut and thin, replacing the right-side formula*)
fun cut_right_tac (sP: string) i = 
    cut_tac sP i  THEN  resolve_tac [thin_right] i;

(*Cut and thin, replacing the left-side formula*)
fun cut_left_tac (sP: string) i = 
    cut_tac sP i  THEN  resolve_tac [thin_left] (i+1);

(*Thinning a formula on the right*)
fun thin_right_tac (sP: string) = res_inst_tac [ ("P",sP,Aform) ] thin_right;

(*Thinning a formula on the left*)
fun thin_left_tac (sP: string) = res_inst_tac [ ("P",sP,Aform) ] thin_left;


(* iff right*)
val iff_right = prove_goal LK_Rule.thy 
    "[| $H,P |- $E,Q,$F |] ==> [| $H,Q |- $E,P,$F |] ==> \
\    [| $H |- $E, P <-> Q, $F |]"
 (fn asms=>
  [ (rewrite_goals_tac [iff_def]),
    (REPEAT (resolve_tac (asms@[conj_right,imp_right]) 1)) ]);


(* iff left*)
val iff_left = prove_goal LK_Rule.thy 
   "[| $H,$G |- $E,P,Q |] ==> [| $H,Q,P,$G |- $E |] ==> \
\    [| $H, P <-> Q, $G |- $E |]"
 (fn asms=>
  [ rewrite_goals_tac [iff_def],
    (REPEAT (resolve_tac (asms@[conj_left,imp_left,basic]) 1)) ]);


(* Thinned version of all_left.  Not complete, but lets search terminate.*)
val all_left_thin = prove_goal LK_Rule.thy 
    "[| $H, P(a), $G |- $E |] ==> [| $H, Forall(P), $G |- $E |]"
 (fn asms=>
  [ (resolve_tac [all_left] 1),
    (thin_left_tac "Forall(P)" 1),
    (resolve_tac asms 1) ]);


(* Thinned version of exists_right.  Not complete, but lets search terminate.*)
val exists_right_thin = prove_goal LK_Rule.thy 
    "[| $H |- $E, P(a), $F |] ==> [| $H |- $E, Exists(P), $F |]"
 (fn asms=>
  [ (resolve_tac [exists_right] 1),
    (thin_right_tac "Exists(P)" 1),
    (resolve_tac asms 1) ]);



(* Symmetry of equality in hypotheses *)
val sym_left = prove_goal LK_Rule.thy 
    "[| $H, $G, B = A |- $E |] ==> [| $H, A = B, $G |- $E |]" 
 (fn asms=>
  [ (resolve_tac [cut] 1),
    (resolve_tac [thin_left] 2),
    (resolve_tac asms 2),
    (resolve_tac [sym RES basic] 1) ]);



(*A theorem pack has the form  (safe-theorems, last-resort-theorems)
  A typical last-resort-theorem is not complete or 
    introduces variables in subgoals,
  and is tried only when the safe theorems are not applicable.  *)


(*The rules of LK*)
val LK_pack = 
   ([conj_left, conj_right, disj_left, disj_right,
     imp_left, imp_right, 
     not_left, not_right, iff_left, iff_right,
     all_right, exists_left],
    [all_left_thin, exists_right_thin]);

val triv_pack = ([basic, refl],  []);

(*Append two packs*)
fun pjoin ((safe1,unsafe1), (safe2,unsafe2)) = (safe1@safe2, unsafe1@unsafe2);


(*Returns the list of all formulas in the sequent*)
fun forms_of_seq (Const("Seqof",_) $ P $ u) = P :: forms_of_seq u
  | forms_of_seq (H $ u) = forms_of_seq u
  | forms_of_seq _ = [];


(*Tests whether two sequences (left or right sides) could be resolved.
  seqp is a premise (subgoal), seqc is a conclusion of an object-rule.
  Assumes each formula in seqc is surrounded by sequence variables
  -- checks that each concl formula looks like some subgoal formula.*)
fun could_res (seqp,seqc) =
      forall (fn Qc => exists (fn Qp => could_unify (Qp,Qc)) 
                              (forms_of_seq seqp))
             (forms_of_seq seqc);


(*Tests whether two sequents G|-H could be resolved, comparing each side.*)
fun could_resolve_seq (prem,conc) =
  case (prem,conc) of
      (_ $ Abs(_,_,leftp) $ Abs(_,_,rightp),
       _ $ Abs(_,_,leftc) $ Abs(_,_,rightc)) =>
	  could_res (leftp,leftc)  andalso  could_res (rightp,rightc)
    | _ => false;


(*Like filt_resolve_tac, using could_resolve_seq
  Much faster than resolve_tac when there are many rules.
  Resolve subgoal i using the rules, unless more than maxr are compatible. *)
fun filseq_resolve_tac rules maxr = SUBGOAL(fn (prem,i) =>
  let val rls = filter_thms could_resolve_seq (maxr+1, prem, rules)
  in  if length rls > maxr  then  no_tac  else resolve_tac rls i
  end);


(*Predicate: does the rule have n premises? *)
fun has_prems n rule =  (length (prems_of rule) = n);


(*Continuation-style tactical for resolution.
  The list of theorems is partitioned into 0, 1, 2 premises.
  The resulting tactic, gtac, tries to resolve with theorems.
  If successful, it recursively applies nextac to the new subgoals only.
  Else fails.  (Treatment of goals due to Ph. de Groote) 
  Bind (RESOLVE_THEN rules) to a variable: it preprocesses the rules. *)

(*Takes theorem lists separated in to 0, 1, 2, >2 premises.
  The abstraction over state prevents needless divergence in recursion.
  The 999 should be a parameter, to delay treatment of flexible goals. *)
fun RESOLVE_THEN rules =
  let val [rls0,rls1,rls2] = partition_list has_prems 0 2 rules;
      fun tac nextac i = STATE (fn state =>  
	  filseq_resolve_tac rls0 999 i 
	  ORELSE
	  (DETERM(filseq_resolve_tac rls1 999 i) THEN  TRY(nextac i))
	  ORELSE
	  (DETERM(filseq_resolve_tac rls2 999 i) THEN  TRY(nextac(i+1))
					THEN  TRY(nextac i)) )
  in  tac  end;

  

(*repeated resolution applied to the designated goal*)
fun reresolve_tac rules = 
  let val restac = RESOLVE_THEN rules;  (*preprocessing done now*)
      fun gtac i = restac gtac i
  in  gtac  end; 


(*tries the theorems repeatedly before the last-resort-theorems. *)
fun repeat_goal_tac thm_packs = 
  let val restac  =    RESOLVE_THEN (flat (map fst thm_packs))
      and lastrestac = RESOLVE_THEN (flat (map snd thm_packs));
      fun gtac i = restac gtac i  ORELSE  lastrestac gtac i
  in  gtac  end; 


(*Tries safe theorems only*)
fun safe_goal_tac thm_packs = reresolve_tac (flat (map fst thm_packs));


(*Tries a safe theorem or else a last-resort theorem .
  Single-step for tracing a proof. *)
fun step_tac thm_packs = 
  let val restac  =    RESOLVE_THEN (flat (map fst thm_packs))
      and lastrestac = RESOLVE_THEN (flat (map snd thm_packs));
      fun stop_tac i = no_tac;
      fun gtac i = restac stop_tac i  ORELSE  lastrestac stop_tac i
  in  gtac  end; 



(* Tactic for reducing a goal, using Predicate Calculus rules.
   A decision procedure for Propositional Calculus, it is incomplete
   for Predicate-Calculus because of all_left_thin and exists_right_thin.  
   Fails if it can do nothing.      *)
val pc_tac = repeat_goal_tac [triv_pack, LK_pack];


(* Rule for duplication on the right.
   Useful since some rules are not complete.*)
val duplicate_right = prove_goal LK_Rule.thy 
    "[| $H |- $E, P, $F, P |] ==> [| $H |- $E, P, $F |]"
 (fn asms=>
  [ (cut_tac "P" 1),
    (REPEAT (resolve_tac (asms@[basic]) 1)) ]);


(* Rule for duplication on the left.*)
val duplicate_left = prove_goal LK_Rule.thy 
    "[| $H, P, $G, P |- $E |] ==> [| $H, P, $G |- $E |]"
 (fn asms=>
  [ (cut_tac "P" 1),
    (REPEAT (resolve_tac (asms@[basic]) 1)) ]);


(* Tactic for duplication of P on the right*)
fun duplicate_right_tac sP i : tactic = 
  res_inst_tac [ ("P",sP,Aform) ] duplicate_right i;


(* Tactic for duplication of P on the left*)
fun duplicate_left_tac sP i : tactic = 
  res_inst_tac [ ("P",sP,Aform) ] duplicate_left i;

end;
end;
