(*  Title: 	HOL/list
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

List type: defined via Tarski's Theorem.

STILL NEEDS WELL-FOUNDED 'TAIL' RELATION for defining recursive functions
*)


val list_const_decs = 
  [ (["listfun"],	[Atype,Aterm]--->Aterm),
    (["list"],  	Atype-->Atype ), 
    (["Nil"],  		Atype-->Aterm ), 
    (["Cons"],		[Atype,Aterm,Aterm]--->Aterm ) ];


val list_thy = Thm.extend_theory wf_thy  "list" 
    ([], list_const_decs)
  [ ("listfun_def",  
     "listfun(A,F) == lam S: nat*A->bool. term \
\		([ S = Nil(A) : nat*A->bool ]  \
\	| EXISTS x:A. EXISTS T: nat*A->bool. \
\		T <: F & [ S = Cons(A,x,T) : nat*A->bool ])"),

    (*written eta-expanded to avoid matching problems*)
    ("list_def", 
     "list(A) == {S: nat*A->bool. S <: fix(nat*A->bool, %(u)listfun(A,u))}"),

    ("Nil_def",  "Nil(A) == lam nx: nat*A . False" ),
    ("Cons_def", 
     "Cons(A,a,l) == lam nx: nat*A. \
\      	term(   ([ nx = <0,a> : nat*A ])    \
\	        | (EXISTS m:nat. EXISTS x:A. <m,x> <: l & \
\			  [ nx = <Succ(m),x> : nat*A ]))" ) ];


local val ax = Thm.get_axiom list_thy
in  val listfun_def = ax"listfun_def"
    and list_def = ax"list_def"
    and Nil_def  = ax"Nil_def"
    and Cons_def = ax"Cons_def"
end;


val list_defs = [Nil_def, Cons_def, listfun_def, list_def];


val Nil_reptype = prove_goal list_thy "[| Nil(A): nat*A -> bool |]"  
 (fn asms=>
  [ (rewrite_goals_tac list_defs),
    (REPEAT (resolve_tac type_rls 1)) ]);

val Cons_reptype = prove_goal list_thy
    "[| a: A |] ==> [| l: nat*A->bool |] ==> [| Cons(A,a,l) : nat*A->bool |]"  
 (fn asms=>
  [ (rewrite_goals_tac list_defs),
    (REPEAT (resolve_tac type_rls 1)) ]);

val asms = goal list_thy
    "[| F : (nat*A->bool)->bool |] ==> \
\    [| listfun(A,F) : (nat*A->bool)->bool |]";
by (rewrite_goals_tac [listfun_def]);
by (Pc.typechk_tac asms);
val listfun_type = result();


val asms = goal list_thy
    "monotone(nat*A->bool, nat*A->bool, listfun(A))";
by (rewrite_goals_tac [monotone_def, listfun_def]);
by (Class.fast_tac [Nil_reptype,Cons_reptype] 1);
val listfun_mono = result();


val asms = goal list_thy
    "[| [ fix(nat*A->bool, listfun(A)) \
\       = listfun(A, fix(nat*A->bool, listfun(A))) : (nat*A->bool)->bool ] |]";
by (REPEAT (ares_tac (asms @ 
	[tarski_theorem, listfun_mono, listfun_type]) 1));
val fix_listfun_unfold = result();



(** Typing of Nil *)

val asms = goal list_thy
    "[| Nil(A): list(A) |]";
by (rewrite_goals_tac [list_def]);
by (REPEAT (resolve_tac [subtype_intr,Nil_reptype] 1));
by (resolve_tac [fix_listfun_unfold RS subst] 1);
by (rewrite_goals_tac [listfun_def]);
by (Class.fast_tac [refl,Nil_reptype,Cons_reptype] 1);
val Nil_type = result();


(** Typing of Cons *)

val asms = goal list_thy
    "[| a: A |] ==> [| l: list(A) |] ==> [| Cons(A,a,l) : list(A) |]";
by (rewrite_goals_tac [list_def]);
by (REPEAT (resolve_tac ([subtype_intr,Cons_reptype] 
			@ subtype_rules [list_def] asms @ asms) 1));
by (resolve_tac [fix_listfun_unfold RS subst] 1);
by (rewrite_goals_tac [listfun_def]);
by (Class.fast_tac ([refl,Nil_reptype,Cons_reptype] 
			@ subtype_rules [list_def] asms @ asms) 1);
val Cons_type = result();


(** Freeness of Nil and Cons *)

(*Lemma gives directional use of 'subset' information*)
val list_distinct_lemma = prove_goal list_thy
    "[| subset(nat*A, Cons(A,a,l), Nil(A)) |] ==>    [| a: A |] ==>  \
\    [| P |]"
 (fn asms=>
  [ (resolve_tac (reslist(asms, 1, subset_elim)) 1),
    (rewrite_goals_tac [Nil_def, Cons_def, list_def]),
    (REPEAT (Class.step_tac ([refl]@asms) 1  ORELSE  assume_tac 1)) ]);

val list_distinct = prove_goal list_thy
    "[| [ Nil(A) = Cons(A,a,l) : list(A) ] |] ==>    [| a: A |] ==>  \
\    [| P |]"
 (fn asms=>
  [ (resolve_tac (reslist(subtype_rules [list_def] asms, 1, equal_elim)) 1),
    (eresolve_tac [list_distinct_lemma] 1),
    (resolve_tac asms 1) ]);


(*** Cons(A,x,l) is injective in x*)

val Cons_inject1_lemma = prove_goal list_thy
    "[| subset(nat*A, Cons(A,a,l), Cons(A,a',l')) |] ==> \
\    [| a : A |] ==> [| a': A |] ==>  \
\    [| [ a = a': A ] |]"
 (fn asms=>
  [ (resolve_tac (reslist(asms, 1, subset_elim)) 1),
    (rewrite_goals_tac [Nil_def, Cons_def]),
    (REPEAT    (eresolve_tac [asm_rl, nat_distinct, pair_inject] 1 
	 ORELSE Class.step_tac ([refl]@asms) 1)) ]);

val Cons_inject1 = prove_goal list_thy
    "[| [ Cons(A,a,l) = Cons(A,a',l') : list(A) ] |] ==> \
\    [| a : A |] ==> [| a': A |] ==>  \
\    [| [ a = a': A ] |]"
 (fn asms=>
  [ (resolve_tac (reslist(subtype_rules [list_def] asms, 1, equal_elim)) 1),
    (eresolve_tac [Cons_inject1_lemma] 1),
    (REPEAT (resolve_tac asms 1)) ]);

(*** Cons(A,x,l) is injective in l*)

val succ_refl = read_instantiate HOL_Rule.sign [("a","<Succ(?x),?y>",Aterm)] refl;

val Cons_inject2_lemma = prove_goal list_thy
    "[| subset(nat*A, Cons(A,a,l), Cons(A,a',l')) |] ==> \
\    [| a : A |] ==> [| a': A |] ==> [| l: nat*A -> bool |] ==> \
\    [| subset(nat*A, l, l') |]"
 (fn asms=>
  [ (resolve_tac [subset_intr_prod] 1),
    (resolve_tac (reslist(asms, 1, subset_elim)) 1),
    (rewrite_goals_tac [Nil_def, Cons_def]),
    (REPEAT    (Class.step_tac ([succ_refl]@asms) 1 
	ORELSE  eresolve_tac [asm_rl, sym RS nat_distinct, 
			Succ_inject, pair_inject, subst] 1)) ]);


val Cons_inject2 = prove_goal list_thy
    "[| [ Cons(A,a,l) = Cons(A,a',l') : list(A) ] |] ==> \
\    [| a : A |] ==> [| l : list(A) |]  ==>  \
\    [| a': A |] ==> [| l': list(A) |]  ==>  \
\    [| [ l = l': list(A) ] |]"
 (fn asms=>
 let val srls = subtype_rules [list_def] asms
 in
  [ (resolve_tac (reslist(srls, 1, equal_elim)) 1),
    (rewrite_goals_tac [list_def]),
    (resolve_tac [equal_intr RS subtype_abs_equal] 1),
    (REPEAT     (eresolve_tac [Cons_inject2_lemma] 1 
	 ORELSE  resolve_tac (srls@asms) 1)) ]
 end);


val Cons_inject = prove_goal list_thy
  "[| [ Cons(A,a,l) = Cons(A,a',l') : list(A) ] |] ==> \
\  [| a : A |] ==> [| l : list(A) |]  ==>  \
\  [| a': A |] ==> [| l': list(A) |] ==> \
\  ([| [ a = a': A ] |] ==> [| [ l = l': list(A) ] |] ==> [| R |]) ==> \
\  [| R |]"
 (fn asms=>
  [ (REPEAT (TRY (resolve_tac [Cons_inject1,Cons_inject2] 1) 
	 THEN (resolve_tac asms 1))) ]);


(*** Induction ***)

(*Elimination rule: used to derive list induction*)
val asms = goal list_thy
    "[| l <: listfun(A,F) |] ==> \
\    [| P(Nil(A)) |] ==> \
\    (!(x,y)[| x: A |] ==> [| y: nat*A->bool |] ==> [| y <: F |] ==> \
\	      [| P(Cons(A,x,y)) |]) ==>    \
\    [| l: nat*A->bool |] ==> \
\    [| P(l) |]";
by (cut_facts_tac asms 1);
by (rewrite_goals_tac [listfun_def]);
by (REPEAT (Pc.step_tac asms 1
      ORELSE  eresolve_tac [cla_elim, subst] 1));
val listfun_elim = result();



val asms = goal list_thy
    "[| l: list(A) |] ==> \
\    [| P(Nil(A)) |] ==>    \
\    (!(x,y)[| x: A |] ==> [| y: list(A) |] ==> [| P(y) |] ==> \
\	      [| P(Cons(A,x,y)) |]) ==>    \
\    [| P(l) |]";
val rasms = map (rewrite_rule [list_def]) asms;
by (res_inst_tac [("a","l",Aterm)] tarski_induction 1);
by (REPEAT_FIRST (ares_tac (reslist(rasms,1,subtype_elim1) @
		 	 reslist(rasms,1,subtype_elim2) @
		 	 [listfun_type,listfun_mono])));
by (REPEAT_FIRST (eresolve_tac [listfun_elim]
	ORELSE'  Class.step_tac ([subtype_intr,listfun_type,fix_type]@rasms)));
val list_induct = result();


(*Induction on the list expression "l"*)
fun list_induct_tac (sl: string) i = 
  res_inst_tac [ ("l",sl,Aterm) ] list_induct i;
