(*  Title: 	tt-syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)

(*Type Theory parsing and printing*)


(*Arities of expressions, types, hypotheses, judgements*)
val Aexp = Ground "exp" : arity;
val Atype = Ground "type" : arity;
val Ahyps = Ground "hyps";
val Ajudge = Ahyps --> Agoal;
 

(*judgements have the form type(A,H), elem(a,A,H), ...
  to allow general rules on hypotheses, judge(ANY,H)
  try instead  judge(type(A), H),  judge(elem(a,A), H), ... 
  parsing such a rule would require special treatment *)

(*Arities of uvariables.  Tree will be badly balanced!
  use Tarjan/Sleator data structure, JACM 1985
  or use algorithm for creating an optimal search tree*)
val tt_lextab : lexsymbol symbol_table = symtab_of_declist
  [ (*Variables*)
    (["A",  "B",  "C",  "D",  "E"],	IdentSy(0, Atype)),
    (["A'", "B'", "C'", "D'", "E'"],	IdentSy(0, Aexp-->Atype)),
    (["A''","B''","C''","D''","E''"],   IdentSy(0, [Aexp,Aexp]--->Atype)),
    (["a",  "b",  "c",  "d",  "e",  
      "f",  "k",  "p", "q"],		IdentSy(0, Aexp)),
    (["a'", "b'", "c'","d'","e'","f'"],	IdentSy(0, Aexp-->Aexp)),
    (["a''","b''","c''","d''","e''","f''"],  IdentSy(0, [Aexp,Aexp]--->Aexp)),
    (["H", "G"],	IdentSy(0, Ahyps)),

    (explode"uvwxyz",	IdentSy(0, Aexp)),
 
    (*constants for the judgements and hypotheses*)
    (["Type"],                 ConstSy(Atype -->Ajudge)),
    (["Eqtype"],       ConstSy([Atype,Atype]--->Ajudge)),
    (["Elem"],         ConstSy([Aexp, Atype]--->Ajudge)),
    (["Eqelem"], ConstSy([Aexp, Aexp, Atype]--->Ajudge)),
    (["reduce"], ConstSy([Aexp, Aexp, Atype]--->Ajudge)),

    (["Hempty"], ConstSy(Ahyps)),
    (["Hadd"],   ConstSy([Ahyps,Aexp,Atype]--->Ahyps)),

    (*Keywords*)
    (["type", ":", "=", "<", ">", ".", "PROD", "SUM", "lam"],	DelimSy),

    (*Infixes, excepting defined ones*)
    (["`"],	InfixSy([Aexp,Aexp]--->Aexp,	49,	50)) ,	(*left assoc*)
    (["+"],	InfixSy([Atype,Atype]--->Atype,	20,	19)) ,	(*right*)

    (["N"],		ConstSy(Atype)),
    (["0"],		ConstSy(Aexp)),
    (["succ"],		ConstSy(Aexp-->Aexp)),
    (["rec"],		ConstSy([Aexp, Aexp, [Aexp,Aexp]--->Aexp] ---> Aexp)),

    (["Prod"],		ConstSy([Atype, Aexp-->Atype] ---> Atype)),
    (["lambda"],	ConstSy((Aexp-->Aexp) --> Aexp)),

    (["Sum"],		ConstSy([Atype, Aexp-->Atype] ---> Atype)),
    (["pair"],		ConstSy([Aexp,Aexp]--->Aexp)),
    (["split"],		ConstSy([Aexp, [Aexp,Aexp]--->Aexp] ---> Aexp)),

    (["inl","inr"],	ConstSy(Aexp-->Aexp)),
    (["when"],		ConstSy([Aexp, Aexp-->Aexp, Aexp-->Aexp] ---> Aexp)),

    (["Eq"],		ConstSy([Atype,Aexp,Aexp]--->Atype)),
    (["eq"],		ConstSy(Aexp)),

    (["F","T"],	ConstSy(Atype)),
    (["contr"],		ConstSy(Aexp-->Aexp)),
    (["tt"],		ConstSy(Aexp)) ];


(*judgements have the form type(A,H), elem(a,A,H), ...
  to allow general rules on hypotheses, judge(ANY,H)
  try instead  judge(type(A), H),  judge(elem(a,A), H), ... 
  parsing such a rule would require special treatment *)

val type_con	= get_const tt_lextab "Type"
and eqtype_con	= get_const tt_lextab "Eqtype"
and elem_con	= get_const tt_lextab "Elem"
and eqelem_con	= get_const tt_lextab "Eqelem"
and Hempty	= get_const tt_lextab "Hempty"
and Hadd	= get_const tt_lextab "Hadd";


(*the hypothesis gotten by adding an assumption to the end;    H, a:A *)
fun add_assum (H, (a,A)) = Hadd$H$a$A;

val Prod   = get_const tt_lextab "Prod"
and lambda = get_const tt_lextab "lambda"
and Sum    = get_const tt_lextab "Sum"
and pair   = get_const tt_lextab "pair";


(*Allow any bound variable name in PROD, SUM, lam, but not %(...).
  It gets expression arity (Aexp)  *)
fun parse_boundvar (lxs0 as ((s, IdentSy(idx,ary)) :: lxs)) =
      if ary=Aexp then (string_of_xname(s,idx), lxs)
      else  raise parserr with 
          ("Bound variable should have expression arity", lxs0)
  | parse_boundvar ((s, OtherSy) :: lxs) = (s,lxs)
  | parse_boundvar lxs = 
      raise parserr with ("Variable expected", lxs);

 
(*Type theory syntax has type operators  PROD x:A. B     SUM x:A. B 
  canonical functions lam x.e,   and pairs <a,b>  *)     
fun parse_tt_term1 boundids =
  let fun parse_tyop (tyop,boundids) lxs =
	let val (s,lxs') = (parse_boundvar thenkey":") lxs;
	    val ((A,B), lxs'') =
	     (parse_tt_term1 boundids  thenkey"."  
	      thenp parse_tt_term1 (s::boundids)) lxs'
	in (tyop $ A $ Abs(s,Aexp,B), lxs'') end;
      fun parsett (boundids: string list) = fn
	 (("PROD",DelimSy) :: lxs) => parse_tyop (Prod,boundids) lxs
       | (("SUM",DelimSy)  :: lxs) => parse_tyop (Sum, boundids) lxs
       | (("lam",DelimSy)  :: lxs) => 
	   let val (s,lxs') = (parse_boundvar thenkey".") lxs;
	       val (b, lxs'') = parse_tt_term1 (s::boundids) lxs'
	   in (lambda $ Abs(s,Aexp,b), lxs'') end
       | ("<",DelimSy) :: lxs =>
	   let val ((a,b), lxs') =
		(parse_tt_term1 boundids  thenkey","  thenp
		 parse_tt_term1 boundids  thenkey">") lxs
	   in (pair$a$b, lxs') end
       | (s,OtherSy) :: lxs => (check_bound(s,0,Aexp,boundids), lxs)
       | lxs => raise parserr with ("Illegal expression", lxs)
  in  genparse parsett (0,boundids)  end;


val parse_tt_term: lexeme list -> term * lexeme list = parse_tt_term1 [];


(* hyp_tail ::= { "," term ":" term }  *)
fun parse_assum_tail (H,lxs) : term * lexeme list =
  let val parse_assum = parse_tt_term   thenkey ":"  thenp parse_tt_term;
      val (hyps, lxs') = parepeat (",", parse_assum) lxs
  in  (itlist_left add_assum (H, hyps), lxs')  end;


(*Parse a non-empty hypothesis, either a:A{,...} or H{,...}  *)
fun parse_hyp lxs = 
  case  parse_tt_term lxs  of
      (a, (":",DelimSy) :: lxs1) =>
	let val (A,lxs2) = parse_tt_term lxs1
	in  parse_assum_tail (Hadd$Hempty$a$A, lxs2)  end
    | (H,lxs1) => parse_assum_tail (H,lxs1);


(* hypothesis ::= "[" hypothesis "]"  |  empty  *)
fun parse_hyp_opt (("[", DelimSy)::lxs) = (parse_hyp thenkey "]") lxs
  | parse_hyp_opt lxs = (Hempty, lxs);


(*Judgements without hypotheses*)
fun parse_judge lxs : term * lexeme list =
  case  parse_tt_term lxs  of
      (A, ("type",DelimSy) :: lxs1) =>  (type_con$A, lxs1)
    | (a, (":",DelimSy) :: lxs1) =>
	let val (A,lxs2) = parse_tt_term lxs1
	in  (elem_con$a$A, lxs2)  end
    | (a, ("=",DelimSy) :: lxs1) =>
	(case  parse_tt_term lxs1  of
	     (b, (":",DelimSy) :: lxs2) =>
	       let val (A,lxs3) = parse_tt_term lxs2
	       in  (eqelem_con$a$b$A, lxs3)  end
	   | (B, lxs2) =>  (eqtype_con$a$B, lxs2))
    | all => all;  (*an arbitrary term, not standard syntax*)


(* theorem ::= judgement hypothesis_opt *)
fun parse_thm lxs : term * lexeme list = 
  let val ((tm,hyp), lxs') = (parse_judge thenp parse_hyp_opt) lxs
  in  (check_arity_term Agoal (tm$hyp), lxs')    end;




(*Printing of Type Theory terms and judgements*)


(*Special treatment of Sum pair Prod lambda Eq 
    pairs are a special case: they are always enclosed in < > *)
fun print_tt_term1 lextab (all as (outleft,outright,boundids)) =
  let val printt = print_tt_term1 lextab;
      val default_print = print_functl (printt, lextab) all;
      val variant = bound_variant boundids;
      fun sround pfn = 
	    if outright>0  then  (prs"(";  pfn();  prs")")  else pfn(); 
      fun print_tyop (pre,id,A,B) () =
	 (prs (pre ^ " " ^ id ^ ":");
	  printt (5, 5,   boundids)  A;  prs ". ";
	  printt (0, 0, id::boundids) B)
  in fn
      Const("Sum",_)  $ A $ Abs(id,_,B) => 
	  sround(print_tyop("SUM",  variant id, A, B))
    | Const("Prod",_) $ A $ Abs(id,_,B) => 
	  sround(print_tyop("PROD", variant id, A, B))
    | Const("lambda",_) $ Abs(id,_,b) => 
	  let val id' = variant id
	  in sround(fn() => (prs ("lam " ^ id' ^ ". ");
	    	   	     printt (0, 0, id'::boundids) b))
	  end
    | Const("Eq",_) $ A $ a $ b =>
	sround(fn() => (printt (0, 5, boundids) a;   prs" == ";
		        printt (5, 5, boundids) b;   prs" : ";
	 	        printt (5, 0, boundids) A))
    | Const("pair",_) $ a $ b =>  
	(prs"<";  
	 printt (0, 0, boundids) a;   prs",";
	 printt (0, 0, boundids) b;   prs">") 
    | tm => default_print tm
  end;


fun print_tt_term lextab : term->unit = print_tt_term1 lextab (0, 0, []);


(*Print a non-empty hypothesis (could be a single variable like ?H)*)
fun print_hyp lextab  = fn
  (Const("Hadd",_) $ H $ a $ A) =>
     (case H of  Const("Hempty",_)   =>  ()
              |  _   =>  (print_hyp lextab H;  prs ", ");
      print_tt_term lextab a;  prs ": "; print_tt_term lextab A)
  | H =>  print_tt_term lextab H;


(*print a judgement, no hypotheses*)
fun print_judge lextab = fn
    (Const("Type",_)   $ A) =>  (print_tt_term lextab A;  prs" type")
  | (Const("Eqtype",_) $ A $ B) =>
      (print_tt_term lextab A;  prs" = ";  print_tt_term lextab B)
  | (Const("Elem",_)   $ a $ A) =>
      (print_tt_term lextab a;  prs" : ";  print_tt_term lextab A)
  | (Const("Eqelem",_) $ a $ b $ A) => 
      (print_tt_term lextab a;  prs" = ";  print_tt_term lextab b;  
       prs" : ";  print_tt_term lextab A)
  | tm => print_tt_term lextab tm;


(*Print as  a:A [ x:B, ...], nicer for goal trees*)
fun print_goal lextab (P $ Const("Hempty",_)) =  (print_judge lextab P)
  | print_goal lextab (P $ H) =
	 (print_judge lextab P;  prs" [ ";  print_hyp lextab H;  prs" ] ")
  | print_goal lextab tm = raise term_error with ("print_goal", [tm]);
