(*  Title: 	print
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)

(*print.ml  Printing of arities and terms
  extend_sign1 should check: NO PARAMS IN DEFINITIONS*)


fun print_arity (Ground name) = prs name 
  | print_arity ary           =  (prs"(";  print_fun ary;  prs")")
(*suppresses brackets for right associativity of -->*)
and print_fun (Ground name) = prs name 
  | print_fun (ary1-->ary2) = (print_arity ary1;  prs"-->";  print_fun ary2);


(*print a list surrounded by the brackets lpar and rpar, with comma separator
  print nothing for empty list*)
fun print_list (lpar, rpar, pre: 'a -> unit)  (l : 'a list) = 
    let fun prec(x) = (prs",";  pre(x)) 
    in  case l of
	    [] => () 
	  | x::l =>  (prs lpar;  pre x;  seq prec l;  prs rpar)
    end;


(*print a list of items separated by newlines*)
fun print_list_ln (pre: 'a -> unit)  : 'a list -> unit = 
    seq (fn x => (pre x;  writeln""));


(*Increment a string like a base 26 number.  For making variants.
  Not very clever: if last char is "z", appends an "a" 
  rather than bumping the previous chars*)
fun bump_string s : string =
  case (rev (explode s)) of
    [] => "a"
  | d::cs => 
      if ord(d) >= ord("z")  then  s^"a"  
      else  implode (rev (chr(ord(d)+1) :: cs));

(*take variant of name of bound variable :   (ids,x)  ---->   x''*)
fun bound_variant bound_ids x : string =
  let fun vary x =
        if (x smem bound_ids) then  vary (bump_string x)  else  x
  in  vary (if x="" then "u" else x)  end;


(* (ids,x)  ---->   x''::ids    this reverses the ids, as required*)
fun add_bound_id (bound_ids, x) : string list =
    (bound_variant bound_ids x) :: bound_ids;

(*print the bound variable numbered "bno", using list of bound vars*)
fun print_bound_var (bound_ids: string list, bno:int) : unit =
    prs (nth_elem(bno,bound_ids)) 
    handle list => (prs"B.";  print_int (bno - length bound_ids));


fun is_print_atom (Abs _) = false
  | is_print_atom (_ $ _)  = false
  | is_print_atom   _      = true;


type printfun = (int * int * string list) -> term -> unit;


(* Printing.  Primes bound variables if clashes exist
   Invents names for anonymous bound variables. 
   For f(a,b), looks up constant's name (f) in lextab.
       If found, prints infix notation.*)
fun print_functl(recprint: printfun, lextab: lexsymbol symbol_table)
  : printfun =
  fn (outleft,outright,bound_ids) =>  fn tm =>
  let fun print_atom tm =
	if (is_print_atom tm) then  recprint (0,0,bound_ids) tm
	else (prs"(";  recprint (0,0,bound_ids) tm;  prs")");
      fun print_comb tm =
	let val (rator,rands) = strip_comb tm
	in  print_atom rator;
	    print_list ("(", ")", recprint (0,0,bound_ids)) rands  
	end
  in case tm of
       Const (id,_)   =>  prs id
     | Param (pname,_) =>  prs (string_of_xname pname)
     | Bound bno => print_bound_var (bound_ids,bno) 
     | Var (uname,_) =>  prs ("?" ^ string_of_xname uname)
     | Abs _ => 
	 let val body = strip_abs_body tm 
	     and ids = map fst (strip_abs_vars tm);
	     val bound_ids' = itlist_left add_bound_id (bound_ids, ids);
	     val newids = rev (front (length ids, bound_ids'))
	 in print_list ("%(", ")", prs) newids;
	    recprint (0, outright, bound_ids') body    end 
     | Const(id,_) $ A $ B =>
	(case (slookup(lextab,id)) of
	    Some (InfixSy(_,inleft,inright)) =>
	      if inleft<=outleft orelse inright<=outright 
	      then (prs"(";  recprint (0,inleft,bound_ids) A;  prs (" "^id^" ");
		    recprint (inright,0,bound_ids) B;  prs")")
	      else (recprint (outleft,inleft,bound_ids) A;  prs (" "^id^" ");
		    recprint (inright,outright,bound_ids) B) 
	  | _ => print_comb tm)    (*None or not infix*)
     | _ $ _ =>  print_comb tm
  end;



(*Signatures of theories.
  Lexical information, user definitions, parsing and printing functions.*)
datatype signat = Signat of
  {lextab: lexsymbol symbol_table,
   defines: (string*term) list,
   parse_tm: lexeme list -> term * lexeme list,
   parse_thm : lexeme list -> term * lexeme list,
   print_tm: lexsymbol symbol_table -> term -> unit,
   print_thm : lexsymbol symbol_table -> term -> unit};


exception definition : string;


(*Look up a definition*)
fun get_definition (defines: (string*term) list)  id  :  string * term=
      case sassoc(defines,id) of
          None => raise definition with "No definition for "^id
        | Some tm => (id,tm);


(*get the subset of definitions named by the ids*)
fun get_definitions defines ids : (string*term) list =
  map (get_definition defines) ids;


fun defs_of_sign (Signat{defines,...}) ids : (string*term) list =
  get_definitions defines ids;


(*read, print functions that work via the signature*)

fun read_term (Signat{lextab,parse_tm,...}) s : term =
  parse_end (parse_tm (lex_string (lextab,s)));

fun read_theorem (Signat{lextab,parse_thm,...}) s : term =
  parse_end (parse_thm (lex_string (lextab,s)));

fun print_term (Signat{print_tm,lextab,...}) = print_tm lextab;

fun print_theorem (Signat{print_thm,lextab,...}) = print_thm lextab;

(*print an environment*)
fun print_env sign env : unit = 
  let fun prpair (name, tm) =
	    (prs ("?" ^ string_of_xname name ^ " -> ");
	     print_term sign (norm_term env tm);
	     prs " : ";  print_arity (arity_of tm));
      val Envir{asol,...} = env
  in  print_list_ln prpair (alist_of_olist asol)  end;


fun arity_of_lex (ConstSy ary) = ary
  | arity_of_lex (InfixSy (ary,_,_)) = ary
  | arity_of_lex (IdentSy (_,ary)) = ary
  | arity_of_lex (VarSy (_,ary)) = ary
  | arity_of_lex _ = raise definition with "illegal lexymbol in definition"; 


(*add the definition id=stm to the signature, extending lextab and defines
  tests: arities are the same, term contains no free variables*)
fun extend_sign1 (sign, (id,lx,stm)) =
  let val tm = 	read_term sign stm;
      val Signat{lextab,defines,parse_tm,parse_thm,print_tm,print_thm} = sign
  in if arity_of_lex lx = arity_of tm  andalso  null(pvars_of_terms[tm])
     then Signat{lextab= supdate_new ((id,lx), lextab),
	    defines= (id,tm) :: defines,
	    parse_tm=parse_tm, parse_thm=parse_thm, 
	    print_tm=print_tm, print_thm=print_thm}
     else raise definition with "bad definition for " ^ id
  end;

(*add a list of definitions*)
val extend_sign : signat * (string*lexsymbol*string) list -> signat =
  itlist_left extend_sign1;
