(*  Title: 	tt-resolve
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)

(*Type Theory resolution tactics
*)

val form_rls = [N_form, Prod_form, Sum_form, Plus_form, 
		Eq_form, F_form, T_form];

val form_long_rls = [Prod_form_long, Sum_form_long, Plus_form_long, Eq_form_long];

(*OMITTED: Eq_intr, because its premise is an eqelem, not an elem*)
val intr_rls = 
    [N_intr0, N_intr_succ, Prod_intr, Sum_intr, 
     Plus_intr_inl, Plus_intr_inr, T_intr];

val intr_long_rls = 
    [N_intr_succ_long, Prod_intr_long, Sum_intr_long, 
     Plus_intr_inl_long, Plus_intr_inr_long];

val assume_rls = [assume_elem, thin_elem];


(*OMITTED: Eq_elim, because its conclusion is an eqelem,  not an elem
           T_elim, because it does not involve a constructor *)
val elim_rls = [N_elim, Prod_elim, Sum_elim, Plus_elim, F_elim];

val elim_long_rls =
    [N_elim_long, Prod_elim_long, Sum_elim_long, Plus_elim_long, F_elim_long];

(*eq_comp,  T_comp not included because they cause rewriting to loop:
  ?p = un = un = un = ... *)
val comp_rls = 
    [N_comp0, N_comp_succ, 
     Prod_comp, Sum_comp, Plus_comp_inl, Plus_comp_inr];

val subst_rls = [subst_type, subst_elem, subst_type_long, subst_elem_long];

(*rules with conclusion a:A, an elem judgement*)
val element_rls = intr_rls @ elim_rls @ assume_rls;


(*   ?c = ?a : ?A     ?d = ?b : ?B'(?a)
     ----------------------------------
      <?c,?d> = <?a,?b> : Sum(?A,?B')
  Compare with standard version: B is applied to UNSIMPLIFIED expression! *)
val Sum_intr_long2 = resolvelist (sym_elem,
    [ (1, [Sum_intr_long]), (1, [sym_elem]), (2, [sym_elem]) ]);

val intr_long2_rls = 
    [N_intr_succ_long, Prod_intr_long, Sum_intr_long2, 
     Plus_intr_inl_long, Plus_intr_inr_long];



(*like the sequent rules:
p : Prod(A,C1)      a : A      b1(se) : B1(se) [ se : C1(a) ]
--------------------------------------------------------------
                    b1(p*a) : B1(p*a)                *)
val subst_prod_elim = resolve(subst_elem, 1, [Prod_elim]);


(*Makes a list of rules for grabbing the 0th, 1st, 2nd, ... assumption
  Resolves with accumulated result in first argument position;  
  resolving with result last causes exponential blow-up! *)
fun make_assume_n_rls n =
  let fun thinn (0,rl) = [ rl RES assume_elem ]
	| thinn (n,rl) = 
	    (rl RES assume_elem)  ::  thinn (n-1, rl RES thin_elem)
  in assume_elem :: thinn (n-1,thin_elem) end;


(*Can only handle up to 21 assumptions*)
val assume_n_rls = make_assume_n_rls 20;


fun elim_hyp_tac hyn = 
  let val hyprl = nth_elem(hyn, assume_n_rls);
      fun reshyp rl = rl RES hyprl
  in  filt_resolve_tac (map reshyp (elim_rls @ [Eq_elim,subst_prod_elim])) end;


(*Derived rules for eliminating (and discarding) the last assumption
  Could as well use (elim_hyp 0) as of resolution with assume_elem *)

val Sum_elim0 = resolvelist(Sum_elim,
                  [ (1, [assume_elem]), (2, [thin_elem2]), (1, form_rls)] );

val Plus_elim0 = resolvelist(Plus_elim,
                  [ (1, [assume_elem]), (2, [thin_elem1]),
		    (3, [thin_elem1]), (1, form_rls)]);

val Prod_elim0 = resolvelist(Prod_elim, [ (1, [assume_elem]), (1, form_rls)]);

val N_elim0 = resolvelist(N_elim, 
                  [ (1, [assume_elem]), (2, [thin_elem]),
		    (3, [thin_elem2]), (1, form_rls) ]);

val Eq_elim0 = resolvelist(Eq_elim, [ (1, [assume_elem]), (1, form_rls) ]);

val T_elim0 = resolvelist(T_elim,
		    [ (1, [assume_elem]), (2, [thin_elem]), (1, form_rls) ]);


(*the only equality version so far, used to derive bool_elim_long*)
val T_elim_long0 = resolvelist(T_elim_long,
		    [ (1, [refl_elem]),   (1, [assume_elem]),
		      (2, [thin_eqelem]), (1, form_rls) ]);

val F_elim0 = resolvelist(F_elim, [ (1, [assume_elem]), (1, form_rls) ]);

(*Eq_elim0 NOT INCLUDED*)
val elim0_rls = [N_elim0, Prod_elim0, Sum_elim0, Plus_elim0,
		 T_elim0, F_elim0];


(*"0" version of generalized product elimination rule*)
val subst_prod_elim0 = resolvelist(subst_prod_elim,
                  [ (1, [assume_elem]), (2, [thin_elem]),
		    (3, [thin_elem1]), (1, form_rls)]);


(*deterministic tactics*)
val form_tac = determ_resolve_tac (form_rls, 1);
val determ_assume_tac = determ_resolve_tac (form_rls@assume_rls, 2);

fun new_typechk_tac new_element_rls  : tactic =
    determ_resolve_tac (form_rls @ new_element_rls @ element_rls, 3)
    THEN  merge_premises_tac;

val typechk_tac = new_typechk_tac[];

fun new_equal_tac new_long_rls =
    determ_resolve_tac (form_rls @
	new_long_rls @ intr_long_rls @ elim_long_rls @ [refl_elem], 3)
    THEN  merge_premises_tac;

val equal_tac = new_equal_tac[];


(*Ignores goal if no rules appear unifiable
  Danger: if a rule appears unifiable but is not, then goal fails *)
fun determ_choose (rules,wlimit) prem =
    let val goodrules = filter_rules(wlimit+1, prem, rules);
	val nrules = length goodrules
    in  if nrules>wlimit  orelse  nrules=0  then  None  else Some goodrules
    end;


(*Ignores other than elem judgements
  filter_rules is faster than resolution in the branching test*)
fun elem_choose (elem_rls,wlimit) prem = case prem of
      (Const("Elem",_) $ _ $ _ $ _) =>
	 let val goodrules = filter_rules(wlimit+1, prem, elem_rls)
	 in  if length goodrules>wlimit  then  None   else Some goodrules
	 end
    | _ => None;


(*breakup by introduction and by elimination of assumptions
  useful for stripping quantifiers*)
val elintr_rls = elim0_rls @ intr_rls;

(*fancy backtracking tactics*)


(*note! the deterministic tactic form_tac does not throw away previous
  backtracking points:  backtracking is possible afterwards!*)
val elintr_tac =
    depth_resolve_tac (elem_choose(elintr_rls @ assume_rls, 4))
    THEN form_tac;

(*allow up to 3 rules because the 2 assume_rls always appear to apply*)
val depth_assume_tac = depth_resolve_tac (elem_choose(assume_rls, 3));


(*introduction rules work badly in stringtrees because 
  goals like ?a1:SUM(A,B') have a trivial head-string*)
val intr_tac = depth_resolve_tac (determ_choose(form_rls@intr_rls,1));


(*for simplification: prove a=c from c=b and a=b *)
val transym_elem = resolvelist(sym_elem, [ (1, [trans_elem]), (2, [sym_elem])]);


(*simplify the type in a goal
  ?a : ?A [ ?H ]     ?B = ?A [ ?H ]     
  ----------------------------------
          ?a : ?B [ ?H ]           *)
val equalsym_types = resolve(equal_types, 2, [sym_type]);


(*simplify a type by simplifying its argument
  intended for type variables (parameters)
  Maybe better to have a type simplifier using refl_type and formation rules
  ?a = ?c : ?A [ ?H ]     ?B'(s) type [ ?H, s: ?A ]     
---------------------------------------------------
	     ?B'(?a) = ?B'(?c) [ ?H ] 
s NOT IN> ?B' ?H                         *)
val subst_eqtyparg =
  resolve(read_constrain [ ("?D'","?B'") ] subst_type_long, 2, [refl_type]);


(*    [ "?a = ?b : ?A  [ ?H ]",  "?b = ?c : ?A [ ?H ]" ] ,
      --------------------------------------------------
		 "reduce(?a,?c,?A)  [ ?H ]"  *)
val red_trans = resolve(red_if_equal, 1, [trans_elem]);


(*Make a reduction rule for simplification.
  A goal a=c becomes b=c, by virtue of a=b *)
fun resolve_trans rl = resolve(trans_elem, 1, [rl]);


(*Simplification.....................
  COULD MINIMIZE GROWTH OF INDEXES BY MAKING MAXIDX=0 IN RED_TRANS, ETC.*)

(*resolve transym with each nontrivial equality goal*)
fun transym_choose prem = case prem of
      (Const("Eqelem",_) $ (_$_) $ (_$_) $ _ $ _) => Some [transym_elem]
    | (Const("Eqelem",_) $   _   $   _   $ _ $ _) => None
    | _ => None;

val transym_tac = depth_resolve_tac transym_choose;


(*converts each goal "e : Eq(A,a,b)" into "a=b:A" for simplification
  Includes other intro rules to avoid changing flexible goals.*)
val eqintr_tac = depth_resolve_tac (determ_choose(Eq_intr::intr_rls,1));

(*tactic to prepare for simplication by 'trans'ing each equality goal*)
val presimp_tac = eqintr_tac THEN transym_tac;



(*rule for simplifying subterms, for example
  ?a1 = ?b1 : N [ ?H ]     reduce(succ(?b1),?c,N) [ ?H ]     
  ----------------------------------------------------------
		succ(?a1) = ?c : N [ ?H ]    *)
fun subconv_rule rl = resolve(trans_red, 1, [rl]);


(*rule for resimplifying if possible, for example
  ?a: ?A [?H]   ?b: ?B'(?a) [?H]  ... ?c''(?a,?b) = ?d : ?C'(<?a,?b>) [?H]     
  --------------------------------------------------------------------------
	 reduce(split(<?a,?b>,?c''),?d,?C'(<?a,?b>)) [?H]        *)
fun resimp_rule rl = resolve(red_trans, 1, [rl]);


(*Equality elimination only takes place via Eq_elim0:
      the last assumption may be an equality
  Other elimination rules provide typechecking*)
val reduction_rls = map resolve_trans (comp_rls @ [Eq_elim0]);

val subconv_rls = map subconv_rule (intr_long2_rls @ elim_long_rls);


(*rules for simplification (top-level and subterms) *)
fun make_simp_rules (new_comp_rls, new_long_rls) =
   map resolve_trans new_comp_rls  @    reduction_rls @
   map subconv_rule new_long_rls   @    subconv_rls   @   [refl_elem];


val resimp_rls = map resimp_rule (comp_rls @ [Eq_elim0]);

(*rules for resimplification*)
fun make_resimp_rules new_comp_rls =
    map resimp_rule new_comp_rls  @  resimp_rls  @  [refl_red];

fun all_simp_rules (new_elem_rls, new_comp_rls, new_long_rls) =
    form_rls @ form_long_rls @ [refl_type] @
    new_elem_rls @ element_rls @
    make_simp_rules(new_comp_rls, new_long_rls) @
    make_resimp_rules new_comp_rls;

(*The form_long_rls and refl_type allow simplification of types.*)
fun new_simp_tac (new_elem_rls, new_comp_rls, new_long_rls, wlimit) =
    eqintr_tac THEN
    determ_resolve_tac
      (all_simp_rules(new_elem_rls, new_comp_rls, new_long_rls),  wlimit) 
    THEN determ_resolve_tac([red_if_equal], 1)
    THEN merge_premises_tac;

(*4 allows for a comp_ rule, Eq_elim0, a _long rule, and refl_elem*)
val simp_tac = new_simp_tac([],[],[],4);



(*induction on the term denoted by sp*)
fun rN_elim_tac (sp: string) gno : tactic =
  (resolve_tac [Eq_elim] gno   ORELSE  all_tac)  THEN
  rconstrain_res_tac [ ("?p",sp) ] N_elim gno;


(*substitution for a:A in goal*)
fun rsubst_tac (sa: string, sA: string) : int->tactic =
  rconstrain_res_tac [ ("?a",sa), ("?A",sA) ] subst_elem;
