(*  Title: 	tt-ex-add
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)

(*examples involving addition and multiplication
  tests the treatment of definitions and the simplifier
*)


(**********
  Addition
 **********)

(*typing of add: short and long versions*)

read_goal TypeThy "a #+ b : N [H]";
expand (unfold_goal_tac ["#+"]);
expand typechk_tac;
val add_typing = tidyrule(top_rule());

read_goal TypeThy "a #+ b = c #+ d : N  [H]";
expand (unfold_goal_tac ["#+"]);
expand equal_tac;
expand typechk_tac;
val add_typing_long = tidyrule(top_rule());


(*computation for add: 0 and successor cases*)

read_goal TypeThy "0 #+ b = b : N  [H]";
expand (unfold_goal_tac ["#+"]);
expand simp_tac; 
val add_comp0 = tidyrule(top_rule());

read_goal TypeThy "succ(a) #+ b = succ(a #+ b) : N  [H]";
expand (unfold_goal_tac ["#+"]);
expand simp_tac; 
val add_comp_succ = tidyrule(top_rule());


(*simplification*)
fun add_simp_tac new_comp_rls : tactic = 
  new_simp_tac([add_typing],
	       new_comp_rls @ [add_comp0,add_comp_succ],
	       [add_typing_long], 6);



(*Associative law for addition*)
read_goal TypeThy "(a #+ b) #+ c = a #+ (b #+ c) : N  [H]";
expand (rN_elim_tac "a" 1);
expand presimp_tac;
expand (add_simp_tac[]);  (*7 secs*)
val add_assoc = tidyrule(top_rule());


(*Commutative law for addition.
  Two lemmas are helpful: proving the law directly leads to a mess.*)

(*zero is right identity*)
read_goal TypeThy "b #+ 0 = b : N  [H]";
expand (rN_elim_tac "b" 1);
expand (add_simp_tac[]);  (*1 sec*)
val add_right0 = tidyrule(top_rule());


(*right successor lemma*)
read_goal TypeThy "a #+ succ(b) = succ(a #+ b) : N  [H]";
expand (rN_elim_tac "a" 1);
expand presimp_tac;
expand (add_simp_tac[]);  (*3 secs*)
val add_right_succ = tidyrule(top_rule());


read_goal TypeThy "a #+ b = b #+ a : N  [H]";
expand (rN_elim_tac "a" 1);
expand presimp_tac;
expand (add_simp_tac[add_right0,add_right_succ]);  (*3 secs*)
val add_commute = tidyrule(top_rule());




(****************
  Multiplication
 ****************)

(*typing of mult: short and long versions*)

read_goal TypeThy "a #* b : N [H]";
expand (unfold_goal_tac ["#*"]);
expand (new_typechk_tac[add_typing]);
val mult_typing = tidyrule(top_rule());


read_goal TypeThy "a #* b = c #* d : N  [H]";
expand (unfold_goal_tac ["#*"]);
expand (new_equal_tac[add_typing_long]);
expand typechk_tac;
expand (determ_resolve_tac ([thin_eqelem], 1));
val mult_typing_long = tidyrule(top_rule());


(*computation for mult: 0 and successor cases*)

read_goal TypeThy "0 #* b = 0 : N  [H]";
expand (unfold_goal_tac ["#*"]);
expand (add_simp_tac[]);
val mult_comp0 = tidyrule(top_rule());


read_goal TypeThy "succ(a) #* b = b #+ (a #* b) : N  [H]";
expand (unfold_goal_tac ["#*"]);
expand (add_simp_tac[]);
val mult_comp_succ = tidyrule(top_rule());


(*simplify using addition and multiplication laws*)
fun mult_simp_tac new_comp_rls : tactic = new_simp_tac
   ([add_typing,mult_typing],
    new_comp_rls @ [add_comp0,add_comp_succ, mult_comp0,mult_comp_succ],
    [add_typing_long,mult_typing_long], 8);



(*right annihilation in product*)
read_goal TypeThy "a #* 0 = 0 : N  [H]";
expand (rN_elim_tac "a" 1);
expand (mult_simp_tac[]);
val mult_right0 = tidyrule(top_rule());


(*swap round the associative law of addition*)
val multsoc_simp_tac = mult_simp_tac [ sym_elem RES add_assoc ];

(*right successor law for multiplication*)
read_goal TypeThy "a #* succ(b) = a #+ (a #* b) : N  [H]";
expand (rN_elim_tac "a" 1);
expand presimp_tac;
expand multsoc_simp_tac;  (*8 secs, leaves a goal*)
expand (determ_resolve_tac
         ([add_commute,mult_typing_long,add_typing_long]@
	  intr_long_rls@[refl_elem], 3));
expand determ_assume_tac;
val mult_right_succ = tidyrule(top_rule());


(*Commutative law for multiplication*)
read_goal TypeThy "a #* b = b #* a : N  [H]";
expand (rN_elim_tac "a" 1);
expand presimp_tac;
expand (mult_simp_tac [mult_right0, mult_right_succ]);  (*4 secs*)
val mult_commute = tidyrule(top_rule());


(*addition distributes over multiplication*)
read_goal TypeThy "(a #+ b) #* c = (a #* c) #+ (b #* c) : N  [H]";
expand (rN_elim_tac "a" 1);
expand presimp_tac;
expand multsoc_simp_tac; (*11 secs*)
val add_mult_dist = tidyrule(top_rule());


(*Associative law for multiplication*)
read_goal TypeThy "(a #* b) #* c = a #* (b #* c) : N  [H]";
expand (rN_elim_tac "a" 1);
expand presimp_tac;
expand (mult_simp_tac [add_mult_dist]);  (*7 secs*)
val mult_assoc = tidyrule(top_rule());



(************
  Difference
 ************

Difference on natural numbers, without negative numbers
  a - b = 0  iff  a<=b    a - b = succ(c) iff a>b
*)

(*typing of difference*)

read_goal TypeThy "a - b : N [H]";
expand (unfold_goal_tac ["-"]);
expand typechk_tac;
val diff_typing = tidyrule(top_rule());

read_goal TypeThy "a - b = c - d : N  [H]";
expand (unfold_goal_tac ["-"]);
expand equal_tac;
expand typechk_tac;
val diff_typing_long = tidyrule(top_rule());


(*computation for difference: 0 and successor cases*)

read_goal TypeThy "a - 0 = a : N  [H]";
expand (unfold_goal_tac ["-"]);
expand simp_tac; 
val diff_comp0 = tidyrule(top_rule());


read_goal TypeThy "a - succ(b) = rec(a - b, 0, %(z,w)z)  : N  [H]";
expand (unfold_goal_tac ["-"]);
expand simp_tac; 
val diff_comp_succ = tidyrule(top_rule());


(*simplification*)
val diff_simp_tac : tactic = 
  new_simp_tac([diff_typing], [diff_comp0,diff_comp_succ], [diff_typing_long],
		6);


read_goal TypeThy "0 - b = 0 : N  [H]";
expand (rN_elim_tac "b" 1);
expand intr_tac;
expand diff_simp_tac;  (*2 secs*)
val diff_0_eq_0 = tidyrule(top_rule());



read_goal TypeThy "succ(a) - succ(b) = a - b : N  [H]";
expand (resolve_tac [ trans_elem RES diff_comp_succ ] 1);
expand (rN_elim_tac "b" 3);
expand presimp_tac;
expand diff_simp_tac;  (*5 secs*)
val diff_succ_succ = tidyrule(top_rule());


(*no longer want diff_comp_succ: it unfolds to predecessor function*)
fun addiff_simp_tac new_comp_rls : tactic = 
  new_simp_tac([add_typing, diff_typing],
	       new_comp_rls @ [add_comp0,add_comp_succ,
			       diff_comp0,diff_0_eq_0,diff_succ_succ],
	       [add_typing_long, diff_typing_long],  6);



read_goal TypeThy "a - a = 0 : N  [H]";
expand (rN_elim_tac "a" 1);
expand (addiff_simp_tac[]);
val diff_self_eq_0 = tidyrule(top_rule());



(*addition is the inverse of subtraction;
  rule has the premise   b - a = 0 : N    i.e.    a >= b
  An example of rewriting with a quantified, implicative assumption.*)
read_goal TypeThy "b #+ (a - b) = a : N [H]";
expand (resolve_tac [Eq_elim] 1);
(*attach condition, then generalize goal by two Prod_elims
  Sadly we must specify a, else it gets assigned something random.
  Alternative: state the result of these eliminations as the starting goal,
    as a lemma.  But using this lemma requires the same eliminations.*)
expand (rconstrain_res_tac [ ("?A", "Eq(N, b - a, 0)"),
			     ("?a", "eq") ]   Prod_elim 1);
expand (rconstrain_res_tac [ ("?a", "a") ]   Prod_elim 1);
expand (rN_elim_tac "b" 1);
expand (resolve_tac intr_rls 3);  (*strip off the PROD in just this goal*)
expand (rN_elim_tac "w6" 4);
expand (resolve_tac [equalsym_types] 6);
expand (resolve_tac [equalsym_types] 5);
expand intr_tac;  (*strips PRODs, but not in the eqtype goals just created*)
expand (addiff_simp_tac[]);  (*10 secs: solves first 0 goal, simplifies others*)
expand intr_tac;  (*strips remaining PRODs*)
expand (resolve_tac [ F_elim RES zero_ne_succ ] 3);
expand (resolve_tac [ sym_elem RES Eq_elim0 ] 4);
(*prepare the quantified, implicative rewrite: use subst_prod_elim twice*)
expand (elim_hyp_tac 4 9);
expand (resolve_tac [subst_prod_elim0] 11);
expand (resolve_tac assume_rls 13);
expand (addiff_simp_tac[]);  (*8 secs*)
val add_inverse_diff = tidyrule(top_rule());



(*typing of absolute difference: short and long versions*)

read_goal TypeThy "a |-| b : N [H]";
expand (unfold_goal_tac ["|-|"]);
expand (new_typechk_tac[add_typing,diff_typing]);
val absdiff_typing = tidyrule(top_rule());

read_goal TypeThy "a |-| b = c |-| d : N  [H]";
expand (unfold_goal_tac ["|-|"]);
expand (new_equal_tac[add_typing_long, diff_typing_long]);
expand typechk_tac;
val absdiff_typing_long = tidyrule(top_rule());


read_goal TypeThy "a |-| a = 0 : N  [H]";
expand (unfold_goal_tac ["|-|"]);
expand (addiff_simp_tac[diff_self_eq_0]);
val absdiff_self_eq_0 = tidyrule(top_rule());


read_goal TypeThy "0 |-| a = a : N  [H]";
expand (unfold_goal_tac ["|-|"]);
expand (addiff_simp_tac[]);
val absdiff_comp0 = tidyrule(top_rule());


read_goal TypeThy "succ(a) |-| succ(b)  =  a |-| b : N  [H]";
expand (unfold_goal_tac ["|-|"]);
expand (addiff_simp_tac[]);
val absdiff_succ_succ = tidyrule(top_rule());


(*Note how easy using commutative laws can be?  ...not always... *)
read_goal TypeThy "a |-| b = b |-| a : N  [H]";
expand (unfold_goal_tac ["|-|"]);
expand (resolve_tac [add_commute] 1);
expand (new_typechk_tac [diff_typing]);
val absdiff_commute = tidyrule(top_rule());


(*lemma: if a+b=0 then a=0.   Surprisingly tedious*)
read_goal TypeThy "a = 0 : N  [H]";
expand (resolve_tac [Eq_elim] 1);
expand (rconstrain_res_tac [ ("?A", "Eq(N, a #+ b, 0)"),
			     ("?a", "eq") ]   Prod_elim 1);
expand (rN_elim_tac "a" 1);
expand (resolve_tac [equalsym_types] 3);
expand (add_simp_tac[]);
expand intr_tac;  (*strips remaining PRODs*)
expand (resolve_tac [ F_elim RES zero_ne_succ ] 4);
expand (resolve_tac [ sym_elem RES Eq_elim0 ] 5);
expand (new_typechk_tac [add_typing]);
val add_eq0_lemma = tidyrule(top_rule());


(*if  a |-| b = 0  then  a = b  
  proof: a-b=0 and b-a=0, so b = a+(b-a) = a+0 = a *)
read_goal TypeThy "a = b : N [H]";
expand (resolve_tac [ trans_elem RES sym_elem RES add_inverse_diff ] 1);
expand (resolve_tac [ trans_elem RES add_commute ] 4);
expand (resolve_tac [ trans_elem RES add_typing_long ] 6);
expand (resolve_tac [add_comp0] 8);
expand (resolve_tac [refl_elem] 7);
expand (resolve_tac [add_eq0_lemma] 6);
expand (resolve_tac [add_eq0_lemma] 3);
expand (resolve_tac [ trans_elem RES add_commute ] 5);
expand (identify_goals_tac (7,12));
expand merge_premises_tac;
expand (new_typechk_tac [diff_typing]);
expand (fold_tac ["|-|"]);
val absdiff_eq0 = tidyrule(top_rule());




(***********************
  Remainder and Quotient
 ***********************)



(*typing of MOD: short and long versions*)

read_goal TypeThy "a MOD b : N [H]";
expand (unfold_goal_tac ["MOD"]);
expand (new_typechk_tac [absdiff_typing]);
val MOD_typing = tidyrule(top_rule());

read_goal TypeThy "a MOD b = c MOD d : N  [H]";
expand (unfold_goal_tac ["MOD"]);
expand (new_equal_tac [absdiff_typing_long]);
expand typechk_tac;
expand (determ_resolve_tac ([thin_eqelem], 1));
val MOD_typing_long = tidyrule(top_rule());


(*computation for MOD: 0 and successor cases*)

(*       ?b : N [ ?H ]     
    -----------------------
    0 MOD ?b = 0 : N [ ?H ] 	*)
read_goal TypeThy "0 MOD b = 0 : N  [H]";
expand (unfold_goal_tac ["MOD"]);
expand (new_simp_tac([absdiff_typing],[],[],5)); 
val MOD_comp0 = tidyrule(top_rule());


(*  ?a : N [ ?H ]                       ?b : N [ ?H ]     
-----------------------------------------------------------------------------
succ(?a) MOD ?b = rec(succ(?a MOD ?b) |-| ?b,0,%(x,y)succ(?a MOD ?b)) : N [?H]
*)
read_goal TypeThy "succ(a) MOD b = \
\    rec(succ(a MOD b) |-| b, 0, %(x,y)succ(a MOD b)) : N  [H]";
expand (unfold_goal_tac ["MOD"]);
expand (new_simp_tac([absdiff_typing],[],[],5)); 
val MOD_comp_succ = tidyrule(top_rule());





(*typing of quotient: short and long versions*)

read_goal TypeThy "a / b : N [H]";
expand (unfold_goal_tac ["/"]);
expand (new_typechk_tac [absdiff_typing,MOD_typing]);
val quo_typing = tidyrule(top_rule());

read_goal TypeThy "a / b = c / d : N  [H]";
expand (unfold_goal_tac ["/"]);
expand (new_equal_tac [absdiff_typing_long, MOD_typing_long]);
expand typechk_tac;
expand (determ_resolve_tac ([thin_eqelem], 1));
val quo_typing_long = tidyrule(top_rule());


(*computation for quotient: 0 and successor cases*)

(*      ?b : N [ ?H ]     
    ---------------------
    0 / ?b = 0 : N [ ?H ] 	*)
read_goal TypeThy "0 / b = 0 : N  [H]";
expand (unfold_goal_tac ["/"]);
expand (new_simp_tac([MOD_typing, absdiff_typing], [], [], 5)); 
val quo_comp0 = tidyrule(top_rule());


read_goal TypeThy "succ(a) / b = \
\    rec(succ(a) MOD b, succ(a / b), %(x,y) a / b) : N  [H]";
expand (unfold_goal_tac ["/"]);
expand (new_simp_tac([MOD_typing],[],[], 5));  (*4 secs*)
val quo_comp_succ = tidyrule(top_rule());


(*Version of above with same condition as the MOD one*)
read_goal TypeThy "succ(a) / b = \
\    rec(succ(a MOD b) |-| b, succ(a / b), %(x,y) a / b) : N  [H]";
expand (resolve_tac [ trans_elem RES quo_comp_succ ] 1);
expand (new_simp_tac([MOD_typing, quo_typing, absdiff_typing],
		     [MOD_comp_succ],   [], 5));   (*2 secs*)
expand (rN_elim_tac "succ(a MOD b)|-|b" 3);
expand presimp_tac;
expand simp_tac;  (*6 secs*)
expand (new_typechk_tac [MOD_typing, quo_typing, absdiff_typing]); 
val quo_comp_succ2 = tidyrule(top_rule());


(*for case analysis on whether a number is 0 or a successor*)
read_goal TypeThy "?b : Eq(N,a,0) + SUM x:N. Eq(N,a, succ(x)) [H]";
expand (rN_elim_tac "a" 1);
expand (resolve_tac [Plus_intr_inr] 3);
expand (resolve_tac [Plus_intr_inl] 2);
expand eqintr_tac;
expand equal_tac;
expand typechk_tac;
val iszero_decidable = tidyrule(top_rule());



val arith_typing_rls =
    [add_typing, diff_typing, mult_typing,
     MOD_typing, quo_typing, absdiff_typing];


(*should be proved by resolution tactics*)
val iszero_elim = resolvelist(Plus_elim,
	[ (1, [iszero_decidable]), (3, [Sum_elim0]),
	  (3, form_rls), (3, form_rls) ]);


(*For using embedded assumptions in simplification*)
val Eq_elim1 = Eq_elim RES thin_elem RES assume_elem;
val Eq_elim2 = Eq_elim RES thin_elem RES thin_elem RES assume_elem;



(*     ?a : N [ ?H ]              ?b : N [ ?H ]     
    ----------------------------------------------
    ?a MOD ?b  #+  (?a/?b) #* ?b  =  ?a : N [ ?H ]    

  Holds when b is 0 since   a MOD 0 = a     and    a/0 = 0  *)
read_goal TypeThy "a MOD b  #+  (a/b) #* b = a : N  [H]";
expand (rN_elim_tac "a" 1);
expand (new_simp_tac(arith_typing_rls,
		[MOD_comp0,MOD_comp_succ, quo_comp0, quo_comp_succ2],
		[add_typing_long, mult_typing_long], 6)); 
(*5 secs*)
expand (resolve_tac [ Eq_elim ] 4);
expand (rconstrain_res_tac [ ("?a1", "succ(u2 MOD b)|-|b") ] iszero_elim 4);
expand presimp_tac;
expand (new_simp_tac(arith_typing_rls,
		[Eq_elim2,   (*for using an embedded assumption*)
		 add_comp0,add_comp_succ, mult_comp0,mult_comp_succ, 
		 MOD_comp0,MOD_comp_succ, quo_comp0, quo_comp_succ2],
		[add_typing_long, mult_typing_long], 6)); 
(*36 secs*)
expand (resolve_tac [ trans_elem RES add_typing_long ] 3);
expand (resolve_tac [ sym_elem RES absdiff_eq0 ] 3);
expand (resolve_tac [Eq_elim0] 5);
expand (new_simp_tac(arith_typing_rls,  [Eq_elim1, add_comp_succ], [], 5));
(*4 secs*)
val MOD_quo_equality = tidyrule(top_rule());


