(*  Title: 	tactics
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)


(*A proof tree is represented by a rule: premises=subgoals, concl = main goal.
  A tactic maps a proof tree to a sequence of proof trees:
    if length of sequence = 0 then the tactic does not apply;
    if length > 1 then backtracking on the alternatives can occur.*)

type tactic = rule -> rule sequence;


fun PRIMITIVE (rulefun: rule->rule) : tactic =
  fn prf => sequence_of_list [ rulefun prf ];

fun SIGNAT_TAC (signtac: signat->tactic) : tactic = 
  fn prf => 
    let val {sign,...} = rep_rule prf 
    in  signtac sign prf  end;
  
fun unfold_goal_tac ids = PRIMITIVE (unfold_def_in_prems ids);

fun fold_goal_tac ids = PRIMITIVE (fold_def_in_prems ids);

fun fold_tac ids = PRIMITIVE (fold_def ids);

val unify_constraints_tac : tactic = unify_constraints;

val merge_premises_tac = PRIMITIVE merge_premises;

(*Tactics to put constraints on the current proof.
  The constraints become visible as equations,
  and affect later unifications.*)
fun constrain_tac tpairs = PRIMITIVE (constrain tpairs);

fun rconstrain_tac stpairs = PRIMITIVE (read_constrain stpairs);


(*Constrains goal n to be the theorem sthm*)
fun rconstrain_goal_tac sthm gno prf =
    let val prem = nth_elem (gno-1, prems_of_rule prf);
   	val {sign,...} = rep_rule prf
    in  sequence_of_list [ constrain [ (prem, read_theorem sign sthm) ] prf ]
    end
    handle list => null_sequence;



(*Constrain the goal so that the goals gno1 and gno2 are equal*)
fun identify_goals_tac (gno1,gno2) prf =
    let val prems = prems_of_rule prf;
	fun goal n = nth_elem (n-1, prems)
    in  constrain_tac [ (goal gno1, goal gno2) ] prf  end
    handle list => null_sequence;
  

(*Remove exception "rule" from proof sequence
  Could have "resolves" do this, but would hide these exceptions at too
  low a level, would complicate debugging.*)
fun strip_exceptions str = sequenceof (fn() =>
   (case spull str of
	None => None
      | Some(y,rest) => Some(y, strip_exceptions rest))
    handle rule => None);

(*Tactic returns null sequence instead of raising exception "rule"*)
fun NOEXCEPT tac x = strip_exceptions (tac x); 


(*No standardization:  used?*)
fun compose_tac (rl: rule) (gno: int) prf = 
  compose (prf,gno,rl);


(*Resolution*)
fun resolve_tac (rules: rule list) (gno: int) prf = 
  strip_exceptions (resolves (prf,gno,rules));


(*Resolves proof tree with a rule constrained by
	tpairs = [ (t1,u1), ..., (tn,un) ]
  The ti match subterms of the rule, the ui subterms of the goal.
  The ti are standardized but the ui are not.*)
fun constrain_res_tac tpairs rule gno prf =
    let val {maxidx,...} = rep_rule prf;
        fun incpair (t,u) = (incr_indexes (maxidx+1) t,  u);
	val csrule = constrain (map incpair tpairs) 
			       (standardize (maxidx+1) rule)
    in  compose  (prf,gno,csrule)  end;


fun rconstrain_res_tac (stpairs: (string*string)list) rule gno =
    SIGNAT_TAC (fn sign =>
	constrain_res_tac (map (read_termpair sign) stpairs) rule gno);
	

fun SELECT_GOAL (tac: tactic) gno prf =
  flat_sequence (maps (fn rl => compose (prf,gno,rl))
		   (tac (triv_of_prem(prf,gno))));


(*string of symbols following first argument of combinations
  symbols are constants, parameters
  example:  rec(succ(x),...,...)  has   [ "rec", "succ", "*param*" ]  *)
fun head_string t : string list =
  let val (head,args) = strip_comb t;
      fun tail_string [] = []
	| tail_string(arg::_) = head_string arg
  in  case head of
      Const(name,_) =>      name :: tail_string args
    | Param _       => "*param*" :: tail_string args  (*Use Param name??*)
    | Var (xname,ary) => []
    | _ => []
  end;


(*head string of conclusion of rule*)
fun head_string_concl rl : string list =
  let val {concl,...} = rep_rule rl 
  in  head_string concl  end;



(*Trees indexed by string lists: each arc is labelled by a string
  Each node contains a list of items, and arcs to children
  Empty string addresses entire tree 
  Vital to return items the proper order:
    Items stored deeper are more specific and must preceed items stored
    above them;  order must be preserved in items stored at same level.
*)
abstype 'a stringtree = stree of 'a list * (string * 'a stringtree) list
  with

  val null_stree = stree([],[]);

  (*add an item to the list at the node addressed by the keys
    create node if not already present*)
  fun insert_stree ((  []  ,    x), stree(xs,alist)) = stree(x::xs, alist)
    | insert_stree ((key::keys, x), stree(xs,alist)) =
	let fun newpair tr = (key, insert_stree((keys,x), tr)) 
	    fun inslist [] = [ newpair null_stree ]
	      | inslist((keyi,tri)::alist) =
		  if key=keyi then newpair tri :: alist
		  else if key<keyi then (*absent, insert in alist*)
		      newpair null_stree :: (keyi,tri) :: alist
		  else (keyi,tri) :: inslist alist
	in  stree(xs, inslist alist)  end;

  (*Return the list of items at the given node, [] if no such node*)
  fun tree_lookup (stree(xs,alist), []) = xs
    | tree_lookup (stree(xs,alist), key::keys) =
	 (case sassoc(alist,key) of 
             None =>  []
           | Some tr' => tree_lookup(tr',keys));

  (*Return the list of all items in the tree*)
  fun tree_below (stree(xs,alist)) =
	let fun bel [] = []
	      | bel ((_,tr)::alist) = tree_below tr  @  bel alist
	in  bel alist @ xs  end;

  (*Return all items with compatible addresses:
    those where one address is a prefix of the other*)
  fun tree_compat (tr  ,  []) = tree_below tr
    | tree_compat (stree(xs,alist), key::keys) = 
	 (case sassoc(alist,key) of 
             None =>  xs
           | Some tr' => tree_compat(tr',keys) @ xs)
  end;


(*insert a rule in a rule tree using its head_string*)
fun insert_rule (rl,rtr) = insert_stree((head_string_concl rl, rl), rtr);


(*Find next goal for deterministic resolution:
    number of possible resolvents must not exceed wlimit
  Returns number and resolvent of goal, if found.*)
fun determ_find_goal (rtr,wlimit,goalno) prf : (int * rule) option = 
  let val {prems, ...} = rep_rule prf;
      fun dfind gno = 
          case nth_tail (gno-1, prems) of
              [] => None
            | prem::_ => 
		let val rules = tree_compat(rtr, head_string prem)
		in  if length rules > wlimit  then  dfind (gno+1)
                    else case spull (resolve_tac rules gno prf) of
                      None          => dfind(gno+1)
                    | Some (prf',_) => Some(gno,prf')
		end
   in  dfind goalno  end;



(*deterministic depth_first search: ignores all but first choice
    if no choices, tries next goal*)
fun determ_resolve_tac (rules,wlimit) prf =
  let val rtr = itlist_right insert_rule (rules, null_stree);
      fun determ (goalno,prf) =
	  case determ_find_goal (rtr,wlimit,goalno) prf of
	      None => if goalno=1 then prf else determ(1,prf)
	    | Some (gno,prf') => determ(gno,prf')
  in sequence_of_list [ determ(1,prf) ] end;



(*LCF tacticals*)

(*the tactical THEN performs one tactic followed by another*)
infix THEN;

fun (tac1 THEN tac2) : tactic = 
  fn prf => flat_sequence (maps tac2 (tac1 prf));


(*The tactical ORELSE uses the first tactic that returns a nonempty sequence.
  Like in LCF, ORELSE commits to either tac1 or tac2 immediately.
  Does not backtrack to tac2 if tac1 was initially chosen. *)
infix ORELSE;

fun (tac1 ORELSE tac2) : tactic = 
  fn prf =>  
    case spull(tac1 prf) of
	None       => tac2 prf
      | sequencecell => sequenceof(fn()=> sequencecell);


(*The tactical APPEND combines the results of two tactics.
  Like ORELSE, but allows backtracking on both tac1 and tac2*)
infix APPEND;

fun (tac1 APPEND tac2) : tactic = 
  fn prf =>  tac1 prf  @@  tac2 prf;


(*passes all proofs through unchanged;  identity of THEN*)
val all_tac : tactic = fn prf => sequence_of_list [ prf ];


(*passes no proofs through;  identity of ORELSE and APPEND*)
val no_tac  : tactic = fn prf => null_sequence;


fun DETERM tac x =
  let val (prfs,_) = chop_sequence(1, tac x)
  in  sequence_of_list prfs  end
  handle rule => null_sequence;


(*Conditional tactical: testfun controls which tactic to use next.
  Beware: due to eager evaluation, both thentac and elsetac are evaluated.*)
fun COND testfun (thentac,elsetac) prf = 
  if testfun prf then  thentac prf
  else  elsetac prf;


(*Performs no backtracking: quits when it gets stuck*)
fun REPEAT tac x = ((tac THEN REPEAT tac) ORELSE all_tac) x;

(*Repeat 1 or more times*)
fun REPEAT1 tac x = (tac THEN (REPEAT1 tac ORELSE all_tac)) x;


(*Repeats again and again until "satfun" reports proof tree as satisfied*)
fun DEPTH_FIRST satfun tac x = 
  COND satfun  (all_tac,  tac  THEN  DEPTH_FIRST satfun tac) x;


(*Repeats until rule has been reduced to a theorem: no subgoals*)
val THM_DEPTH_FIRST = DEPTH_FIRST (is_theorem);


(*Breadth-first search for a proof tree satisfying the "satfun"*)
fun BREADTH_FIRST satfun tac prf =
  let fun tacf p = list_of_sequence (tac p);
      fun bfs [] = null_sequence
     	| bfs prfs = 
	    let val prfs' = flat (map tacf prfs);
		fun sfilt [] = bfs (filter_out satfun prfs')
		  | sfilt(x::xs) =
		      if satfun x then sequenceof(fn()=>Some(x, sfilt xs))
		      else sfilt xs
	    in  (*DEBUG*) prs"breadth= ";  pri(length prfs');  prs"\n";
		sfilt prfs'  end
  in  bfs [prf]  end;
 

(*Breadth-first search with depth bound: for debugging*)
fun BREADTH_FIRST_TO satfun depth tac prf =
  let fun tacf p = list_of_sequence (tac p);
      fun bfs ([],_) = null_sequence
     	| bfs (prfs,0) = sequence_of_list prfs
     	| bfs (prfs,n) = 
	      let val prfs' = flat (map tacf prfs);
		  fun sfilt [] = bfs (filter_out satfun prfs', n-1)
		    | sfilt(x::xs) =
			if satfun x then sequenceof(fn()=>Some(x, sfilt xs))
			else sfilt xs
	      in  sfilt prfs'  end
  in  bfs ([prf], depth)  end;
 
 

(*True if the two terms could possibly be unified (derived from SIMPL)
  Often incorrect: treats all variables as distinct,
        all flex-rigid pairs as unifiable:  %(x)x  and  %(x)?a'(succ(x))
  More efficient than trying resolution, more accurate than string_trees*)
fun could_resolve (tm1,tm2) =
  let fun matchrands (rator1$rand1, rator2$rand2) =
	      could_resolve (rand1,rand2)  andalso   matchrands(rator1,rator2)
	| matchrands _ = true;
      val head1 = head_of tm1  and  head2 = head_of tm2
  in case (head1,head2) of
	(_, Var _) => true
      | (Var _, _) => true
      | (Const(id1,_), Const(id2,_)) =>
	    id1=id2 andalso matchrands(tm1,tm2)
      | (Param(id1,_), Param(id2,_)) =>
	    id1=id2 andalso matchrands(tm1,tm2)
      | (Bound bno1, Bound bno2) =>
	    bno1=bno2 andalso matchrands(tm1,tm2)
      | (Abs(_,_,body1), Abs(_,_,body2)) =>
            could_resolve (body1, body2)
      | _ => false
  end;

fun filter_rules (limit, tm, rls) =
  let fun filtr (limit, []) = []
	| filtr (limit, rl::rls) =
	    if limit=0 then []    else
	    let val {concl, ...} = rep_rule rl
	    in  if could_resolve (tm, concl)  then rl :: filtr(limit-1, rls)
	        else filtr(limit,rls)
	    end
  in  filtr(limit,rls)  end;


(*Like resolve_tac;  filters rules first, strips exceptions after*)
fun filt_resolve_tac (rules: rule list) (gno: int) prf =
  case nth_tail(gno-1, prems_of_rule prf) of
	  [] => null_sequence
	| prem::_ =>
	    strip_exceptions
	      (resolves (prf,gno, filter_rules (~1,prem,rules)));


(*Starting from goalno, searches for a goal that is ready to resolve.
  "choose" suggests rules for goal, if None then goal is ignored.*)
fun find_goal (choose,goalno) prf : (int * rule sequence) option = 
  let val prems = prems_of_rule prf;
      fun find gno = case nth_tail (gno-1, prems) of
	  [] => None
	| prem::_ => 
	    (case  choose(prem)  of
	       None =>  find (gno+1)
	     | Some rules =>  Some (gno, resolve_tac rules gno prf))
   in  find goalno  end;


(*Depth-first search by resolution, where "choose" selects rules to try.*)
fun depth_resolve_tac choose prf =
  let fun depthf goalno (prf, backsequence) =
	  case find_goal (choose,goalno) prf of
	      None => if goalno=1 then scons(prf, backsequence)
		      else depthf 1 (prf,backsequence)
	    | Some (gno,sg_str) =>  (*start from new gno*)
	        itsequence_right (depthf gno) (sg_str,backsequence)
  in depthf 1 (prf, null_sequence) end;
