(*  Title: 	sequence
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)

(*sequence.ml  Unbounded sequences implemented by closures
  Does not "memoize" sequence elements:  
    recomputation occurs each time sequence is inspected
    Memo sequences require extended reference typechecking
*)


(*operations:
    spull: try to get next sequence element, returns None or Some(x,str)
    sequenceof: the abstraction for making a sequence
    scons: prefix an element to the sequence
    null_sequence: the empty sequence
    
    scons(x,s) should only be used if s can be evaluated without divergence,
      otherwise use sequenceof(fn()=> Some(x,s)), which delays evaluation *)
abstype 'a sequence = Sequence of unit -> ('a * 'a sequence)option
  with fun spull(Sequence f) = f();
       val sequenceof = Sequence;
       fun scons all = Sequence(fn()=>Some all);
       val null_sequence = sequenceof(fn()=>None)
  end;


(*The list of the first n elements, paired with rest of sequence.
  If length of list is less than n, then sequence had less than n elements.*)
fun chop_sequence (n:int, s: 'a sequence): 'a list * 'a sequence =
  if n<=0 then ([],s)
  else case spull(s) of
           None => ([],s)
         | Some(x,s') => let val (xs,s'') = chop_sequence (n-1,s')
		         in  (x::xs, s'')  end;

(*conversion from sequence to list*)
fun list_of_sequence (s: 'a sequence) : 'a list = case spull(s) of
        None => []
      | Some(x,s') => x :: list_of_sequence s';


(*conversion from list to sequence*)
fun sequence_of_list []     = null_sequence
  | sequence_of_list (x::l) = scons (x, sequence_of_list l);


(*functional to print a sequence, up to "count" elements
  the function prelem should print the element number and also the element*)
fun print_sequence (prelem: int -> 'a -> unit) count (s: 'a sequence) : unit =
  let fun pr (k,s) = 
	   if k>count then ()
	   else case spull(s) of
	       None => ()
	     | Some(x,s') => (prelem k x;  prs"\n";  pr (k+1, s'))
  in  pr(1,s)  end;

(*Map the function f over the sequence, making a new sequence*)
fun maps (f : 'a->'b) (s: 'a sequence) = sequenceof (fn()=> case spull(s) of
        None       => None
      | Some(x,s') => Some(f x,  maps f s'));

(*Sequence append:  put the elements of s1 in front of the closure sc2*)
infix @@;
fun (s1: 'a sequence) @@ (s2: 'a sequence) : 'a sequence =
  let fun copy s = sequenceof (fn()=>
	    case spull(s) of  
		 None       => spull s2
	       | Some(x,s') => Some (x, copy s'))
  in  copy s1 end;

(*Map a function over a list, prefixing results to sequence;
  use with I to concatenate a list and a sequence.*)
fun map_to_sequence (f: 'a->'b) (l: 'a list, str: 'b sequence) : 'b sequence =
  let fun mapp []     = str
        | mapp (x::l) = scons (f x, mapp l)
  in  mapp l  end;

(*map over a sequence s1, append the sequence s2*)
fun mapp_sequence (f : 'a->'b) (s1: 'a sequence) (s2: 'b sequence) : 'b sequence =
  let fun copy s = sequenceof (fn()=> 
            case spull(s) of
	        None       => spull(s2)
              | Some(x,s') => Some(f x, copy s'))
  in  copy s1 end;

(*flattening a sequence of sequences to a single sequence*)
fun flat_sequence ss = sequenceof (fn()=> case spull(ss) of
        None => None
      | Some(s,ss') =>  spull(s @@ flat_sequence ss'));



(*accumulating a function over a sequence;  this is lazy*)
fun itsequence_right (f: 'a * 'b sequence -> 'b sequence) 
 		   (s: 'a sequence, bstr: 'b sequence) : 'b sequence =
  let fun its s = sequenceof (fn()=>
            case spull(s) of
	        None       => spull bstr
	      | Some(a,s') => spull(f(a, its s')))
  in  its s end;
