(*  Title: 	library
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)


(*library of basic functions
  intended for general use, not just theorem proving*)


exception list : string;

(*discriminator and selectors.  
 Rarely used but the names should be made standard for compatibility (?)*)
fun null   []   = true
  | null (_::_) = false;

fun hd   []   = raise list with "hd"
  | hd (a::_) = a;

fun tl   []   = raise list with "tl"
  | tl (_::l) = l;

fun fst(x,y) = x and snd(x,y) = y;

(*Handy combinators*)
fun curry f x y = f(x,y);
fun uncurry f(x,y) = f x y;
fun I x = x  and  K x y = x;

(*preconcat"A"  puts A on the front;  postconcat"A" puts it at the end!*)
fun preconcat x y = x^y;
fun postconcat y x = x^y;

fun plus x y : int = x+y;
fun square (n: int) = n*n;

fun fact 0 = 1
  | fact n = n * fact(n-1);


(*The following versions of itlist are designed to fit nicely with infixes.
 itlist_left is related to the old revitlist.*)

(*  (op @) (e, [x1,...,xn])  ======>   ((e @ x1) @ x2) ... @ xn
    for operators that associate to the left.  TAIL RECURSIVE*)
fun itlist_left (f: 'a * 'b -> 'a) : 'a * 'b list -> 'a =
  let fun itl (e, [ ])  = e
        | itl (e, a::l) = itl (f(e,a), l)
  in  itl end;

(*  (op @) ([x1,...,xn], e)  ======>   x1 @ (x2 ... @ (xn @ e))
    for operators that associate to the right.  TAIL RECURSIVE*)
fun itlist_right (f: 'a * 'b -> 'b) : 'a list * 'b -> 'b =
  let fun itl ([ ],  e) = e
        | itl (a::l, e) = itl (l, f(a,e))
  in  fn (l,e) => itl (rev l, e) end;


(*The infixes !* *! and ** handle combinations of function applications
 and lists.  The ! means singleton, the * means list.  Example:
     > (concat !* ["a","b","c"]) *! "9";
     ["a9", "b9", "c9"] : string list   
     > (concat !* ["a","b","c"]) ** ["x", "y", "z"];
     ["ax", "by", "cz"] : string list   
*)

(*f !* [x1,...,xn]   ======>   [f x1, ..., f xn] *)
infix !*;  fun (f !* xs) = map f xs;

(*[f1,...,fn] *! x  ======>   [f1 x, ..., fn x] *)
infix *!;
fun ( []   *! x)    =  []
  |   (f::fs *! x) =  f x :: (fs *! x);

(*[f1,...,fn] ** [x1,...,xn]  ======>   [f1 x1, ..., fn xn] *)
infix **;
fun ( []   ** [] )   = []
  | (f::fs ** x::xs) = f x :: (fs ** xs)
  | ( _    ** _  )   = raise list with "**";


(*Length of a list.  Should unquestionably be a standard function*)
local fun length1 (n, [ ])  = n   (*TAIL RECURSIVE*)
        | length1 (n, x::l) = length1 (n+1, l)   
in  fun length l = length1 (0,l) end;


(*front (n, [e1, ..., en, e[n+1], ... , e[n+m]])   ======>   [e1, ..., en] 
 If n is negative then runs off end, raising exception.
 Raises EXCEPTION if list too short, perhaps should not.*)
fun front (0, _ ) = []
  | front (n, x::l) = x :: front(n-1, l)
  | front _  = raise list with "front";

(*Chop n elements from the front of l.
 If n is negative then runs off end, returning [].
 Does NOT raise exception;  for style of use see nth_elem, next.*)
fun nth_tail (_,[]) = []
  | nth_tail (0, l) = l
  | nth_tail (n, _::l) = nth_tail (n-1, l);

(*Return nth element of l, where 0 designates the first element;
  raise EXCEPTION if list too short.*)
fun nth_elem NL = case (nth_tail NL) of
    [] => raise list with "nth_elem" 
  | x::l =>  x;

(*I/O.  Defines separate identifier for terminal output channel.*)
(***********val std_out = open_out "";*)
fun print_string (s:string) = output(std_out,s);
fun writeln (s:string) = print_string (s ^ "\n");

(*Typed printing functions.  More convenient than print because they yield
 unit type, preventing type errors in imperative code.*)
fun print_int (x:int) = (print x; ());
fun print_bool (x:bool) = (print x; ());
fun print_stringq (x:string) = (print x; ());  (*prints quotes around string*)

(*Abbreviations*)
val pri = print_int  and  prs = print_string;

(*write a string repeatedly, 'count' times*)
fun writerep(count,str) =
    if count>0 then (print_string str;  writerep(count-1, str))   else ();

(*Function exponentiation: f(...(f x)...) with n applications of f *)
fun funpow n f x =
  let fun rep (0,x) = x
        | rep (n,x) = rep (n-1, f x)
  in  rep (n,x)  end;

(*The old 'combine'.
  [x1,...,xn] ~~ [y1,...,yn]  ======>   [(x1,y1), ..., (xn,yn)] *)
infix ~~;
fun ( []   ~~  [])   = []
  | (x::xs ~~ y::ys) = (x,y) :: (xs ~~ ys)
  | (  _   ~~   _  ) = raise list with "~~";

(*Inverse of ~~;  the old 'split'.
   [(x1,y1), ..., (xn,yn)]  ======>  ( [x1,...,xn] , [y1,...,yn] ) *)
fun split_list l = (map fst l, map snd l);


(*make the list [x; x; ...; x] of length n*)
fun replicate n (x: 'a) : 'a list =
  let fun rep (0,xs) = xs
        | rep (n,xs) = rep(n-1, x::xs) 
  in   if n<0 then raise list with "replicate"
       else rep (n,[])
  end;

(*Flatten a list of lists to a list.*)
fun flat (ls: 'a list list) : 'a list = itlist_right (op @) (ls,[]);

(*The membership (mem) and assoc functions cannot be polymorphic
 because they involve equality.  Integer and string versions are provided.
 Modules should correct this problem.*)
infix smem;
fun (x:string) smem []  =  false  
  | (x:string) smem (y::l)  =  (x=y) orelse (x smem l);

infix imem;
fun (x:int) imem []  =  false
  | (x:int) imem (y::l)  =  (x=y) orelse (x imem l);


(*Tests if the list contains any repeated elements.
 If so returns the tail beginning with the first one.*)
fun is_sdistinct (ss : string list) : string list = case ss of
    [] => []
  | s::ss' => if  s smem ss'  then  ss   else   is_sdistinct ss';

fun sdistinct1 (seen, []) = rev seen
  | sdistinct1 (seen, s::ss) =
      if s smem seen then sdistinct1 (seen, ss)
    		     else sdistinct1 (s::seen, ss);

fun sdistinct ss = sdistinct1([],ss);


datatype 'a option = None  |  Some of 'a;


fun sassoc ([], key:string) = None
  | sassoc ((keyi,xi)::pairs, key) =
      if key=keyi then Some xi  else sassoc (pairs,key);

fun iassoc ([], key:int) = None
  | iassoc ((keyi,xi)::pairs, key) =
      if key=keyi then Some xi  else iassoc (pairs,key);


fun mapfilter (f: 'a -> 'b option) ([]: 'a list) = [] : 'b list
  | mapfilter f (x::xs) =
      case (f x) of
	  None => mapfilter f xs
	| Some y => y :: mapfilter f xs;


(*Copy the list preserving elements that satisfy the predicate*)
fun filter (pred: 'a -> bool) : 'a list -> 'a list = 
  let fun filt [] = []
        | filt (x::xs) =  if pred(x) then x :: filt xs  else  filt xs
  in  filt   end;

(*Copy the list removing elements that satisfy the predicate*)
fun filter_out (pred: 'a->bool) : 'a list -> 'a list = filter (not o pred);


(*A crude quicksort;  stable;   partitions using first element.
  Slow if list is nearly sorted (or reverse sorted) already.
  'lesq' is a curried predicate for <= on type 'a. 
  See O'Keefe, A Smooth Applicative Merge Sort, Dept of AI, Edinburgh*)
fun quicksort (lesq: 'a -> 'a -> bool) (xs: 'a list) : 'a list =
  let fun sort1 (xs,cont) = case xs of
        [] => cont
      | x::xs' =>  
          sort1(filter_out (lesq x) xs',
	        x :: sort1(filter (lesq x) xs', cont))
  in  sort1 (xs,[])  end;


(*make the list [from, from+1, ..., to]*)
infix upto;
fun (from upto to) =
    if from>to then []  else  from :: ((from+1) upto to);

(*make the list [from, from-1, ..., to]*)
infix downto;
fun (from downto to) =
    if from<to then []  else  from :: ((from-1) downto to);

(*Like Lisp's MAPC -- seq proc [x1,...,xn] evaluates 
  proc(x1); ... ; proc(xn) for side effects.*)
fun seq (proc: 'a -> unit) : 'a list -> unit = 
  let fun seqf []     = ()
        | seqf (x::l) = (proc x;  seqf l)
  in  seqf end;


(* rotate_left [1,2,3,4]   ======>  [2, 3, 4, 1] *)
fun rotate_left (a::l) = l @ [a]
  | rotate_left []     = raise list with "rotate_left";

(* rotate_right [1,2,3,4]   ======>  [4, 1, 2, 3] *)
fun rotate_right l = case (rev l) of
    (ra::rl) => ra :: (rev rl)  
  |  []   => raise list with "rotate_right";



(*operators for combining predicates*)
infix orf;  fun (p orf q)x = p x orelse q x ;
infix andf;  fun (p andf q)x = p x andalso q x ;
fun notf p x = not (p x) ;

fun swap(x,y)=(y,x);


fun orl [] = false
  | orl (x::l) =  x  orelse  orl l;

fun andl [] = true
  | andl (x::l) =  x  andalso  andl l;


(*exists pred [x1,...,xn] ======>  pred(x1)  orelse  ...  orelse  pred(xn)*)
fun exists (pred: 'a -> bool) : 'a list -> bool = 
  let fun boolf [] = false
        | boolf (x::l) = (pred x) orelse boolf l
  in boolf end;


(*forall pred [x1,...,xn] ======>  pred(x1)  andalso  ...  andalso  pred(xn)*)
fun forall (pred: 'a -> bool) : 'a list -> bool = 
  let fun boolf [] = true
        | boolf (x::l) = (pred x) andalso (boolf l)
  in boolf end;

(*Expand the number in the given base 
 example: radixpand(2, 8)  gives   [1, 0, 0, 0] *)
fun radixpand (base,num) : int list =
  let fun radix (n,tail) =
  	if n<base then n :: tail
	else radix (n div base, (n mod base) :: tail)
  in  radix (num,[])  end;

(*Expands a number into a string of characters starting from "zerochar"
 example: radixstring(2,"0", 8)  gives  "1000" *)
fun radixstring (base,zerochar,num) =
  let val offset = ord(zerochar); 
      fun chrof n = chr(offset+n)
  in  implode (map chrof (radixpand (base,num)))  end;

fun string_of_int n = 
  if n < 0 then "~" ^ radixstring(10,"0",~n)  else radixstring(10,"0",n);


fun max[m : int]  = m
  | max(m::n::ns) = if m>n  then  max(m::ns)  else  max(n::ns)
  | max []        = raise list with "max";

fun min[m : int]  = m
  | min(m::n::ns) = if m<n  then  min(m::ns)  else  min(n::ns)
  | min []        = raise list with "min";


(* [x1,...,xi,...,xn]  --->  ([x1,...,x(i-1)], [xi,..., xn])
   where xi is the first element that does not satisfy the predicate*)
fun take_prefix (pred : 'a -> bool)  (xs: 'a list) : 'a list * 'a list =
  let fun take (rxs, []) = (rev rxs, [])
	| take (rxs, x::xs) =
	    if  pred x  then  take(x::rxs, xs)  else  (rev rxs, x::xs)
  in  take([],xs)  end;


(* [x1,...,xi,...,xn]  --->  [xi,..., xn]
   where xi is the first element that does not satisfy the predicate*)
fun strip_prefix (pred : 'a -> bool) : 'a list -> 'a list  =
  let fun strip [] = []
	| strip (x::xs) = if  pred x  then  strip xs  else  x::xs
  in  strip  end;


(*make a memo function that holds the values of f(start),...,f(stop)
  if its argument is outside that range, calls f *)
fun imemoize (start,stop,f) =
  let fun make_ilist k =
            if k<start then []
	    else (k, f(k)) :: make_ilist(k-1);
      fun fastf pairs k =
        let fun ilook ((i,x)::pairs) =
		  if i>k then  ilook pairs  else  x
	      | ilook [] = raise list with "imemoize"
        in  if k<start orelse k>stop then  f(k)  else  ilook pairs  end
  in  fastf (make_ilist stop)  end;

