(*  Title: 	int-resolve
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)

(*Predicate Calculus resolution tactics -- constructive
*)


(*Detection of terms like %(hobj)...
  This is meaningless since it cannot represent any list *)
fun is_hobj_abs (Abs(_, Ground "hobj", _)) = true
  | is_hobj_abs _ = false;


(*hypotheses diagreement pair*)
fun is_hobj_abs_pair (t,u) = is_hobj_abs t orelse is_hobj_abs u;


(*Smash disagreement pairs if all refer to hypotheses.
  May smash meaningful pairs, we may need to improve this tactic.*)
fun smash_hobj_tac prf =
  let val {tpairs,...} = rep_rule prf
  in (case tpairs of
	  [] => all_tac
        | _  => if (forall is_hobj_abs_pair tpairs) then unify_constraints_tac
	        else all_tac)
     prf
  end;


(*Returns the list of all formulas in the hypothesis*)
fun form_hyps_of (Const("Hypof",_) $ P $ rand) = P :: form_hyps_of rand
  | form_hyps_of (H $ rand) = form_hyps_of rand
  | form_hyps_of _ = [];


(*Handles associative unification: if form2 has hyps = [?H, P, ...]
  then looks for a hyp P in form1.   *)
fun could_resolve_hyp (P1 $ (Abs(_,_,body)),
	    	  P2 $ Abs(_,_, Var _ $ (Const("Hypof",_) $ Q2 $ _))) =
    	could_resolve (P1,P2) andalso
    	exists (fn Q1 => could_resolve (Q1,Q2))  (form_hyps_of body)
  | could_resolve_hyp (P1, P2) = could_resolve (P1,P2);


(*Like filter_rules, using could_resolve_hyp*)
fun filter_hyp_rules (limit, tm, rls) =
  let fun filtr (limit, []) = []
	| filtr (limit, rl::rls) =
	    if limit=0 then []    else
	    let val {concl, ...} = rep_rule rl
	    in  if could_resolve_hyp (tm, concl) 
	        then rl :: filtr(limit-1, rls)
	        else filtr(limit,rls)
	    end
  in  filtr(limit,rls)  end;


(*Like filt_resolve_tac, using could_resolve_hyp*)
fun filhyp_resolve_tac (rules: rule list) (gno: int) prf =
  case nth_tail(gno-1, prems_of_rule prf) of
	  [] => null_sequence
	| prem::_ =>
	    strip_exceptions
	      (resolves (prf,gno, filter_hyp_rules (~1,prem,rules)));


(*For "cutting in" an arbitrary formula*)
fun rcut_tac stm pno : tactic =
    rconstrain_res_tac [ ("?P", stm) ] cut pno;


(*For "thinning out" an arbitrary formula*)
fun rthin_tac stm pno : tactic =
    rconstrain_res_tac [ ("?Q", stm) ] thin pno;


(*iff introduction
    ?Q [ ?H, ?P ]     ?P [ ?H, ?Q ]     
    -------------------------------
	    ?P <=> ?Q [ ?H ]	*)
read_goal SetThy "P<=>Q [ H ]";
expand (unfold_goal_tac ["<=>"]);
expand (determ_resolve_tac ([conj_intr,imp_intr],1));
val iff_intr = ttop_rule();


(*tactic for proving both forms of iff elimination*)
val iff_elim_tac =
    (rcut_tac "P<=>Q" 1)  THEN
    (unfold_goal_tac ["<=>"])  THEN
    (resolve_tac [ conj_elim RES assume ] 2)  THEN
    (resolve_tac [ imp_elim RES assume ] 2)  THEN
    (REPEAT (resolve_tac [thin] 2))  THEN
    (fold_tac ["<=>"]);


(*  ?P <=> ?Q [ ?H ]     ?P [ ?H ]     
    ------------------------------
           ?Q [ ?H ]     *)
read_goal SetThy "Q [ H ]";
expand iff_elim_tac;
val iff_elim1 = ttop_rule();


(*  ?P <=> ?Q [ ?H ]     ?Q [ ?H ]     
    ------------------------------
           ?P [ ?H ]     *)
read_goal SetThy "P [ H ]";
expand iff_elim_tac;
val iff_elim2 = ttop_rule();



val intr_rls = 
    [all_intr, exists_intr, conj_intr, disj_intr1, disj_intr2,
     imp_intr, iff_intr];

val elim_rls = [all_elim, exists_elim, conj_elim, disj_elim,
                imp_elim, F_elim];


(*deterministic tactics*)
val intr_tac = determ_resolve_tac (assume::intr_rls, 2);



(*elimination rules in sequent style: adding new assumptions*)

(*?P1 ==> ?P [ ?H ]     ?P1 [ ?H ]     ?Q [ ?H, ?P ]     
  --------------------------------------------------
		?Q [ ?H ]  	*)
val cut_imp_elim = cut RES imp_elim;


(*All(?P'1) [ ?H ]     ?Q [ ?H, ?P'1(?a1) ]     
  -----------------------------------------
		?Q [ ?H ] 		*)
val cut_all_elim = cut RES all_elim;



(*  ?R [ ?H, ?G, ?P, ?Q ]     
    ----------------------
    ?R [ ?H, ?P & ?Q, ?G ]    *)
read_goal SetThy "R [ H, P&Q, G ]";
expand (resolve_tac [ conj_elim RES assume ] 1);
expand (resolve_tac [thin] 1);
val conj_elimh = ttop_rule();


(*  ?R [ ?H, ?G, ?P ]     ?R [ ?H, ?G, ?Q ]     
    ---------------------------------------
 	   ?R [ ?H, ?P | ?Q, ?G ]     *)
read_goal SetThy "R [ H, P|Q, G ]";
expand (resolve_tac [ disj_elim RES assume ] 1);
expand (resolve_tac [thin] 1);
expand (resolve_tac [thin] 2);
val disj_elimh = ttop_rule();


(*imp_elim: NO THINNING, may need implication again!
    ?P [ ?H, ?P ==> ?Q, ?G ]     
   --------------------------
    ?Q [ ?H, ?P ==> ?Q, ?G ]            *)
read_goal SetThy "Q [ H, P==>Q, G ]";
expand (resolve_tac [ imp_elim RES assume ] 1);
val imp_elimh = ttop_rule();


(*  ?P [ ?H, ?P ==> ?Q, ?G ]     ?R [ ?H, ?G, ?Q ]     
    ----------------------------------------------
	     ?R [ ?H, ?P ==> ?Q, ?G ]        *)
read_goal SetThy "R [ H, P==>Q, G ]";
expand (resolve_tac [ cut_imp_elim RES assume ] 1);
expand (resolve_tac [thin] 2);
val cut_imp_elimh = ttop_rule();


(*  ?R [ ?H, F, ?G ]  
  Constructively valid though derived from the classical contradiction rule.*)
read_goal SetThy "R [ H, F, G ]";
expand (resolve_tac [ F_elim RES assume ] 1);
val F_elimh = ttop_rule();



(*  ?P'(?a) [ ?H, All(?P'), ?G ]  *)
read_goal SetThy "P'(a) [ H, All(P'), G ]";
expand (resolve_tac [ all_elim RES assume ] 1);
val all_elimh = ttop_rule();



(*All-elimination WITHOUT thinning: complete but may run forever
    ?R [ ?H, All(?P'), ?G, ?P'(?a) ]     
    --------------------------------
       ?R [ ?H, All(?P'), ?G ]   

  Version WITH thinning:
    ?R [ ?H, ?G, ?P'(?a) ]     
    -----------------------
    ?R [ ?H, All(?P'), ?G ]    *)
read_goal SetThy "R [ H, All(P'), G ]";
expand (rcut_tac "P'(a)" 1);
expand (resolve_tac [all_elimh] 1);
val cut_all_elimh_nothin = ttop_rule();
expand (resolve_tac [thin] 1);
val cut_all_elimh = ttop_rule();



(*  ?R [ ?H, ?G, ?P'(y1) ]     
  ---------------------------
  ?R [ ?H, Exists(?P'), ?G ] 
  Param restrictions:            y1> ?G ?H ?P' ?R	   *)
read_goal SetThy "R [ H, Exists(P'), G ]";
expand (resolve_tac [ exists_elim RES assume ] 1);
expand (resolve_tac [thin] 1);
val exists_elimh = ttop_rule();


(*Tactic to turn the implication P==>Q into the rule  R [Q] / R [P]  ,
  with R [Q] further simplified using elimh rules*)
fun use_imp_tac impth =
    resolve_tac [cut RES imp_elim RES impth RES assume] 1  THEN
    resolve_tac [thin] 1  THEN
    determ_resolve_tac ([conj_elimh,disj_elimh,exists_elimh], 3);





(*To use implication efficiently requires considering the form of the 
    antecedent;  the rule cut_imp_elimh allows too much branching.
  Such rules are central to the classical logic prover;
    some of the following have stronger versions in classical logic.*)


(*  ?Q [ ?H, ?Q ==> ?S, ?G, ?P ]     ?R [ ?H, ?Q ==> ?S, ?G, ?P ==> ?S ]     
    --------------------------------------------------------------------
			     ?R [ ?H, ?Q ==> ?S, ?G ]	*)
read_goal SetThy "R [ H, Q==>S, G ]";
expand (rcut_tac "P==>S" 1);
expand (resolve_tac [ imp_intr RES imp_elimh ] 1);
val new_imp_hyp = ttop_rule();

     
(*  ?R [ ?H, ?G, ?P ==> ?S, ?Q ==> ?S ]     
    -----------------------------------
      ?R [ ?H, ?P | ?Q ==> ?S, ?G ]    *)
read_goal SetThy "R [ H, (P|Q)==>S, G ]";
expand (resolve_tac [new_imp_hyp] 1);
expand (resolve_tac [disj_intr1 RES assume] 1);
expand (resolve_tac [new_imp_hyp] 1);
expand (resolve_tac [disj_intr2 RES assume] 1);
expand (resolve_tac [thin] 1);
val disj_imp_elimh = ttop_rule();

(*instead of
  expand (rcut_tac "P==>S" 1);
  expand (rcut_tac "Q==>S" 2);
  expand (resolve_tac [thin] 3);
  expand (REPEAT (resolve_tac
    [imp_intr, imp_elimh, disj_intr1 RES assume, disj_intr2 RES assume ] 1));
  *)


(*  ?Q [ ?H, ?G, ?P, ?Q ==> ?S ]     ?R [ ?H, ?G, ?S ]     
    --------------------------------------------------
            ?R [ ?H, (?P ==> ?Q) ==> ?S, ?G ] 
  Allows the implication to be simplified;  classical version is stronger. *)
read_goal SetThy "R [ H, (P==>Q)==>S, G ]";
expand (resolve_tac [ cut_imp_elimh RES imp_intr ] 1);
expand (resolve_tac [new_imp_hyp] 1);
expand (resolve_tac [imp_intr RES assume] 1);
expand (resolve_tac [thin] 1);
val imp_imp_elimh = ttop_rule();

(*instead of
  expand (rcut_tac "Q==>S" 1);
  expand (resolve_tac [thin] 2);
  expand (REPEAT (resolve_tac [assume,imp_intr,imp_elimh] 1));
  *)


(*  ?R [ ?H, ?G, ?P ==> ?Q ==> ?S ]     
   ---------------------------------
     ?R [ ?H, ?P & ?Q ==> ?S, ?G ]  
  Classical version has two premises. *)
read_goal SetThy "R [ H, (P&Q)==>S, G ]";
expand (rcut_tac "P==>(Q==>S)" 1);
expand (resolve_tac [thin] 2);
expand intr_tac;
expand (resolve_tac [ imp_elimh ] 1);
expand (REPEAT (resolve_tac [assume,conj_intr] 1));
val conj_imp_elimh = ttop_rule();



(*  ?P'(x2) [ ?H, ?G ]     ?R [ ?H, ?G, ?S ]     
    ----------------------------------------
       ?R [ ?H, All(?P') ==> ?S, ?G ]    
  Meaning:  a universal implication is only useful once.
  Constructively,  ALL x.~ ~ P(x)  does not imply ~ ~ ALL x.P(x),
    so an assumption ~ ALL x.P(x) cannot be reduced to ~P(x).
  NOT COMPLETE: what if (ALL x.~ ~ P(x))==>(~ ~ ALL x.P(x)) is an assumption? *)
read_goal SetThy "R [ H, All(P')==>S, G ]";
expand (resolve_tac [ cut_imp_elimh RES all_intr ] 1);
expand (resolve_tac [thin] 1);
val all_imp_elimh = ttop_rule();



(*  ?R [ ?H, ?G, ?P'(?a) ==> ?S ]     
  ---------------------------------
  ?R [ ?H, Exists(?P') ==> ?S, ?G ]   
  could use new_imp_hyp but then ?a would be ?Ga1  
  Probably not complete: should not thin the assumption.*)
read_goal SetThy "R [ H, Exists(P')==>S, G ]";
expand (rcut_tac "P'(a)==>S" 1);
expand (resolve_tac [thin] 2);
expand (resolve_tac [imp_intr] 1);
expand (resolve_tac [ imp_elimh RES exists_intr RES assume ] 1);
val exists_imp_elimh = ttop_rule();


(*      ?R [ ?H, ?G ]     
    -----------------------
    ?R [ ?H, F ==> ?S, ?G ]	
  Deletion of a useless assumption*)
read_goal SetThy "R [ H, F==>S, G ]";
expand (resolve_tac [thin] 1);
val F_imp_elimh = ttop_rule();


(*Modus Ponens in the assumptions:
  If hyps include P==>Q and P, replace P==>Q by Q
      ?R [ ?H, ?G, ?Q ]     
  ------------------------
  ?R [ ?H, ?P ==> ?Q, ?G ] 
	constraint:   [ ?H, ?P ==> ?Q, ?G ] = [ ?H1, ?P, ?G1 ]
  WARNING: RESOLUTION WITH THIS RULE TAKES SECONDS! *)
val cut_imp_elimboth = cut_imp_elimh RES assume;




(*Sort rules to put those with fewest premises first*)
fun ruleseq rl1 rl2 =
  length(prems_of_rule rl1) <= length(prems_of_rule rl2);


(*assume is not really deterministic:
  there may be many unifiers between goal and various assumptions.
  Likewise for other rules if goal is too flexible: commits to one case.*)
val determ_rls = quicksort ruleseq
    [assume, all_elimh, F_elimh,
     all_intr, conj_intr, imp_intr, iff_intr,
     exists_elimh, conj_elimh, disj_elimh,
     disj_imp_elimh, conj_imp_elimh, exists_imp_elimh, F_imp_elimh];


(*These rules are last because they may falsify goal.
  Quantifier rules that introduce variables must also be last,
    but perhaps they are deterministic.*)

val nondeterm_rls = quicksort ruleseq
   [imp_imp_elimh, all_imp_elimh, exists_intr, cut_all_elimh];

fun pc_determ_tac pno = DETERM (REPEAT1 (filt_resolve_tac determ_rls pno));


(*Performs a single step.  Useful for examining a proof.*)


fun pc_step_tac pno =
    DETERM (filt_resolve_tac determ_rls pno)  ORELSE
    (filhyp_resolve_tac [cut_imp_elimboth] pno)  ORELSE
    (filhyp_resolve_tac [disj_intr1, disj_intr2] pno)  ORELSE
    (filhyp_resolve_tac nondeterm_rls pno);


(*Performs a big step*)
fun pc_leap_tac pno =
    (REPEAT1 (pc_determ_tac pno  ORELSE
             (filhyp_resolve_tac [cut_imp_elimboth] pno)  ORELSE
             (filhyp_resolve_tac [disj_intr1, disj_intr2] pno))
     ORELSE (filhyp_resolve_tac nondeterm_rls pno))
    THEN  smash_hobj_tac;


(*Cannot use filter_rules because could_resolve is fooled by the representation
  of assumptions.  There is no way to unify %(x)x  and  %(x)?H(P(x)), even
  though ?H is a variable.*)

val pc_tac : tactic = BREADTH_FIRST is_theorem (pc_leap_tac 1);

fun p_tac depth : tactic = BREADTH_FIRST_TO is_theorem depth (pc_leap_tac 1);


(*For enumerating proofs: does not chop alternatives.
  Doesn't work because unification diverges soon*)
fun pc_enum_tac pno = (filhyp_resolve_tac determ_rls pno    ORELSE
    filhyp_resolve_tac [cut_imp_elimboth] pno   ORELSE
    filhyp_resolve_tac [disj_intr1, disj_intr2] pno    ORELSE
    filhyp_resolve_tac nondeterm_rls pno)
    THEN  unify_constraints_tac;


(*Useful?*)
read_goal SetThy "R [ H, P, HA, P, HB ]";
expand (resolve_tac [thin] 1);
val duplicate_elimh = ttop_rule();


