(*  Title: 	goalstack
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)

(*  goal stack package (no internal proof tree!) *)

exception gstack : string;


type gstack = (rule * rule sequence) list;


fun expand_com (tac:tactic) (pairs as ((rl,_) :: _)) : gstack =
      (case  spull(tac rl)  of
	   None      => raise gstack with "expand: tactic returned no results"
	 | Some pair => (pair::pairs))
  | expand_com _ _ = raise gstack with "Empty goal stack";

fun backtrack_com [] = raise gstack with "backtrack: no alternatives"
  | backtrack_com ((_,rlstr) :: pairs') =
      (case spull rlstr of
	   None      => backtrack_com pairs'
	 | Some pair => pair::pairs');

fun state_goal_com thy (concl: term) = [ (trivial(thy,concl), null_sequence) ];

fun chop_top_com [] = raise gstack with "chop_top: Stack is empty"
  | chop_top_com (pair::pairs) = pairs;

fun chop_level_com n pairs : gstack =
  nth_tail (length pairs - n, pairs);

fun print_top_unit (pairs as ((rl,_) :: _)) = 
     (writeln ("Level " ^ string_of_int(length pairs));
      print_goal_rule rl)
  | print_top_unit [] = raise gstack with "Empty goal stack";

fun top_rule_fun ((rl,_) :: _) : rule = rl
  | top_rule_fun  []                  = raise gstack with "Empty goal stack";


fun get_goal_fun glno ((rl,_) :: _) : term =
      (case  nth_tail (glno-1, prems_of_rule rl)  of
	    [] => raise gstack with "get_goal: Goal number out of range"
	  | Q::_ => Q)
  | get_goal_fun  _ []      = raise gstack with "Empty goal stack";




(*should be a standard module for states*)


val undo_list = ref([] : gstack list);

fun get_state() : gstack = 
    case !undo_list of 
      []   => raise gstack with "get_state: No current state"
    | x::_ => x;


fun top_rule () : rule = top_rule_fun (get_state());

fun ttop_rule () : rule = tidyrule(top_rule());

fun get_goal glno : term = get_goal_fun glno (get_state());

(*Which rules could apply to goal n? *)
fun filter_rules_goal rls gno : rule list =
  filter_rules (~1, get_goal gno, rls);

fun compat_rules_goal rules gno : rule list =
  let val rtr = itlist_right insert_rule (rules, null_stree)
  in  tree_compat(rtr, head_string(get_goal gno))  end;

val print_top : unit->unit =  print_top_unit o get_state;

fun print_level n : unit =  print_top_unit (chop_level_com n (get_state()));

(*Print just the given goal number*) 
fun print_goal_n glno = 
  let val {sign, ...} = rep_rule (top_rule())
  in  print_theorem sign (get_goal glno);  prs"\n"  end;


(*Printing can raise exceptions, so the assignment occurs last*)
fun set_state newgoals : unit = 
  (print_top_unit newgoals;  undo_list := newgoals :: !undo_list);

(*Restore the previous proof state;  discard current state. *)
fun undo() = case !undo_list of
    _::newundo =>  (undo_list := newundo;  print_top()) 
  |  [] => writeln"Undo list is empty!!";

(*Set the top-level goal, initialize *)
fun state_goal thy (concl: term) : unit =  
    (undo_list := [];  set_state (state_goal_com thy concl));


fun read_goal thy (sconcl: string) =
  let val {sign,...} = rep_theory thy
  in  state_goal thy (read_theorem sign sconcl)  end;


fun make_command (com : gstack->gstack) : unit = 
  let val pairs = com (get_state())
  in set_state pairs  end;

fun expand tac : unit = make_command (expand_com tac);

fun backtrack() : unit = make_command backtrack_com;

fun chop_level level : unit = make_command (chop_level_com level);
fun chop_top () : unit = make_command (chop_top_com);
