(* LK-syntax.

   Classical Predicate Calculus + ZF Set Theory, 
   using a classical Sequent Calculus.

   Adapted from L. C. Paulson's file "pcset-syntax";
   the modifications are pointed out by comments.
*)

val Aexp  = Ground "exp";
val Asobj = Ground "sobj";  (* "hobj" in L.C.P.'s file. *)
val Asequ = Asobj-->Asobj;
val Aform = Ground "form";


val LK_lextab : lexsymbol symbol_table = symtab_of_declist
  [ 
    (* Variables. *) 

    (["A",  "B",  "C",  "D"],		IdentSy(0, Aexp)),
    (["A'", "B'", "C'", "D'"],		IdentSy(0, Aexp-->Aexp)),
    (["A''", "B''", "C''", "D''"],  	IdentSy(0, [Aexp,Aexp]--->Aexp)),

    (explode"abcdefghpquvwxyz",		IdentSy(0, Aexp)),
    (["a'","b'","c'","d'","e'","f'"],	IdentSy(0, Aexp-->Aexp)),
    (["a''","b''","c''","d''","e''","f''"],
	 				IdentSy(0, [Aexp,Aexp]--->Aexp)),
    (["P",  "Q",  "R",  "S"],		IdentSy(0, Aform)),
    (["P'", "Q'", "R'", "S'"],		IdentSy(0, Aexp-->Aform)),
    (["P''","Q''","R''","S''"], 	IdentSy(0, [Aexp,Aexp]--->Aform)),


    (["E", "F", "G", "H"],	IdentSy(0, Asequ)),

    (* Representation of sequents. *)

    (["True"],		ConstSy([Asequ,Asequ]--->Agoal)),
    (["Seqof"],		ConstSy(Aform-->Asequ)), (* "Hypof" in L.C.P.'s file*)

    (* Keywords. *)

    ([ "|-", "{", "}", "<", ">", ".", "~", "ALL", "EXISTS"], DelimSy),

    (* Infixes. *)

    (["`"],	InfixSy([Aexp,Aexp]--->Aexp,	69,	70)) ,	(*left assoc*)

    (["::"],	InfixSy([Aexp,Aexp]--->Aexp,	45,	44))  ,	(*right*)

    (["="],	InfixSy([Aexp,Aexp]--->Aform,	39,	40)) ,	(*left*)
    ([":"],	InfixSy([Aexp,Aexp]--->Aform,	39,	40)) ,	(*left*)

    (["==>"],	InfixSy([Aform,Aform]--->Aform,	10,	9)),  (*right*)
    (["|"],	InfixSy([Aform,Aform]--->Aform, 20,	19)), (*right*)
    (["&"],	InfixSy([Aform,Aform]--->Aform,	30,	29)), (*right*)

    (* Others constants. *) 

    (["not"],  	ConstSy(Aform-->Aform)), (* Negation is no longer represented
                                            by "implies false". *)

    (["All","Exists"],	ConstSy((Aexp-->Aform) --> Aform)),

    (["Collect"], 	ConstSy([Aexp, Aexp-->Aform] ---> Aexp)),

    (["Pow","Union"],	ConstSy(Aexp-->Aexp)),

    (["0","N"],		ConstSy(Aexp)),

    (["empty"],	ConstSy(Aexp)) ];

 
(* Get constants from table. *)

val True	= get_const LK_lextab "True"
and Seqof	= get_const LK_lextab "Seqof";

val All		= get_const LK_lextab "All"
and Exists	= get_const LK_lextab "Exists"
and imp   	= get_const LK_lextab "==>"
and not   	= get_const LK_lextab "not"
and zero   	= get_const LK_lextab "0"
and setcons   	= get_const LK_lextab "::"
and Collect   	= get_const LK_lextab "Collect";


(* Pair and Set notation: copied from L.C.P.'s file without modification. *)

val Pair = Const("Pair", [Aexp,Aexp]--->Aexp);

fun mk_setcons(x,y) = setcons$x$y;

fun set_elems (Const("::",_) $ a $ b) = a :: set_elems b
  | set_elems (Const("0",_)) = []
  | set_elems _ = raise match;


(* Abstract over "sobj" 
 - representation of a sequence of hypotheses or conclusions-. *)

fun abs_sobj t = Abs("sobj", Asobj, t);

(* Representation of empty sequence *)

val Sempty =  abs_sobj (Bound 0);


(* Scan a bound variable : copied from L.C.P.'s file without modification. *)

fun parse_boundvar (lxs0 as ((s, IdentSy(idx,ary)) :: lxs)) =
      if ary=Aexp then (string_of_xname(s,idx), lxs)
      else  raise parserr with 
          ("Bound variable should have expression arity", lxs0)
  | parse_boundvar ((s, OtherSy) :: lxs) = (s,lxs)
  | parse_boundvar lxs = 
      raise parserr with ("Variable expected", lxs);


(* Parsing of terms. *)

fun parse_LK_term1 boundids =
   let fun parse_quant (quant,boundids) lxs =
	let val (id, lxs') = (parse_boundvar thenkey ".") lxs;
	    val (P, lxs'') = parse_LK_term1 (id::boundids) lxs'
	in (quant $ Abs(id,Aexp,P), lxs'') end;
       fun parsetm (boundids: string list) = fn
	 (("ALL",DelimSy)  :: lxs) => parse_quant (All, boundids) lxs
       | (("EXISTS",DelimSy) :: lxs) => parse_quant (Exists,boundids) lxs
       | (("~",DelimSy) :: lxs) => 
	    let val (P,lxs') = genparse parsetm (35,boundids) lxs
	    in (not$P, lxs') end 
                                 (* Negation is now represented 
                                    by the constant "not".        *)
       | ("{",DelimSy) :: lxs =>
	    let val (elems, lxs') =
		(parepin (",", parse_LK_term1 boundids) thenkey"}") lxs
	    in (itlist_right mk_setcons (elems,zero), lxs') end
       | ("<",DelimSy) :: lxs =>
	    let val ((a,b), lxs') =
		(parse_LK_term1 boundids thenkey","  thenp
		 parse_LK_term1 boundids  thenkey">") lxs
	    in (Pair$a$b, lxs') end
       | (s,OtherSy) :: lxs => (check_bound(s,0,Aexp,boundids), lxs)
       | lxs => raise parserr with ("Illegal expression", lxs)
  in  genparse parsetm (0,boundids)  end;

val parse_LK_term = parse_LK_term1 [];


(* Parsing of sequences and sequents. The adaptations from L.C.P.'s functions
   are a consequence of a modification in the syntax:
     
   - L.C.P.'s syntax: sequent ::= conclusion "[" list of hypotheses  "]",  
   - present syntax: sequent ::= list of hypotheses "|-" list of conclusions. 
*)

(* Parsing of a non-empty sequence of hypotheses or conclusions. *)

fun parse_seq lxs = 
  let fun add_assum (P,H) =
	if (arity_of P) = Aform  then  Seqof$P$H  else  P$H;
      val (sequ, lxs') = parepin (",", parse_LK_term) lxs;
      val sbody = itlist_right add_assum (sequ, Bound 0)
  in  (abs_sobj sbody, lxs')  end;

(* Parsing of a sequence of hypotheses - hypotheses and conclusions 
   are separated by the symbol "|-" -.                                 *)

fun parse_hyp_opt (("|-", DelimSy)::lxs) = (Sempty, lxs)
  | parse_hyp_opt lxs = (parse_seq thenkey "|-") lxs;

(* Parsing of a sequence of conclusions. *) 

fun parse_con_opt [] = (Sempty, [])
  | parse_con_opt lxs = parse_seq lxs; 

(* Parsing of a complete sequent - WARNING: there is no symbol to indicate 
   the end of a sequent, therefore the function can only parse a list
   containing one sequent -.                                                *)

fun parse_thm lxs : term * lexeme list = 
  let val ((hyp,con), lxs') = (parse_hyp_opt thenp parse_con_opt) lxs
  in  (check_arity_term Agoal (True$hyp$con), lxs')    
  end;
                  (* Notice the difference between the present syntax and
                     L.C.P.'s one: hypotheses come before conclusions.     *)


(* Printing of terms. *)

fun print_LK_term1 lextab (All as (outleft,outright,boundids)) tm =
  let val printm = print_LK_term1 lextab;
      val default_print = print_functl (printm, lextab) All;
      val variant = bound_variant boundids;
      fun sround pfn = 
	    if outright>0  then  (prs"(";  pfn();  prs")")  else pfn(); 
      fun print_quant (pre,id,P) () =
	 (prs (pre ^ " " ^ id ^ ".");  printm (0, 0, id::boundids) P)
  in case tm of
      Const("All",_) $ Abs(id,_,P) => 
  	sround(print_quant("ALL", variant id, P))
    | Const("Exists",_)  $ Abs(id,_,P) =>
	sround(print_quant("EXISTS",  variant id, P))
    | Const("not",_)  $ P  =>                          (* Treatment of *)
	(prs"~ ";  printm (35, outright, boundids) P)  (* negation.    *)
    | Const("::",_) $ _ $ _ =>  
	print_list("{", "}", printm (0, 0, boundids)) (set_elems tm)
	handle match => default_print tm
    | Const("Pair",_) $ a $ b =>  
	(prs"<";  
	 printm (0, 0, boundids) a;   prs",";
	 printm (0, 0, boundids) b;   prs">") 
    | _ => default_print tm
  end;

fun print_LK_term lextab : term->unit = print_LK_term1 lextab (0, 0, []);


(* Conversion of a sequence representation into a list of hypotheses 
   or conclusions.                                                     *)

fun sequ_of_sobj (Const("Seqof",_) $ P $ rand) = P :: sequ_of_sobj rand
  | sequ_of_sobj (H $ rand) = H :: sequ_of_sobj rand
  | sequ_of_sobj (Bound 0) = []
  | sequ_of_sobj tm = raise term_error with ("Ill-formed sequence", [tm]);

fun sequ_of_term (Abs(_,_,body)) = sequ_of_sobj body
  | sequ_of_term H = [H];    (* This case should not occur. *)


(* Printing of a sequence - adaptations similar to the ones for parsing -. *)

fun print_seq lextab [] = ()
  | print_seq lextab [P] = print_LK_term lextab P
  | print_seq lextab (P::PS) = 
        (print_LK_term lextab P;  prs ", "; print_seq lextab PS);

(* Printing of a sequent - adaptations similar to the ones for parsing -. *)

fun print_goal lextab (Const("True",_) $ H $ C) =
      (print_seq lextab (sequ_of_term H); prs" |- "; 
       print_seq lextab (sequ_of_term C))  
  | print_goal lextab tm = raise term_error with ("print_goal", [tm]);
