(* LK-setres.

   Introduction and eliminations rules for set theory.
*)


(* Tactic to remove meaningless constraints 
 - copied from L. C. Paulson's file: "int-resolve" -. *) 

fun is_sobj_abs (Abs(_, Ground "sobj", _)) = true
  | is_sobj_abs _ = false;

fun is_sobj_abs_pair (t,u) = is_sobj_abs t orelse is_sobj_abs u;

fun smash_sobj_tac prf =
  let val {tpairs,...} = rep_rule prf
  in (case tpairs of
	  [] => all_tac
        | _  => if (forall is_sobj_abs_pair tpairs) then unify_constraints_tac
	        else all_tac)
     prf
  end;


(* Resolution tactic which also removes meaningless constraints. *)

fun res_and_smash_tac rls gno : tactic =
    (resolve_tac rls gno) THEN smash_sobj_tac;


(* Tactic for "cutting in" a formula given as argument. *)

fun rcut_tac stm pno : tactic = 
    (rconstrain_res_tac [ ("?P", stm) ] cut pno) THEN smash_sobj_tac; 


(* Tactic for "thinning out" the hypothesis which is given as argument. *)

fun rthinh_tac stm pno : tactic =
    (rconstrain_res_tac [ ("?Q", stm) ] thinh pno) THEN smash_sobj_tac; 


(* Tactic for "thinning out" the conclusion which is given as argument. *)

fun rthinc_tac stm pno : tactic =
    (rconstrain_res_tac [ ("?Q", stm) ] thinc pno) THEN smash_sobj_tac; 


(* iff introduction.

?H, ?P |- ?E, ?F, ?Q    ?H, ?Q |- ?E, ?F, ?P    
--------------------------------------------
     ?H |- ?E, ?P <=> ?Q, ?F
*)
read_goal LKThy "H |- E, P <=> Q, F";
expand (unfold_goal_tac ["<=>"]); 
expand (resolve_tac [and_intr RES imp_intr] 1);
expand (resolve_tac [imp_intr] 2);
val iff_intr = ttop_rule();


(* iff elimination.

?H, ?G, ?P ==> ?Q, ?Q ==> ?P |- ?E    
----------------------------------
    ?H, ?P <=> ?Q, ?G |- ?E
*)
read_goal LKThy "H, P <=> Q, G |- E";
expand (unfold_goal_tac ["<=>"]); 
expand (resolve_tac [and_elim] 1);
val iff_elim = ttop_rule();


(* Elimination rule for universal quantifier. 
   This rule is not complete, but it does not run forever.
 
?H, ?G, ?P'(?a1) |- ?E    
----------------------
?H, All(?P'), ?G |- ?E
*)
read_goal LKThy "H, All(P'), G |- E";
expand(resolve_tac [all_elim] 1);
expand(rthinh_tac "ALL u.P'(u)" 1);
val thin_all_elim = ttop_rule();


(* Introduction rule for existential quantifier.
   This rule is not complete, but it does not run forever.

 ?H |- ?E, ?F, ?P'(?a1)    
-------------------------
?H |- ?E, Exists(?P'), ?F
*)
read_goal LKThy "H |- E, Exists(P'), F";
expand(resolve_tac [exists_intr] 1);
expand(rthinc_tac "EXISTS u.P'(u)" 1);
val thin_exists_intr = ttop_rule();


(* List of Predicate-Calculus rules with one premise. *)

val rlist1 = [imp_intr, and_elim, or_intr, not_intr, not_elim, iff_elim,
             all_intr, thin_all_elim, thin_exists_intr, exists_elim];
 

(* List of Predicate-Calculus rules with two premises. *)

val rlist2 = [imp_elim, and_intr, or_elim, iff_intr];


(* Tactic for reducing a goal, using Predicate-Calculus rules.
   This tactic is a decision procedure for Propositional-Calculus.
   It is incomplete for Predicate-Calculus because of the use of
   the rules "thin_all_elim" and "thin_exists_intr".              *)

fun pc_tac gno prf = 
                ((res_and_smash_tac [assume] gno)
                 ORELSE
                 ((res_and_smash_tac rlist1 gno) THEN (pc_tac gno))
                 ORELSE
                 ((res_and_smash_tac rlist2 gno) THEN (pc_tac (gno + 1))
                                                 THEN (pc_tac gno))
                 ORELSE
                 all_tac) prf;


(* Symmetry of equality in hypotheses 

?H, ?F, ?B = ?A |- ?E    
---------------------
?H, ?A = ?B, ?F |- ?E
*)
read_goal LKThy "H, A = B, F |- E"; 
expand(rcut_tac "B = A" 1);
expand(resolve_tac [sym RES assume] 1);
expand(rthinh_tac "A = B" 1);
val symh = ttop_rule();


(* Null set introduction. 

   ?H |- ?E, ?F    
--------------------
?H |- ?E, ?a : 0, ?F
*)
read_goal LKThy "H |- E, a : 0, F";
expand(rthinc_tac "a : 0" 1);
val nullset_intr = ttop_rule();


(* Null set elimination.

?H, ?a : 0, ?G |- ?E
*)
read_goal LKThy "H, a : 0, G |- E";
expand(resolve_tac [nullset RES assume] 1);
val nullset_elim = ttop_rule();


(* Membership congruence, w.r.t. sets.

   ?H, ?G |- ?E, ?F, ?a : ?B    
----------------------------------
?H, ?A = ?B, ?G |- ?E, ?a : ?A, ?F
*)
read_goal LKThy "H, A = B, G |- E, a : A, F";
expand(resolve_tac [member_cong] 1);
expand(resolve_tac [refl] 1);
expand(resolve_tac [assume] 1);
expand(rthinh_tac "A = B" 1);
val set_member_cong = ttop_rule();


(* Membership congruence, w.r.t. elements.

   ?H, ?G |- ?E, ?F, ?b : ?A    
----------------------------------
?H, ?a = ?b, ?G |- ?E, ?a : ?A, ?F
*)
read_goal LKThy "H, a = b, G |- E, a : A, F";
expand(resolve_tac [member_cong] 1);
expand(resolve_tac [assume] 1);
expand(resolve_tac [refl] 1);
expand(rthinh_tac "a = b" 1);
val elem_member_cong = ttop_rule();


(* Replacement of a conclusion by an equivalent one.

?H |- ?E, ?F, ?Q    ?H |- ?E, ?F, ?P <=> ?Q    
-------------------------------------------
           ?H |- ?E, ?P, ?F
*)
read_goal LKThy "H |- E, P, F";
expand(rcut_tac "Q" 1);
expand(rthinc_tac "P" 1);
expand(rcut_tac "(P ==> Q) & (Q ==> P)" 2);
expand(rthinh_tac "Q" 2);
expand(rthinc_tac "P" 2);
expand(pc_tac 3);
expand(fold_goal_tac ["<=>"]);
val equivc = ttop_rule();


(* Replacement of a hypothesis by an equivalent one.

?H, ?G |- ?E, ?P <=> ?Q    ?H, ?G, ?Q |- ?E    
-------------------------------------------
          ?H, ?P, ?G |- ?E
*)
read_goal LKThy "H, P, G |- E"; 
expand(rcut_tac "P==>Q" 1);
expand(rcut_tac "(P ==> Q) & (Q ==> P)" 1);
expand(rthinh_tac "P" 1);
expand(rthinc_tac "P ==> Q" 1);
expand(pc_tac 2);
expand(pc_tac 2);
expand(rthinh_tac "P" 2);
expand(fold_goal_tac ["<=>"]);
val equivh = ttop_rule();


(* Tactic to generate an introduction rule from an iff axiom. *)

fun intr_tac rl =
    (res_and_smash_tac [equivc] 1) THEN (res_and_smash_tac [rl] 2);


(* Tactic to generate an elimination rule from an iff axiom. *)

fun elim_tac rl =
    res_and_smash_tac [equivh RES rl] 1;


(* Collection introduction.

?H |- ?E, ?F, ?a : ?A    ?H |- ?E, ?F, ?P'(?a)    
----------------------------------------------
    ?H |- ?E, ?a : Collect(?A,?P'), ?F
*)
read_goal LKThy "H |- E, a : Collect(A,P'), F";
expand (intr_tac Collect_iff);
expand (resolve_tac [and_intr] 1);
val Collect_intr = ttop_rule();


(* Collection elimination.

  ?H, ?G, ?a : ?A, ?P'(?a) |- ?E    
----------------------------------
?H, ?a : Collect(?A,?P'), ?G |- ?E
*)
read_goal LKThy "H, a : Collect(A,P'), G |- E";
expand (elim_tac Collect_iff);
expand (resolve_tac [and_elim] 1);
val Collect_elim = ttop_rule();


(* Power set introduction.

  ?H |- ?E, ?F, ?A <= ?B    
--------------------------
?H |- ?E, ?A : Pow(?B), ?F
*)
read_goal LKThy "H |- E, A : Pow(B), F";
expand (intr_tac Pow_iff);
val Pow_intr = ttop_rule();


(* Power set elimination.

  ?H, ?G, ?A <= ?B |- ?E    
--------------------------
?H, ?A : Pow(?B), ?G |- ?E
*)
read_goal LKThy "H, A : Pow(B), G |- E";
expand (elim_tac Pow_iff);
val Pow_elim = ttop_rule();


(* Big Union Introduction. Not complete.

?H |- ?E, ?F, ?A : ?a6    ?H |- ?E, ?F, ?a6 : ?C    
------------------------------------------------
       ?H |- ?E, ?A : Union(?C), ?F
*)
read_goal LKThy "H |- E, A : Union(C), F";
expand (intr_tac Union_iff);
expand (pc_tac 1);
val Union_intr = ttop_rule();


(* Big Union elimination.

?H, ?G, ?A : x5, x5 : ?C |- ?E    
------------------------------
 ?H, ?A : Union(?C), ?G |- ?E

PROVISO x5> ?A ?C ?E ?G ?H
*)
read_goal LKThy "H, A : Union(C), G |- E";
expand (elim_tac Union_iff);
expand (pc_tac 1);
val Union_elim = ttop_rule();


(* Setcons introduction.

?H |- ?E, ?F, ?a = ?b, ?a : ?B    
------------------------------
 ?H |- ?E, ?a : ?b :: ?B, ?F
*)
read_goal LKThy "H |- E, a : (b :: B), F";
expand (intr_tac setcons_iff);
expand (resolve_tac [or_intr] 1);
val setcons_intr = ttop_rule();


(* Setcons elimination.

?H, ?G, ?a = ?b |- ?E    ?H, ?G, ?a : ?B |- ?E    
----------------------------------------------
       ?H, ?a : ?b :: ?B, ?G |- ?E
*)
read_goal LKThy "H, a : (b :: B), G |- E";
expand (elim_tac setcons_iff);
expand (resolve_tac [or_elim] 1);
val setcons_elim = ttop_rule();


(* Subset introduction.

?H, x1 : ?A |- ?E, ?F, x1 : ?B    
------------------------------
   ?H |- ?E, ?A <= ?B, ?F

PROVISO x1> ?A ?B ?E ?F ?H
*)
read_goal LKThy "H |- E, A <= B, F";
expand(unfold_goal_tac ["<="]);
expand(pc_tac 1);
val subset_intr = ttop_rule();


(* Subset elimination.

?H, ?G |- ?E, ?a2 : ?A    ?H, ?G, ?a2 : ?B |- ?E    
------------------------------------------------
           ?H, ?A <= ?B, ?G |- ?E
*)
read_goal LKThy "H, A <= B, G |- E";
expand(unfold_goal_tac ["<="]);
expand(pc_tac 1);
val subset_elim = ttop_rule();


(* Equality introduction, using extensionality.

?H, x3 : ?A |- ?E, ?F, x3 : ?B    ?H, x5 : ?B |- ?E, ?F, x5 : ?A    
----------------------------------------------------------------
                     ?H |- ?E, ?A = ?B, ?F

PROVISO x3> ?A ?B ?E ?F ?H and x5> ?A ?B ?E ?F ?H
*)
read_goal LKThy "H |- E, A = B, F";
expand(resolve_tac [extensionality] 1);
expand(resolve_tac [subset_intr] 1);
expand(resolve_tac [subset_intr] 2);
val ext_intr = ttop_rule();


(* Equality elimination, using extensionality.

?H, ?G, ?A <= ?B, ?B <= ?A |- ?E    
---------------------------------
     ?H, ?A = ?B, ?G |- ?E
*)
read_goal LKThy "H, A = B, G |- E";
expand(rcut_tac "A <= B & B <= A" 1);
expand(resolve_tac [and_intr] 1);
expand(resolve_tac [subset_intr RES symh RES set_member_cong RES assume] 1);
expand(resolve_tac [subset_intr RES set_member_cong RES assume] 1);
expand(resolve_tac [and_elim] 1);
expand(rthinh_tac "A = B" 1);
val ext_elim = ttop_rule();


(* Big intersection introduction.

?H, x9 : ?A |- ?E, ?F, ?a : x9    ?H |- ?E, ?F, ?a8 : ?A    
--------------------------------------------------------
           ?H |- ?E, ?a : Inter(?A), ?F

PROVISO x9> ?A ?E ?F ?H ?a
*)
read_goal LKThy "H |- E, a : Inter(A), F";
expand(unfold_goal_tac ["Inter"]);
expand(resolve_tac [Collect_intr RES Union_intr] 1);
expand(rcut_tac "(ALL y.y : A ==> a : y) & (?a8 : A)" 1);
expand(rthinc_tac "a : ?a8" 1);
expand(res_and_smash_tac [and_intr] 1);
expand(merge_premises_tac);
expand(pc_tac 1);
expand(pc_tac 3);
val Inter_intr = ttop_rule();


(* Big intersection elimination. Not complete.

?H, ?G |- ?E, ?a4 : ?A    ?H, ?G, ?a : ?a4 |- ?E    
------------------------------------------------
        ?H, ?a : Inter(?A), ?G |- ?E
*)
read_goal LKThy "H, a : Inter(A), G |- E";
expand(unfold_goal_tac ["Inter"]);
expand(resolve_tac [Collect_elim] 1);
expand(rthinh_tac "a : Union(A)" 1);
expand(pc_tac 1);
val Inter_elim = ttop_rule();


(* Rule for duplicating a conclusion - useful since some introduction or
   elimination rules are not complete -.

?H |- ?E, ?P, ?F, ?P    
--------------------
  ?H |- ?E, ?P, ?F
*)
read_goal LKThy "H |- E, P, F";
expand(rcut_tac "P" 1);
expand(resolve_tac [assume] 2);
val duplicatec = ttop_rule();


(* Rule for duplicating a hypothesis - useful since some introduction or
   elimination rules are not complete -.

?H, ?P, ?G, ?P |- ?E    
--------------------
  ?H, ?P, ?G |- ?E
*)
read_goal LKThy "H, P, G |- E";
expand(rcut_tac "P" 1);
expand(resolve_tac [assume] 1);
val duplicateh = ttop_rule();


(* Tactic for duplicating the conclusion which is given as argument *)

fun rduplicatec_tac stm pno : tactic = 
    (rconstrain_res_tac [ ("?P", stm) ] duplicatec pno) THEN smash_sobj_tac;


(* Tactic for duplicating the hypothesis which is given as argument *)

fun rduplicateh_tac stm pno : tactic = 
    (rconstrain_res_tac [ ("?P", stm) ] duplicateh pno) THEN smash_sobj_tac;


(* Union introduction.

?H |- ?E, ?F, ?a : ?A, ?a : ?B    
------------------------------
 ?H |- ?E, ?a : ?A Un ?B, ?F
*)
read_goal LKThy "H |- E, a : A Un B, F";
expand(unfold_goal_tac ["Un"]);
expand(rduplicatec_tac "a : Union({A,B})" 1);
expand(resolve_tac [Union_intr] 1);
expand(resolve_tac [Union_intr] 1);
expand(resolve_tac [setcons_intr] 3);
expand(resolve_tac [refl] 3);
expand(resolve_tac [setcons_intr RES setcons_intr] 2);
expand(resolve_tac [refl] 2);
backtrack();
val Un_intr = ttop_rule();


(* Union elimination.

?H, ?G, ?a : ?A |- ?E    ?H, ?G, ?a : ?B |- ?E    
----------------------------------------------
       ?H, ?a : ?A Un ?B, ?G |- ?E
*)
read_goal LKThy "H, a : A Un B, G |- E";
expand(unfold_goal_tac ["Un"]);
expand(resolve_tac [Union_elim RES setcons_elim] 1);
expand(rcut_tac "a : A" 1);
expand(resolve_tac [symh RES set_member_cong RES assume] 1);
expand(rthinh_tac "a : x6" 1);
expand(rthinh_tac "x6 = A" 1);
expand(resolve_tac [setcons_elim] 2);
expand(rcut_tac "a : B" 2);
expand(resolve_tac [symh RES set_member_cong RES assume] 2);
expand(rthinh_tac "a : x6" 2);
expand(rthinh_tac "x6 = B" 2);
expand(resolve_tac [nullset_elim] 3);
val Un_elim = ttop_rule();


(* Intersection introduction.

?H |- ?E, ?F, ?a : ?A    ?H |- ?E, ?F, ?a : ?B    
----------------------------------------------
        ?H |- ?E, ?a : ?A Int ?B, ?F
*)
read_goal LKThy "H |- E, a : A Int B, F";
expand(unfold_goal_tac ["Int"]);
expand(resolve_tac [Inter_intr] 1);
expand(resolve_tac [setcons_elim] 1);
expand(resolve_tac [set_member_cong] 1);
expand(resolve_tac [setcons_elim] 2);
expand(resolve_tac [set_member_cong] 2);
expand(resolve_tac [nullset_elim] 3);
expand(resolve_tac [setcons_intr RES refl] 3);
val Int_intr = ttop_rule();


(* Intersection elimination.

?H, ?G, ?a : ?A, ?a : ?B |- ?E    
------------------------------
 ?H, ?a : ?A Int ?B, ?G |- ?E
*)
read_goal LKThy "H, a : A Int B, G |- E";
expand(unfold_goal_tac ["Int"]);
expand(rduplicateh_tac "a : Inter({A,B})" 1);
expand(resolve_tac [Inter_elim] 1);
expand(resolve_tac [Inter_elim] 2);
expand(resolve_tac [setcons_intr RES refl] 1);
expand(resolve_tac [setcons_intr RES setcons_intr RES refl] 1);
backtrack();
val Int_elim = ttop_rule();
