`include "malwa_defines.v"

module malwa_decoder(
	input [31:0]	instr,
	output reg [13:0]	controls,
	output reg [4:0]	alucontrol,
	output reg [2:0]	branchtype,
	output [4:0]	destreg
);

  assign destreg = controls[5]			?	5'd31			// LINK
					: controls[12]		?	instr[20:16]	// USEIMM
					:						instr[15:11];

  wire [5:0] op = instr[31:26];
  wire [5:0] funct = instr[5:0];

  //Zero extend this operation
  parameter ZEROX=		14'b0000_0001_0000_00;
  //Ask for 16 bits from the bus
  parameter MEM16=		14'b0000_0000_0100_00;
  //Ask for 8 bits from the bus
  parameter MEM8=		14'b0000_0000_0010_00;
  //Ask for left from the bus
  parameter MEML=		14'b0000_0000_0001_00;
  //Ask for right from the bus
  parameter MEMR=		14'b0000_0000_0000_10;

  //Do a link in addition to another instruction, by ensuring
  // regwrite is high to store to register $31 and the link flag
  parameter LINK=		14'b1000_0000_1000_00;

  //Set the regwrite flag to store result in a register
  parameter ALU=		14'b1000_0000_0000_00;
  //Set the regjump flag to do a register jump
  parameter REGJUMP=	14'b0000_0010_0000_00;
  //Set the branch flag to do a branch
  parameter BRANCH=		14'b0010_0000_0000_00;
  //Set the jump flag to do a jump
  parameter JUMP=		14'b0000_0100_0000_00;

  //Set the regwrite flag to store results in a register
  // and the useimm flag to use the 16bit immediate as an operand
  parameter IMMOP=		14'b1100_0000_0000_00;

  //Set the irqreturn flag to return from the interrupt
  parameter ERET=		14'b0000_0000_0000_01;

  //Set regwrite to store memory to a register									
  // Set useimm to calculate the address
  // Set memread to do a memory read
  parameter LOAD=		14'b1100_1000_0000_00;

  //Set useimm to calculate the address
  // Set memwrite to do a memory write
  parameter STORE=		14'b0101_0000_0000_00;

  parameter UNKNOWN=	14'bXXXX_XXXX_XXXX_XX;


  always @(*)
    case(op)
      6'b000000:
		begin
			if( funct == 6'b001000 )
				controls <= REGJUMP;
			else if( funct == 6'b001001 )
				controls <= REGJUMP | LINK;
			else
				controls <= ALU;
			case(funct)          // RTYPE
			 6'b010000: alucontrol <= `ALU_MFHI;
			 6'b010001: alucontrol <= `ALU_MTHI;
			 6'b010010: alucontrol <= `ALU_MFLO;
			 6'b010011: alucontrol <= `ALU_MTLO;
			
			 6'b011000: alucontrol <= `ALU_MULT;
			 6'b011001: alucontrol <= `ALU_MULT | `ALU_UNSIGNED;
			
			 6'b011010: alucontrol <= `ALU_DIV;
			 6'b011011: alucontrol <= `ALU_DIV | `ALU_UNSIGNED;
			
	         6'b100000: alucontrol <= `ALU_ADD;
    	     6'b100001: alucontrol <= `ALU_ADD | `ALU_UNSIGNED;
        	 6'b100010: alucontrol <= `ALU_SUB;
	         6'b100011: alucontrol <= `ALU_SUB | `ALU_UNSIGNED;

	         6'b100100: alucontrol <= `ALU_AND;
    	     6'b100101: alucontrol <= `ALU_OR;
        	 6'b100110: alucontrol <= `ALU_XOR;
	         6'b100111: alucontrol <= `ALU_NOR;

    	     6'b101010: alucontrol <= `ALU_SLT;
	         6'b101011: alucontrol <= `ALU_SLT | `ALU_UNSIGNED;

	         6'b000000: alucontrol <= `ALU_SL | `ALU_UNSIGNED;
	         6'b000010: alucontrol <= `ALU_SR | `ALU_UNSIGNED;
	         6'b000011: alucontrol <= `ALU_SR;

	         6'b000100: alucontrol <= `ALU_SLV | `ALU_UNSIGNED;
	         6'b000110: alucontrol <= `ALU_SRV | `ALU_UNSIGNED;
	         6'b000111: alucontrol <= `ALU_SRV;
	
        	 default:   alucontrol <= `ALU_NONE;
	        endcase
			branchtype <= `BR_NONE;
		end

      6'b000001:	// BLTZ or BGEZ
		begin
			controls <= BRANCH;
			alucontrol <= `ALU_NONE;
			if( instr[16] == 0 )
				branchtype <= `BR_LTZ;
			else
				branchtype <= `BR_GEZ;
		end

      6'b000010:	// J
		begin
			controls <= JUMP;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_NONE;
		end

      6'b000011:	// JAL
		begin
			controls <= JUMP | LINK;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_NONE;
		end

      6'b000100:	 // BEQ
		begin
			controls <= BRANCH;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_EQ;
		end

      6'b000101:	// BNE
		begin
			controls <= BRANCH;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_NE;
		end

      6'b000110:	// BLEZ
		begin
			controls <= BRANCH;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_LEZ;
		end

      6'b000111:	// BGTZ
		begin
			controls <= BRANCH;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_GTZ;
		end

      6'b001000:	// ADDI
		begin
			controls <= IMMOP;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b001001:	// ADDIU
		begin
			controls <= IMMOP;
			alucontrol <= `ALU_ADD | `ALU_UNSIGNED;
			branchtype <= `BR_NONE;
		end
		
      6'b001010:	// SLTI
		begin
			controls <= IMMOP;
			alucontrol <= `ALU_SLT;
			branchtype <= `BR_NONE;
		end

      6'b001011:	// SLTIU
		begin
			controls <= IMMOP;
			alucontrol <= `ALU_SLT | `ALU_UNSIGNED;
			branchtype <= `BR_NONE;
		end

      6'b001100:	// ANDI
		begin
			controls <= IMMOP | ZEROX;
			alucontrol <= `ALU_AND;
			branchtype <= `BR_NONE;
		end

      6'b001101:	// ORI
		begin
			controls <= IMMOP | ZEROX;
			alucontrol <= `ALU_OR;
			branchtype <= `BR_NONE;
		end

      6'b001110:	// XORI
		begin
			controls <= IMMOP | ZEROX;
			alucontrol <= `ALU_XOR;
			branchtype <= `BR_NONE;
		end

	  6'b001111:	// LUI
		begin
			controls <= IMMOP;
			alucontrol <= `ALU_LUI; 
			branchtype <= `BR_NONE;
		end
		
	  6'b010000:	// ERET (possibly)
		begin
			if( {instr[25:21], instr[5:0]} == 11'b10000_011000 )
				controls <= ERET;
			else
				controls <= UNKNOWN;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_NONE;
		end

	  6'b010001:	// JERET
		begin
			controls <= ERET | JUMP;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_NONE;
		end

	  6'b011100:	// MUL (possibly)
		begin
			if( funct == 6'b000010 )
			begin
				controls <= ALU;
				alucontrol <= `ALU_MUL;
			end
			else
			begin
				controls <= UNKNOWN;
				alucontrol <= `ALU_NONE;
			end
			branchtype <= `BR_NONE;
		end

      6'b100000:	// LB
		begin
			controls <= LOAD | MEM8;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b100001:	// LH
		begin
			controls <= LOAD | MEM16;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end
		
      6'b100010:	 // LWL
		begin
			controls <= LOAD | MEML;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b100011:	 // LW
		begin
			controls <= LOAD;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b100100:	// LBU
		begin
			controls <= LOAD | MEM8 | ZEROX;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b100101:	// LHU
		begin
			controls <= LOAD | MEM16 | ZEROX;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b100110:	 // LWR
		begin
			controls <= LOAD | MEMR;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b101000:	// SB
		begin
			controls <= STORE | MEM8;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b101001:	// SH
		begin
			controls <= STORE | MEM16;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      6'b101011:	// SW
		begin
			controls <= STORE;
			alucontrol <= `ALU_ADD;
			branchtype <= `BR_NONE;
		end

      default:
		begin
			controls <= UNKNOWN;
			alucontrol <= `ALU_NONE;
			branchtype <= `BR_NONE;
		end
    endcase
endmodule