/* (C) Cambridge University Computer Laboratory, 2002
 *     All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Systems Research
 *      Group at Cambridge University Computer Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <string.h>
#include "stm0.h"
#include "stm_test.h"

#ifndef EXTERNAL_STM_CONFIG
#define MAX_THREADS 5

/***********************************************************************/

/*
 * Algorithm configuration
 *
 * DO_STM_ABORT_SELF => cause current transaction to abort if it finds
 * it needs an owned location when committing.  Does not give any progress 
 * guarantee in the case of stuck transactions.
 *
 * DO_STM_DEFER_OTHER => cause current transaction to spin if it finds
 * another one in progress.  Again, no progress guarantee but if
 * commit steps are fast then it avoids aborting.
 *
 * DO_STM_ABORT_SELF => cause current transaction to cause others it
 * sees to abort.  Obstruction free non-blocking behaviour.
 */

/* #define DO_STM_ABORT_SELF   */
#define DO_STM_DEFER_OTHER  
/* #define DO_STM_ABORT_OTHER  */

#define STM_OWNERSHIP_RECORDS 2048

/***********************************************************************/

/*
 * Memory management configuration.  Un-comment one line to select the
 * scheme used to manage STM descriptors.
 *
 * Malloc / GC indicates how fresh storage space is obtained: malloc
 * fetches it in blocks of size MM_BLOCK_SIZE to per-thread local
 * allocators.  GC allocates descriptors individually under the 
 * assumption that the upstream allocator is optimised for
 * frequent small allocations.
 *
 * Simple => do not attempt to re-use storage space.
 *
 * RC => perform reference counting on descriptors.  Per-thread
 * free lists hold descriptors eligible for re-use.
 *
 * PD => 'private or discard' algorithm.  A thread continues to
 * re-use a descriptor until it has been seen by another thread,
 * at which point it is discarded and a fresh descriptor allocated.
 *
 * SMR => Michael's Safe Memory Re-use algorithm.  Implementation
 * following PODC 2002.
 *
 * PTB => Pass-the-buck ROP solution.  Implementation following 
 * DISC 2002.
 *
 * Note that MallocSimple and MallocPD allow neither de-allocation to
 * the OS nor guaranteed re-use.  They're here for comparison
 * rather than serious use.  MallocRC does not allow de-allocation
 * to the OS.
 */

#define DO_PRIMARY_MALLOC   
/* #define DO_PRIMARY_GC       */

/* #define DO_SECONDARY_SIMPLE */
#define DO_SECONDARY_RC     
/* #define DO_SECONDARY_PD     */
/* #define DO_SECONDARY_SMR    */
/* #define DO_SECONDARY_PTB    */

/***********************************************************************/

#endif /* EXTERNAL_STM_CONFIG */

#define MAX_LOCATIONS 10
#define OREC_HASH(ptr) ((((uintptr_t) (ptr)) >> 2) & (STM_OWNERSHIP_RECORDS - 1))

/*
 * Size of malloc() requests made by malloc-based allocators
 */

#define MALLOC_BLOCK_SIZE    64*1024

/* 
 * If a per-thread object cache reaches the high water mark then it places
 * items beyond the low water mark on a shared stealable queue.  Other
 * threads will steal the entire queue before doing a MALLOC.  NB: the
 * "LWM" amount of objects are saved by linear traversal of the object
 * cache lists.  It is intended as a simple safeguard against oscillating,
 * not as a decent sharing mechanism.
 */

#define OBJECT_CACHE_HWM         4*1024*1024
#define OBJECT_CACHE_LWM                2048

#if (OPT_CASPTRINT_AVAILABLE == 1)
#define DO_BALANCE_OBJECT_CACHES
#endif

/*
 * Number of objects that a thread de-allocates before performing
 * a scan when using the SMR or PTB algorithms.
 */

#define DEFERRED_FREE_BATCH_SIZE     128

/***********************************************************************/

#define GUARDS_PER_THREAD       (MAX_LOCATIONS + 2)
#define MAX_GUARDS (MAX_THREADS * GUARDS_PER_THREAD)
#define GUARD_PROTECTS(gp, op) ((gp) == (op))

#define GET_ST(_e) ((STMState_t *) GET_ENV_STATE(_e))
#define SET_ST(_e, _s) SET_ENV_STATE((_e), ((void *) (_s)))

/***********************************************************************/

/* Testing options.  None of these should be enabled for production use */

/*
 * KILL_ON_DEALLOCATE causes de-allocated storage to be filled with
 * DEAD_VALUE at the point at which it could be returned to the OS.
 * Helps trap use-after-freeing errors.
 */

#ifdef DEBUG
#define KILL_ON_DEALLOCATE
#define do_debug 1
#else
#define do_debug 0
#endif

#define DO_TIMINGS 
#define DO_COUNTS

#define DO_ALLOCATION_COUNT

/* #define DO_DUMP_ORECS */

/***********************************************************************/

/* Convert 'public' options above into internal ones and test for sanity */

#ifdef DO_STM_ABORT_SELF
#define OPT_ABORT_SELF 1
#else
#define OPT_ABORT_SELF 0
#endif

#ifdef DO_STM_DEFER_OTHER 
#define OPT_DEFER_OTHER  1
#else
#define OPT_DEFER_OTHER  0
#endif

#ifdef DEBUG
#define OPT_DEBUG 1
#define OPT_DEBUG_MSG 0
#else
#define OPT_DEBUG 0
#define OPT_DEBUG_MSG 0
#endif

/*
 * DO_PRIMARY_GC => may need to flush data structures on each collection
 */

#if (defined (DO_PRIMARY_GC))
#define OPT_NEED_POST_GC_HOOK 1
#define OPT_NEED_PRE_GC_HOOK 1
#else
#define OPT_NEED_POST_GC_HOOK 0
#define OPT_NEED_PRE_GC_HOOK 1
#endif

#ifdef DO_PRIMARY_MALLOC
#define MM_PRIMARY_SCHEME Malloc
#endif

#ifdef DO_PRIMARY_GC
#define MM_PRIMARY_SCHEME GC
#endif

#ifdef DO_SECONDARY_SIMPLE
#define MM_SECONDARY_SCHEME Simple
#endif

#ifdef DO_SECONDARY_RC
#define MM_SECONDARY_SCHEME RC
#endif

#ifdef DO_SECONDARY_PD
#define MM_SECONDARY_SCHEME PD
#endif

#ifdef DO_SECONDARY_SMR
#define MM_SECONDARY_SCHEME SMR
#endif

#ifdef DO_SECONDARY_PTB
#define MM_SECONDARY_SCHEME PTB
#endif

#define PASTE2(x,y) x ## y
#define PASTE(x,y) PASTE2(x,y)

#define PrimaryAllocateBytes   PASTE (PrimaryAllocateBytes, MM_PRIMARY_SCHEME)
#define PrimaryDeallocateObj   PASTE (PrimaryDeallocateObj, MM_PRIMARY_SCHEME)
#define PrimaryInitState       PASTE (PrimaryInitState, MM_PRIMARY_SCHEME)
#define PrimaryReclaimState    PASTE (PrimaryReclaimState, MM_PRIMARY_SCHEME)
#define PrimaryDoneGC          PASTE (PrimaryDoneGC, MM_PRIMARY_SCHEME)
#define PrimaryInitScheme      PASTE (PrimaryInitScheme, MM_PRIMARY_SCHEME)

#define MMAllocateBytes        PASTE (MMAllocateBytes, MM_SECONDARY_SCHEME)
#define MMDeallocateObj        PASTE (MMDeallocateObj, MM_SECONDARY_SCHEME)
#define MMGuardObj             PASTE (MMGuardObj, MM_SECONDARY_SCHEME)
#define MMReleaseObj           PASTE (MMReleaseObj, MM_SECONDARY_SCHEME)
#define MMInitState            PASTE (MMInitState, MM_SECONDARY_SCHEME)
#define MMReclaimState         PASTE (MMReclaimState, MM_SECONDARY_SCHEME)
#define MMDoneGC               PASTE (MMDoneGC, MM_SECONDARY_SCHEME)
#define MMInitScheme           PASTE (MMInitScheme, MM_SECONDARY_SCHEME)

/***********************************************************************/

/*
 * Timing settings
 */

#define TIMING_ALLOCATION          0
#define TIMING_COMMIT              1
#define TIMING_ABORT               2
#define TIMING_WRITE               3
#define TIMING_READ                4
#define TIMING_WAIT                5
#define TIMING_ASLEEP              6

#define NUM_TIMINGS 7

#define COUNT_COMMIT_SUCCEEDS      0
#define COUNT_COMMIT_FAILS         1
#define COUNT_ABORTS               2
#define COUNT_WAITS                3
#define COUNT_RC_UP                4
#define COUNT_RC_DOWN              5

#define NUM_COUNTS 6

#ifdef DO_TIMINGS
#define START_TIMING(_i) { tick_t start ## _i = RDTICK(); 
#define END_TIMING(_st,_i) (_st) -> timings[_i] += RDTICK() - (start ## _i);  stm_assert (RDTICK() > (start ## _i)); }
#define TEMP_EXIT_TIMING(_i) { tick_t start_exit ## _i = RDTICK();
#define TEMP_ENTER_TIMING(_i) start ## _i += (RDTICK() - start_exit ## _i); }
#else
#define START_TIMING(_i) 
#define END_TIMING(_st,_i)
#define TEMP_EXIT_TIMING(_i)
#define TEMP_ENTER_TIMING(_i)
#endif

#ifdef DO_COUNTS
#define COUNT(_st,_i) (_st) -> counts[_i] ++;
#endif


/***********************************************************************/
/*
 * Booleans
 */

#if (OPT_NEED_BOOL_T == 1)
typedef enum { FALSE = 0,
	       TRUE = 1 } bool_t;
#endif

/*
 * Dummy heap definitions when not actually available.  Lets us compile
 * as much of this file as possible to avoid ifdefs later and to aid testing.
 */

#if (OPT_GC_HEAP_AVAILABLE != 1)

static void *tentativelyAllocateBytes (void *e, int bytes)
{
  stm_panic (("GC heap used but not available"));
}

#define GC_WRITE_BARRIER(_a, _b) /* Nothing */

#endif /* OPT_GC_HEAP_AVAILABLE != 1 */

#if (OPT_MALLOC_HEAP_AVAILABLE != 1)
static void *dummy_malloc (int s) {
  stm_panic (("Malloc heap used but not available"));
}

static void dummy_free (void *v) {
  stm_panic (("Malloc heap used but not available"));
}

#define mallocBytes(_x) dummy_malloc (_x)
#define freeMallocObj(_x) dummy_free (_x)
#endif /* OPT_MALLOC_HEAP_AVAILABLE != 1 */

#if (OPT_CASPTRINT_AVAILABLE != 1)
static int CASPtrInt (void *a, void *op, void *np, int oi, int ni)
{
  stm_panic (("CASPtrInt used but not available"));
}
#endif /* OPT_CASPTRINT_AVAILABLE != 1 */

/***********************************************************************/

typedef struct STMDescriptor_t STMDescriptor_t;

static bool_t is_stm_descriptor (void *ptr);

/***********************************************************************/

/*
 * Secondary MM schemes that wish to use the common object-cache
 * functions must place a SecondaryStateOC_t at the start of their
 * own state structure and OCHeader_t at the end of their header 
 * structure.
 */

typedef struct OCHeader_t OCHeader_t;
typedef struct RCHeader_t RCHeader_t;
typedef struct PDHeader_t PDHeader_t;

struct OCHeader_t {
  OCHeader_t *next;
};

typedef struct {
  OCHeader_t *object_cache_1;
  OCHeader_t *object_cache_2;
} SecondaryStateOC_t;

typedef struct {
  char *block;
  int bytes_left;
} PrimaryStateMalloc_t;

struct RCHeader_t {
  volatile int  rc;
  OCHeader_t    och;
};

struct PDHeader_t {
  volatile uintptr_t isPrivate;
};

typedef struct {
  PDHeader_t *last_header;
} SecondaryStatePD_t;

typedef struct {
  SecondaryStateOC_t  oc;
  OCHeader_t         *dlist;
  int                 dcount;
} SecondaryStateSMR_t;

typedef struct {
  SecondaryStateOC_t  oc;
  OCHeader_t         *dlist;
  int                 dcount;
} SecondaryStatePTB_t;


struct STMState_t {
  int                idx;
  STMDescriptor_t   *current_transaction;
  int tn;
  union {
    PrimaryStateMalloc_t malloc;
  } primary_state;
  union {
    SecondaryStateOC_t  oc;
    SecondaryStatePD_t  pd;
    SecondaryStateSMR_t smr;
    SecondaryStatePTB_t ptb;
  } secondary_state;
#ifdef DO_TIMINGS
  tick_t init_time;
  tick_t timings[NUM_TIMINGS];
#endif
#ifdef DO_COUNTS
  int counts[NUM_COUNTS];
#endif
#ifdef DO_ALLOCATION_COUNT
  int allocation_count;
#endif
  int object_cache_1_size;
  int object_cache_2_size;
  STMState_t        *next_state;
  STM_ENV     *env;
};

typedef struct {
  addr_t addr;
  word_t new_val;
  word_t new_version;
  word_t old_val;
  word_t old_version;
  STMDescriptor_t *waiting_with;
  bool_t does_own;
} STMEntry_t;

typedef struct STMSleepers_t STMSleepers_t;

typedef struct OwnershipRecord_t OwnershipRecord_t;

struct STMSleepers_t {
  STMState_t *st; /* Must be a ptr to distinguish from descriptors */
  int tn;
  STMSleepers_t *next;
};

/* 
 * STM_ACTIVE => transaction has been started but not yet committed
 * or aborted.
 *
 * STM_COMMITTED => transaction has been selected as committed, updates
 * may still be in progress.
 *
 * STM_ABORTED => transaction has been selected as aborted, trasnactional
 * read/write may still occur until it notices.
 */

struct STMDescriptor_t {
  volatile STMResult_t status; 
  volatile int                  length;
  volatile bool_t cd;
  STMMutex_t           mutex;
  STMCond_t            cond;
   STMEntry_t           entries[MAX_LOCATIONS];
  STMSleepers_t       *sleepers;
};

/*
 * Version information
 */

union STMVersionInfo_t {
  word_t            version; /* Must be odd */
  STMDescriptor_t * owner;   /* Aligned ptr */
};

#define VI_IS_VERSION(u) (((u).version) & 1)

/*
 * Ownership records
 */

struct OwnershipRecord_t {
  volatile union STMVersionInfo_t u;
  volatile int   updaters;
};

static OwnershipRecord_t *orecs;

#define OREC_OF(ptr) (&orecs[OREC_HASH(ptr)])

#define OREC_IS_OWNED(orec) (!(((orec) -> u.version) & 1))


/***********************************************************************/

static int get_and_adjust_slow (volatile int * ptr, int delta);
static int get_and_adjust (volatile int * ptr, int delta);

static void *PrimaryAllocateBytes (STMState_t *st, int bytes);
static void  PrimaryDeallocateObj (STMState_t *st, void *ptr);
static void  PrimaryInitState (STMState_t *st);
static void  PrimaryReclaimState (STMState_t *st);
static void  PrimaryDoneGC (STMState_t *st);
static void  PrimaryInitScheme (void);

static void *MMAllocateBytes (STMState_t *st, int bytes);
static void  MMDeallocateObj (STMState_t *st, void *ptr);
static void *MMGuardObj (STMState_t *st, int g, void *ptr);
static void  MMReleaseObj (STMState_t *st, int g, void *ptr);
static void  MMInitState (STMState_t *st);
static void  MMReclaimState (STMState_t *st);
static void  MMDoneGC (STMState_t *st);
static void  MMInitScheme (void);

/***********************************************************************/

static int get_and_adjust_slow (volatile int * ptr, int delta)
{
  int old;
  int seen;

  NO_INLINE();
  old = *ptr;
  MB();
  while (TRUE) {
    seen = CASInt (ptr, old, old + delta);
    if (seen == old) break;
    old = seen;
  }
 
  return old;
}

static int get_and_adjust (volatile int * ptr, int delta)
{
  int old;
  
MB();
  old = *ptr;
  if (CASInt (ptr, old, old + delta) != old) {
    old = get_and_adjust_slow (ptr, delta);
  }
MB();

  return old;  
}

/***********************************************************************/

#define SEC_TO_PRI_ALIGN_BYTES(_b) (_b)
#define PRI_TO_SEC_ALIGN_PTR(_p) (_p)

static uintptr_t mm_header_size = 0;

/***********************************************************************/

/*
 * Primary: malloc
 */

static void *PrimaryAllocateBytesMalloc (STMState_t *st, 
					 int bytes) {
  void *result;
  PrimaryStateMalloc_t *psm;

  bytes = (bytes + REQ_NATIVE_ALIGNMENT - 1) & (~(REQ_NATIVE_ALIGNMENT - 1));
  psm = &(st -> primary_state.malloc);

  stm_assert (bytes <= MALLOC_BLOCK_SIZE);
  if (psm -> bytes_left < bytes) {
    psm -> block = malloc(MALLOC_BLOCK_SIZE);
    psm -> bytes_left = MALLOC_BLOCK_SIZE;
  }

  result = psm -> block;
  psm -> block += bytes;
  psm -> bytes_left -= bytes;

#ifdef DO_ALLOCATION_COUNT
  st -> allocation_count += bytes;
#endif

  stm_assert (result != NULL);

  return result;
}

static void PrimaryDeallocateObjMalloc (STMState_t *st, void *ptr) {
  stm_panic (("PrimaryDeallocateObjMalloc not implemented"));
}

static void PrimaryInitStateMalloc (STMState_t *st) {
  PrimaryStateMalloc_t *psm;

  psm = &(st -> primary_state.malloc);
  psm -> block = NULL;
  psm -> bytes_left = 0;
}

static void PrimaryReclaimStateMalloc (STMState_t *st) {
  /* Should avoid losing the remaining space in the current block  */
  stm_panic (("PrimaryReclaimStateMalloc not implemented"));
}

static void PrimaryDoneGCMalloc (STMState_t *st) {
  /* Nothing */
}

static void PrimaryInitSchemeMalloc (void) {
  /* Nothing */
}

/***********************************************************************/

/*
 * Primary: GC
 */

static void *PrimaryAllocateBytesGC (STMState_t *st, int bytes) {
  void       *result;

  result = tentativelyAllocateBytes (st -> env, bytes);

#ifdef DO_ALLOCATION_COUNT
  st -> allocation_count += bytes;
#endif

  return result;
}

static void PrimaryDeallocateObjGC (STMState_t *st, void *ptr) {
  /* Nothing */
}

static void PrimaryInitStateGC (STMState_t *st) {
  /* Nothing */
}

static void PrimaryReclaimStateGC (STMState_t *st) {
  /* Nothing */
}

static void PrimaryDoneGCGC (STMState_t *st) {
  /* Nothing */
}

static void PrimaryInitSchemeGC (void) {
  /* Nothing */
}

/***********************************************************************/

/*
 * Secondary: information supplied by the client
 */

static int clientObjToBytes (void *ptr);
static void clientKillObject (void *ptr);

/*
 * Secondary: common object cache functions
 */

typedef struct {
  OCHeader_t *shared_oc;
  int size;
} OCBalanceInfo_t;

static volatile OCBalanceInfo_t *shared_oc_1;
static volatile OCBalanceInfo_t *shared_oc_2;

static OCHeader_t *OCObjToHeader (void *ptr)
{
  OCHeader_t *h;
  h = ((OCHeader_t *) ptr) - 1;
  return h;
}

static void *OCHeaderToObj (OCHeader_t *och)
{
  void *result;
  result = (void *) (och + 1);
  return result;
}

static void add_to_object_cache (STMState_t *st, void *ptr)
{
  OCHeader_t *och;
  och = OCObjToHeader (ptr);
  stm_assert (och -> next == NULL);

#ifdef DO_BALANCE_OBJECT_CACHES
  if ((st -> object_cache_1_size > OBJECT_CACHE_HWM) &&
      (shared_oc_1 -> shared_oc == NULL))
    {
      OCHeader_t *toch;
      OCHeader_t **ptoch;
      int keep;
      toch = st -> secondary_state.oc.object_cache_1;
      ptoch = &(st -> secondary_state.oc.object_cache_1);
      keep = 0;
      while (keep < OBJECT_CACHE_LWM) {
	keep += clientObjToBytes(OCHeaderToObj(toch));
	ptoch = &(toch -> next);
	toch = toch -> next;
	stm_assert (toch != NULL);
      }
      
      if ((CASPtrInt ((long long *) shared_oc_1, 
		      NULL, 
		      (void *) toch,
		      0,
		      st -> object_cache_1_size - keep)))
	{
	  *ptoch = NULL;
	  st -> object_cache_1_size = keep;
	}
    }

  if ((st -> object_cache_2_size > OBJECT_CACHE_HWM) &&
      (shared_oc_2 -> shared_oc == NULL))
    {
      OCHeader_t *toch;
      OCHeader_t **ptoch;
      int keep;
      toch = st -> secondary_state.oc.object_cache_2;
      ptoch = &(st -> secondary_state.oc.object_cache_2);
      keep = 0;
      while (keep < OBJECT_CACHE_LWM) {
	keep += clientObjToBytes(OCHeaderToObj(toch));
	ptoch = &(toch -> next);
	toch = toch -> next;
	stm_assert (toch != NULL);
      }
      
      if ((CASPtrInt ((long long *) shared_oc_2, 
		      NULL, 
		      (void *) toch,
		      0,
		      st -> object_cache_2_size - keep)))
	{
	  *ptoch = NULL;
	  st -> object_cache_2_size = keep;
	}
    }
#endif

  if (!is_stm_descriptor (ptr)) {
    stm_assert (clientObjToBytes (ptr) == sizeof (STMSleepers_t));
    st -> object_cache_1_size += clientObjToBytes (ptr);
    och -> next = st -> secondary_state.oc.object_cache_1;
    st -> secondary_state.oc.object_cache_1 = och;
  } else {
    stm_assert (clientObjToBytes (ptr) == sizeof (STMDescriptor_t));
    st -> object_cache_2_size += clientObjToBytes (ptr);
    och -> next = st -> secondary_state.oc.object_cache_2;
    st -> secondary_state.oc.object_cache_2 = och;
  }
}

static void *find_from_object_cache (STMState_t *st, int bytes)
{
  OCHeader_t  *result;
  OCHeader_t **ptr;

  if (bytes == sizeof (STMSleepers_t)) 
    {
      ptr = &(st -> secondary_state.oc.object_cache_1);
      result = *ptr;
#ifdef DO_BALANCE_OBJECT_CACHES
      if (result == NULL)
	{
	  int result_size;
	  do {
	    MB();
	    result = shared_oc_1 -> shared_oc;
	    result_size = shared_oc_1 -> size;
	  } while (!CASPtrInt((long long *) shared_oc_1, 
			      result, NULL,
			      result_size, 0));
	  if (result != NULL) {
	    st -> secondary_state.oc.object_cache_1 = result;
	    st -> object_cache_1_size = result_size;
	    result = *ptr;
	  }
	}
#endif
    } else {
      ptr = &(st -> secondary_state.oc.object_cache_2);
      result = *ptr;
#ifdef DO_BALANCE_OBJECT_CACHES
      if (result == NULL)
	{
	  int result_size;
	  do {
	    MB();
	    result = shared_oc_2 -> shared_oc;
	    result_size = shared_oc_2 -> size;
	  } while (!CASPtrInt((long long *) shared_oc_2, 
			      result, NULL,
			      result_size, 0));
	  if (result != NULL) {
	    st -> secondary_state.oc.object_cache_2 = result;
	    st -> object_cache_2_size = result_size;
	    result = *ptr;
	  }
	}
#endif
    }

  if (result != NULL)
    {
      *ptr = result -> next;
#ifdef DEBUG
      result -> next = NULL;
#endif
      if (bytes == sizeof (STMSleepers_t)) 
	{
	  st -> object_cache_1_size -= clientObjToBytes (OCHeaderToObj(result));
	} 
      else
	{
	  st -> object_cache_2_size -= clientObjToBytes (OCHeaderToObj(result));
	}
      return OCHeaderToObj(result);
    }
  else
    {
      return NULL;
    }  
}

static void *allocate_via_object_cache (STMState_t *st, int bytes)
{
  OCHeader_t  *result;

  result = find_from_object_cache (st, bytes);
  if (result == NULL)
    {
      result = PrimaryAllocateBytes (st, bytes + sizeof (OCHeader_t));
      if (result != NULL)
	{
#ifdef DEBUG
	  result -> next = NULL;
#endif
	  result = OCHeaderToObj (result);
	}
    }

  return result;
}

/* Discard cached objects */
static void drop_object_cache (STMState_t *st) 
{
#if (MM_PRIMARY_SCHEME != GC)
  if ((st -> secondary_state.oc.object_cache_1 != NULL) ||
      (st -> secondary_state.oc.object_cache_2 != NULL))
    {
      stm_panic (("Irretrievably dropping object cache contents"));
    }
#endif
  st -> secondary_state.oc.object_cache_1 = NULL;
  st -> secondary_state.oc.object_cache_2 = NULL;
}

/* Start-of-day initialization */
static void init_object_cache (STMState_t *st) 
{
  st -> secondary_state.oc.object_cache_1 = NULL;
  st -> secondary_state.oc.object_cache_2 = NULL;
}

/***********************************************************************/

/*
 * Secondary: Simple
 */

static void *MMAllocateBytesSimple (STMState_t *st, int bytes) {
  void *result;
  result = PrimaryAllocateBytes (st, bytes);
  return result;
}

static void MMDeallocateObjSimple (STMState_t *st, void *ptr) {
  /* Nothing */
}

static void *MMGuardObjSimple (STMState_t *st, int g, void *ptr) {
  return NULL;
}

static void MMReleaseObjSimple (STMState_t *st, int g, void *ptr) {
  /* Nothing */
}

static void MMInitStateSimple (STMState_t *st) {
  PrimaryInitState(st);
}

static void MMReclaimStateSimple (STMState_t *st) {
  PrimaryReclaimState(st);
}

static void MMDoneGCSimple (STMState_t *st) {
  PrimaryDoneGC (st);
}

static void MMInitSchemeSimple (void) {
  mm_header_size = 0;
  PrimaryInitScheme ();
}

/***********************************************************************/

/*
 * Secondary: RC
 */

static void *MMAllocateBytesRC (STMState_t *st, int bytes);
static void  MMDeallocateObjRC (STMState_t *st, void *ptr);
static void *MMGuardObjRC (STMState_t *st, int g, void *ptr);
static void  MMReleaseObjRC (STMState_t *st, int g, void *ptr);

static RCHeader_t *RCObjToHeader (void *ptr) 
{
  RCHeader_t *h;
  h = ((RCHeader_t *) ptr) - 1;
  return h;
}

static void *RCHeaderToObj (RCHeader_t *h)
{
  void *result;
  result = (void *) (h + 1);
  return result;
}

static void *MMAllocateBytesRC (STMState_t *st, int bytes) {
  void        *result;
  RCHeader_t  *rch;
  RCHeader_t **ptr;

  result = find_from_object_cache(st, bytes);
  MB();
  if (result == NULL)
    {
      rch = PrimaryAllocateBytes (st, bytes + sizeof (RCHeader_t));
      if (rch == NULL) {
	return NULL;
      }
#ifdef DEBUG
      memset (rch, 0, bytes + sizeof (RCHeader_t));
#endif

      rch -> rc = 2;
      result = RCHeaderToObj (rch);
      MB();
    }
  else 
    {
      int old_rc;

      rch = RCObjToHeader (result);
      old_rc = get_and_adjust ((int *) &(rch -> rc), 1); 
      stm_assert ((old_rc & 1) == 1);
    }

  MB();

  return result;
}

static void MMDeallocateObjRC (STMState_t *st, void *ptr)
{
  COUNT(st, COUNT_RC_UP);
  MMReleaseObjRC (st, 0, ptr);
}

static void *MMGuardObjRC (STMState_t *st, int g, void *ptr)
{
  RCHeader_t *h;
  h = RCObjToHeader (ptr);
  get_and_adjust ((int *) &(h -> rc), 2);
  MB();
  COUNT(st, COUNT_RC_UP);
  return NULL;
}

static void MMReleaseObjRC (STMState_t *st, int g, void *ptr)
{
  RCHeader_t *h;
  int old_rc;
  int new_rc;

  COUNT(st, COUNT_RC_DOWN);

  h = RCObjToHeader (ptr);
  do
    {
      MB();
      old_rc = h -> rc;
      new_rc = old_rc - 2;
      if (new_rc == 0) 
	{
	  new_rc = 1;
	}
    }
  while (CASInt (&(h -> rc), old_rc, new_rc) != old_rc);

  if (old_rc == 1) {
    stm_panic (("freeing twice"));
  }

  if (((old_rc - new_rc) & 1) == 1)
    {
      stm_assert (old_rc == 2);
      MB();
#ifdef KILL_ON_DEALLOCATE
      clientKillObject (ptr);
#endif
      add_to_object_cache (st, ptr);
    }
}

static void MMInitStateRC (STMState_t *st) {
  PrimaryInitState (st);
  init_object_cache (st);
}

static void MMReclaimStateRC (STMState_t *st) {
  drop_object_cache (st);
  PrimaryReclaimState (st);
}

static void MMDoneGCRC (STMState_t *st) {
  drop_object_cache (st);
  PrimaryDoneGC (st);
}

static void MMInitSchemeRC (void) {
  mm_header_size = sizeof (RCHeader_t);
  PrimaryInitScheme ();
}

/***********************************************************************/

/*
 * Secondary: PD
 */

static void *MMAllocateBytesPD (STMState_t *st, int bytes);
static void  MMDeallocateObjPD (STMState_t *st, void *ptr);
static void *MMGuardObjPD (STMState_t *st, int g, void *ptr);
static void  MMReleaseObjPD (STMState_t *st, int g, void *ptr);

static PDHeader_t *PDObjToHeader (void *ptr) 
{
  PDHeader_t *h;
  h = ((PDHeader_t *) ptr) - 1;
  return h;
}

static void *PDHeaderToObj (PDHeader_t *h)
{
  void *result;
  result = (void *) (h + 1);
  return result;
}

static void *MMAllocateBytesPD (STMState_t *st, int bytes) { 
  PDHeader_t *result;

  result = st -> secondary_state.pd.last_header;
  MB();
  if (result != NULL && result -> isPrivate == FALSE) {
    result = st -> secondary_state.pd.last_header = NULL;
  }

  if ((result == NULL) || 
      (clientObjToBytes (PDHeaderToObj (result)) != bytes))
    {
      result = PrimaryAllocateBytes (st, bytes + sizeof (PDHeader_t));
      if (result == NULL) {
	return NULL;
      }

      result -> isPrivate = TRUE;
    }
  
  return PDHeaderToObj (result);
}

static void MMDeallocateObjPD (STMState_t *st, void *ptr) {
  PDHeader_t *h;
  h = PDObjToHeader (ptr);
  st -> secondary_state.pd.last_header = h;
}

static void *MMGuardObjPD (STMState_t *st, int g, void *ptr) {
  PDHeader_t *h;
  h = PDObjToHeader (ptr);
  h -> isPrivate = FALSE;
  MB();
  return NULL;
}

static void MMReleaseObjPD (STMState_t *st, int g, void *ptr) {
  /* Nothing */
}

static void MMInitStatePD (STMState_t *st) {
  PrimaryInitState (st);
  st -> secondary_state.pd.last_header = NULL;
}

static void MMReclaimStatePD (STMState_t *st) {
  PrimaryReclaimState (st);
}

static void MMDoneGCPD (STMState_t *st) {
  st -> secondary_state.pd.last_header = NULL;
  PrimaryDoneGC (st);
}

static void MMInitSchemePD (void) {
  mm_header_size = sizeof (PDHeader_t);
  PrimaryInitScheme ();
}

/***********************************************************************/

/*
 * Secondary: SMR
 */

static void *MMAllocateBytesSMR (STMState_t *st, int bytes);
static void  MMDeallocateObjSMR (STMState_t *st, void *ptr);
static void *MMGuardObjSMR (STMState_t *st, int g, void *ptr);
static void  MMReleaseObjSMR (STMState_t *st, int g, void *ptr);

/*
 * Hazard pointers are to objs rather than to our headers: we assume that
 * guard / unguard will be common and so defer conversions to scan
 */
static void *hp[MAX_GUARDS];

static void SMRFreeObj (STMState_t *st, void *obj) 
{
#ifdef KILL_ON_DEALLOCATE
  clientKillObject (obj);
#endif
#ifdef DEBUG
  OCObjToHeader(obj) -> next = NULL;
#endif
  add_to_object_cache (st, obj);
}

static void SMRScan (STMState_t *st)
{
  OCHeader_t *plist[MAX_GUARDS];
  OCHeader_t *new_dlist;
  OCHeader_t *t;
  int new_dcount;
  int i, j;
  int p;

  NO_INLINE ();
  /* Stage 1 */
  p = 0;
  for (i = 0; i < MAX_GUARDS; i ++)
    {
      if (hp[i] != NULL) 
	{
	  t = OCObjToHeader (hp[i]);
	  plist[p++] = t;
	}
    }

  /* Stage 2 */

  /* 
   * For now we just use a linear search since there are only a small
   * number of pointers.
   */

  /* Stage 3 */
  t = st -> secondary_state.smr.dlist;
  new_dcount = 0;
  new_dlist = NULL;
  while (t != NULL) {
    OCHeader_t *next_t;

    next_t = t -> next;

    for (i = 0; i < p; i ++) {
      if (GUARD_PROTECTS (plist[i], t)) {
	t -> next = new_dlist;
	new_dlist = t;
	new_dcount ++;
	break;
      }
    }

    if (i == p)
      {
	SMRFreeObj (st, OCHeaderToObj (t));
      } 

    t = next_t;
  }

  /* Stage 4 */
  st -> secondary_state.smr.dlist = new_dlist;
  st -> secondary_state.smr.dcount = new_dcount;
}

static void *MMAllocateBytesSMR (STMState_t *st, int bytes) { 
  return allocate_via_object_cache (st, bytes);
}

static void MMDeallocateObjSMR (STMState_t *st, void *ptr) {
  OCHeader_t *h;
  OCHeader_t *old_dlist;

  h = OCObjToHeader (ptr);
  stm_assert (h -> next == NULL);
  old_dlist = st -> secondary_state.smr.dlist;
  h -> next = old_dlist;
  st -> secondary_state.smr.dlist = h;
  st -> secondary_state.smr.dcount ++;

  if (st -> secondary_state.smr.dcount >= DEFERRED_FREE_BATCH_SIZE) {
    SMRScan (st);
  }
}

static void *MMGuardObjSMR (STMState_t *st, int g, void *ptr) {
  void *result;
  int   idx;

  stm_assert (g >= 0 && g < GUARDS_PER_THREAD);
  idx = ((st -> idx) * GUARDS_PER_THREAD) + g;
  result = hp[idx];
  hp[idx] = ptr;
  MB ();

  return result;
}

static void MMReleaseObjSMR (STMState_t *st, int g, void *ptr) {
  int idx;

  stm_assert (g >= 0 && g < GUARDS_PER_THREAD);
  MB();
  idx = ((st -> idx) * GUARDS_PER_THREAD) + g;
  hp[idx] = NULL;
}

static void MMInitStateSMR (STMState_t *st) {
  int idx;
  int i;
  PrimaryInitState (st);
  init_object_cache (st);
  st -> secondary_state.smr.dcount = 0;
  st -> secondary_state.smr.dlist = NULL;
  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    hp[idx + i] = NULL;
  }
}

static void MMReclaimStateSMR (STMState_t *st) {
  int idx;
  int i;

  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    hp[idx + i] = NULL;
  }

  /* Should not lose any remaining items in the dlist */
  stm_panic (("MMReclaimStateSMR not implemented"));

  PrimaryReclaimState (st);
}

static void MMDoneGCSMR (STMState_t *st) {
  int idx;
  int i;

  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    hp[idx + i] = NULL;
  }

  st -> secondary_state.smr.dlist = NULL;
  st -> secondary_state.smr.dcount = 0;

  drop_object_cache (st);
  PrimaryDoneGC (st);
}

static void MMInitSchemeSMR (void) {
  int i;
  mm_header_size = sizeof (OCHeader_t);
  PrimaryInitScheme ();
  for (i = 0; i < MAX_GUARDS; i ++) {
    hp[i] = NULL;
  }
}

/***********************************************************************/

/*
 * Secondary: PTB
 */

static void *MMAllocateBytesPTB (STMState_t *st, int bytes);
static void  MMDeallocateObjPTB (STMState_t *st, void *ptr);
static void *MMGuardObjPTB (STMState_t *st, int g, void *ptr);
static void  MMReleaseObjPTB (STMState_t *st, int g, void *ptr);

#define PTB_PACK(_hh) ((((long long) ((_hh).h) ) << 32) + ((_hh).ver))

typedef struct PTBHandoff_t PTBHandoff_t;

struct PTBHandoff_t {
  OCHeader_t *h;
  int         ver;
};

static OCHeader_t   **post;
static PTBHandoff_t  *handoff;

static void PTBFreeObj (STMState_t *st, void *obj) {
#ifdef KILL_ON_DEALLOCATE
  clientKillObject (obj);
#endif
#ifdef DEBUG
  OCObjToHeader(obj) -> next = NULL;
#endif
  add_to_object_cache (st, obj);
}

static void *MMAllocateBytesPTB (STMState_t *st, int bytes) { 
  return allocate_via_object_cache (st, bytes);
}

static OCHeader_t *PTBSearchForEscapee (OCHeader_t **hp, OCHeader_t *guard) {
  OCHeader_t *result;
  OCHeader_t *escapee;
  
  result = NULL;
  escapee = *hp;
  while (escapee != NULL) {
    if (GUARD_PROTECTS (guard, escapee)) {
      stm_assert (result == NULL);
      result = escapee;
#ifndef DEBUG
      break;
#endif
    }
    escapee = escapee -> next;
  }

  return result;
}

static void PTBInsertPtr (OCHeader_t **hp, OCHeader_t *ptr) {
  stm_assert (ptr -> next == NULL);
  ptr -> next = *hp;
  *hp = ptr;
}

static void PTBDeletePtr (OCHeader_t **hp, OCHeader_t *ptr) {
  OCHeader_t *h;

  h = *hp;
  while (h) {
    if (ptr == h) {
      *hp = h -> next;
      h -> next = NULL;
      return;
    }
    hp = &(h -> next);
    h = *hp;
  }

  stm_panic (("%p not found in PTB\n", ptr));
}

static void PTBScan (STMState_t *st) {
  OCHeader_t *vs;
  int i;

  /*
   * Determine set to liberate
   */

  vs = st -> secondary_state.ptb.dlist;
  for (i = 0; i < MAX_GUARDS; i ++) {
    int            attempts = 0;
    PTBHandoff_t   ho;
    OCHeader_t    *v;
    OCHeader_t    *escapee;

    ho = handoff[i];
    RMB(); /* Ensure ho was in handoff when we read post */
    v = post[i];
    if (v != NULL && ((escapee = PTBSearchForEscapee (&vs, v)) != NULL)) {
      while (TRUE)
	{
	  PTBDeletePtr (&vs, escapee);
	  if (CASPtrInt ((int64_t *) &handoff[i],
			 (void *) ho.h, (void *) escapee, 
			 ho.ver, ho.ver + 1))
	    {
	      if (ho.h != NULL) PTBInsertPtr (&vs, ho.h);
	      break;
	    }
	  else /* CASPtrInt failed */
	    {
	      PTBInsertPtr (&vs, escapee);
	    }
	  attempts ++;
	  if (attempts == 3) break;
	  ho = handoff [i];
	  RMB ();
	  if (attempts == 2 && ho.h != NULL) break;
	  if (v != post[i]) break;
	}
    } else { /* Guard was stood down or not of our concern */
      if (ho.h != NULL &&
	  !GUARD_PROTECTS(v, ho.h))
	{
	  if (CASPtrInt ((int64_t *) &handoff[i],
			 ho.h, NULL,
			 ho.ver, ho.ver + 1)) 
	    {
	      PTBInsertPtr (&vs, ho.h);
	    }
	}
    }
  }

  /*
   * De-allocate all in vs
   */

  while (vs != NULL) {
    OCHeader_t *next_h;
    next_h = vs -> next;
    PTBFreeObj (st, OCHeaderToObj (vs));
    vs = next_h;
  }

  st -> secondary_state.ptb.dcount = 0;
  st -> secondary_state.ptb.dlist = NULL;
}

static void MMDeallocateObjPTB (STMState_t *st, void *ptr) {
  OCHeader_t *h;
  OCHeader_t *old_dlist;

  h = OCObjToHeader (ptr);
  old_dlist = st -> secondary_state.ptb.dlist;
  h -> next = old_dlist;
  st -> secondary_state.ptb.dlist = h;
  st -> secondary_state.ptb.dcount ++;

  if (st -> secondary_state.ptb.dcount >= DEFERRED_FREE_BATCH_SIZE) {
    PTBScan (st);
  }
}

static void *MMGuardObjPTB (STMState_t *st, int g, void *ptr) {
  OCHeader_t *h;
  OCHeader_t *old;
  int          idx;
  void        *result;

  stm_assert (g >= 0 && g < GUARDS_PER_THREAD);
  idx = ((st -> idx) * GUARDS_PER_THREAD) + g;
  
  h = OCObjToHeader (ptr);
  old = post[idx];
  post[idx] = h;
  MB ();

  result = (old == NULL) ? old : OCHeaderToObj (old);

  return result;
}

static void MMReleaseObjPTB (STMState_t *st, int g, void *ptr) {
  int idx;

  stm_assert (g >= 0 && g < GUARDS_PER_THREAD);
  idx = ((st -> idx) * GUARDS_PER_THREAD) + g;

  MB();
  post[idx] = NULL;
}

static void MMInitStatePTB (STMState_t *st) {

  if (sizeof (PTBHandoff_t) > 8) {
    stm_panic (("PTBHandoff_t too large for atomic update"));
  }

  PrimaryInitState (st);
  init_object_cache (st);
  st -> secondary_state.ptb.dlist = NULL;
  st -> secondary_state.ptb.dcount = 0;
}

static void MMReclaimStatePTB (STMState_t *st) {
  int idx;
  int i;
  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    post[idx + i] = NULL;
  }
  drop_object_cache (st);
  PrimaryReclaimState (st);
}

static void MMDoneGCPTB (STMState_t *st) {
  int idx;
  int i;

  idx = (st -> idx) * GUARDS_PER_THREAD;
  for (i = 0; i < GUARDS_PER_THREAD; i ++) {
    post[idx + i] = NULL;
    handoff[idx + i].h = NULL;
    handoff[idx + i].ver = 0;
  }

  st -> secondary_state.ptb.dlist = NULL;
  st -> secondary_state.ptb.dcount = 0;

  drop_object_cache (st);
  PrimaryDoneGC (st);
}

static void MMInitSchemePTB (void) {
  int i;
  post = mallocBytes (MAX_GUARDS * sizeof (OCHeader_t *));
  handoff = mallocBytes (MAX_GUARDS * sizeof (PTBHandoff_t));
  mm_header_size = sizeof (OCHeader_t);
  PrimaryInitScheme ();
  for (i = 0; i < MAX_GUARDS; i ++) {
    post[i] = NULL;
    handoff[i].h = NULL;
    handoff[i].ver = 0;
  }
}

/***********************************************************************/

/*
 * Management of descriptors
 */

static bool_t is_stm_descriptor (void *ptr)
{
  STMDescriptor_t *tmp;
  bool_t result;

  tmp = (STMDescriptor_t *) ptr;
  if (tmp -> status >= STM_ACTIVE && 
      tmp -> status < BEYOND_LAST_STATUS)
    {
      result = TRUE;
    }
  else
    {
      result = FALSE;
    }
  return result;

}

static int descriptor_bytes (void)
{
  int result;
  result = sizeof (STMDescriptor_t);
  return result;
}

static int clientObjToBytes (void *ptr)
{
  int result;
  result = is_stm_descriptor (ptr) ? sizeof (STMDescriptor_t) : sizeof (STMSleepers_t);
  return result;
}

#define DEAD_VALUE ((void *) 0xDEADCA55)

static void clientKillObject (void *ptr)
{
  if (is_stm_descriptor (ptr))
    {
      STMDescriptor_t *stmd;
      int i;
      
      stmd = (STMDescriptor_t *) ptr;
      for (i = 0; i < MAX_LOCATIONS; i ++) {
	stm_assert ((OREC_OF(stmd -> entries[i].addr) -> u.owner) != stmd);
	stmd -> entries[i].addr = DEAD_VALUE;
	stmd -> entries[i].new_val = (word_t) DEAD_VALUE;
	stmd -> entries[i].new_version = (int) DEAD_VALUE;
	stmd -> entries[i].old_val = (word_t) DEAD_VALUE;
	stmd -> entries[i].old_version = (int) DEAD_VALUE;
	stmd -> entries[i].waiting_with = DEAD_VALUE;
	stmd -> entries[i].does_own = (int) DEAD_VALUE;
      }
    }
  else
    {
      STMSleepers_t *stms;
      stms = (STMSleepers_t *) ptr;
      stms -> st = (STMState_t *) DEAD_VALUE;
      stms -> next = (STMSleepers_t *) DEAD_VALUE;
    }
}

/***********************************************************************/

static STMResult_t stm_set_status (STMDescriptor_t *stmd, 
				   STMResult_t expected_status,
				   STMResult_t desired_status)
{
  STMResult_t seen;

  stm_assert ((expected_status != STM_ABORTED) &&
	      (expected_status != STM_COMMITTED));

  MB ();

  seen = CASInt ((int volatile *) &(stmd -> status), 
		 expected_status, 
		 desired_status);

  MB();

  stm_assert (stmd -> status != STM_ACTIVE);

  return seen;
}

/* If we fail to add a registered sleeper then we must record that a
 * GC must take place.  Otherwise a thread can spin re-attempted a
 * transaction and never sleeping (because it can't) and never GCing
 * (because StartTransaction * always proceeds with a cached
 * desciptor.
 */

static volatile bool_t need_gc = FALSE;

static bool_t add_to_registered_sleepers (STMState_t *st,
					  STMDescriptor_t *master)
{
  STMSleepers_t *stms; 
  bool_t result;

  stms = MMAllocateBytes (st, sizeof (STMSleepers_t));
  if (stms != NULL) 
    {
      stms -> st = st;
      stms -> tn = st -> tn;
      stms -> next = master -> sleepers;
      master -> sleepers = stms;
      result = FALSE;
    } 
  else 
    {
      need_gc = TRUE;
      result = TRUE;
    }
  return result;
}

static void wake_registered_sleepers (STMState_t *st,
				      STMDescriptor_t *master)
{
  STMSleepers_t *stms;
  STMSleepers_t *next_stms;
  int i = 0;

  stms = master -> sleepers;
  while (stms != NULL)
    {
      STMDescriptor_t *other;
      other = stms -> st -> current_transaction;
      i ++;
      stm_assert (i < MAX_THREADS);
      if (other != NULL)
	{
	  void *og;
	  og = MMGuardObj (st, 0, other);
	  stm_assert (og == NULL);
	  if (stms -> st -> current_transaction == other && 
	      stms -> st -> tn == stms -> tn)
	    {
	      STMMutexLock (&other -> mutex);
	      stm_assert (st -> current_transaction == master);
	      STMCondSignal (&other -> cond);
	      stm_set_status (other, STM_WAITING_2, STM_ABORTED);
	      STMMutexUnlock (&other -> mutex);
	    }
	  MMReleaseObj (st, 0, other);
	}
      next_stms = stms -> next;
      MMDeallocateObj (st, stms);
      stms = next_stms;
    }
  master -> sleepers = NULL;
}

/* 
 * Wake stmd if it is in deep sleep.  It is responsible for waking
 * its registered waiters.
 */

static void wake_sleeper (STMState_t *st, STMDescriptor_t *stmd)
{
  int seen;

  STMMutexLock (&stmd -> mutex);

  seen = stm_set_status (stmd,
			 STM_WAITING_2,
			 STM_ABORTED);

  if (seen == STM_WAITING_2) STMCondSignal (&stmd -> cond);

  STMMutexUnlock (&stmd -> mutex);
}

/*......................................................................*/

static void release_ownership (STMDescriptor_t *stmd)
{
  int i;
  int n;
  bool_t committed;

  n = stmd -> length;
  committed = ((stmd -> status) == STM_COMMITTED);

  for (i = 0; i < n ; i++)
    {
      STMEntry_t        *entry = &(stmd -> entries [i]);
      OwnershipRecord_t *orec = OREC_OF (entry -> addr);
      STMDescriptor_t   *new_owner;
      
      if (committed)
	{
	  new_owner = (void *) (stmd -> entries[i].new_version);
	} 
      else 
	{
	  new_owner = (void *) (stmd -> entries[i].old_version);
	}

      CASPtr ((addr_t) &(orec -> u.owner), stmd, new_owner);
    }
}

static bool_t can_deallocate (STMDescriptor_t *stmd)
{
  bool_t result;
  int i, n;

  result = TRUE;
  n = stmd -> length;
  for (i = 0; i < n; i ++)
    {
      STMEntry_t *t = &(stmd -> entries[i]);
      if (t -> does_own &&
	  OREC_OF(t -> addr) -> u.owner == stmd)
	{
	  result = FALSE;
	  break;
	}
    }

  if (result == TRUE) {
    result = CASInt ((int volatile *) &(stmd -> cd), TRUE, FALSE);
  }

  return result;
}

static void release_ownership_nb (STMState_t *st, STMDescriptor_t *stmd)
{
  int i;
  int n;
  bool_t succeeded;

  n = stmd -> length;
  succeeded = ((stmd -> status) == STM_COMMITTED);
  for (i = 0; i < n ; i++)
    {
      STMEntry_t        *my_entry = &(stmd -> entries [i]);
      OwnershipRecord_t *orec = OREC_OF (my_entry -> addr);

      if (my_entry -> does_own)
	{
	  STMEntry_t *update_entry;
	  STMDescriptor_t *exp_owner;
	  STMDescriptor_t *new_owner;
	  int exp_count, new_count;
	  word_t update_val;
	  
	  exp_owner = NULL;
	  do
	    {
	      update_entry = NULL;

	    retry_release:
	      if (exp_owner != NULL && exp_owner != stmd) {
		MMReleaseObj (st, 0, exp_owner);
	      }


	      MB();
	      exp_owner = orec -> u.owner;
	      if (exp_owner == stmd) {
		update_entry = my_entry;
	      } else {
		int update_n;
		int update_i;
		void *og;

		og = MMGuardObj (st, 0, exp_owner);
		stm_assert (og == NULL);
		if (orec -> u.owner != exp_owner) {
		  goto retry_release;
		}

		update_n = exp_owner -> length;
		for (update_i = 0; update_i < update_n; update_i ++)
		  {
		    STMEntry_t *t;
		    t = &(exp_owner -> entries[update_i]);
		    MB();
		    if (t -> addr == my_entry -> addr) {
		      update_entry = t;
		      break;
		    }
		  }
		stm_set_status (exp_owner, STM_ACTIVE, STM_ABORTED);
	      }
	      
	      stm_assert (update_entry != NULL);
	      stm_assert (exp_owner -> status != STM_ACTIVE);

	      if (exp_owner -> status == STM_COMMITTED) {
		update_val = update_entry -> new_val;
	      } else {
		update_val = update_entry -> old_val;
	      }

	      MB();
	      *(update_entry -> addr) = (void *) update_val;
	      GC_WRITE_BARRIER (update_entry -> addr, update_val);
	      MB();

	      exp_count = orec -> updaters;
	      new_count = exp_count - 1;
	      if (new_count == 0)
		{
		  if (exp_owner -> status == STM_COMMITTED)
		    {
		      new_owner = (void *) (update_entry -> new_version);
		    }
		  else
		    {
		      new_owner = (void *) (update_entry -> old_version);
		    }
		}
	      else 
		{
		  new_owner = exp_owner;
		}
	    }
	  while (!CASPtrInt ((long long *) &(orec -> u.owner), 
			     exp_owner, new_owner,
			     exp_count, new_count));

#ifdef DEBUG
	  if ((((int)new_owner) & 1) == 0)
	    {
	      int ikk = 0;
	      for (ikk = 0; ikk < new_owner -> length; ikk++) {
		stm_assert (new_owner -> entries[ikk].addr != DEAD_VALUE);
	      }
	    }
#endif

	  if (exp_owner != stmd)
	    {
	      if ((new_count == 0) &&
		  (can_deallocate (exp_owner)))
		{
		  MMReleaseObj (st, 0, exp_owner);
		  MMDeallocateObj (st, exp_owner);
		}
	      else
		{
		  MMReleaseObj (st, 0, exp_owner);
		}
	    }
	}
    }
}

/*......................................................................*/

static bool_t acquire_ownership (STM_ENV *env,
				 STMState_t *st,
				 STMDescriptor_t *stmd, 
				 bool_t stmd_to_wait)
{
  int i, n;
  bool_t result;
  void *to_release;

  result = TRUE;
  n = stmd -> length;

  for (i = 0; i < n ; i++)
    {
      OwnershipRecord_t *orec;
      STMEntry_t        *entry;
      int                saw;
      STMDescriptor_t *exp_owner, *new_owner;
      union STMVersionInfo_t vi;
      void *to_release;

      entry = &(stmd -> entries[i]);
      orec = OREC_OF (entry -> addr);
      MB();

      do
	{
retry:
	  vi = orec -> u;

	  exp_owner = vi.owner;
	  new_owner = stmd;

	  if (VI_IS_VERSION (vi) && vi.version != entry -> old_version)
	    {
	      /* This is not the version we are looking for */
	      result = FALSE;
	      goto done_acquire;
	    }

	  if ((!VI_IS_VERSION(vi)) && exp_owner != stmd)
	    {
	      /* This orec is already owned */
	      STMResult_t owners_status;
	      void *og;

	      MB();
	      og = MMGuardObj (st, i + 2, exp_owner);
	      stm_assert (og == NULL);
	      if (orec -> u.owner != exp_owner) {
		MMReleaseObj (st, i+2, exp_owner);
		goto retry;
	      }

	      owners_status = exp_owner -> status;

	      /* 
	       * Try to wake it if it is in light sleep -- either 
	       * we need to own the location (so must get it to abort)
	       * or we want to become a registered sleeper with it
	       * (so we want it in STM_WAITING_2).
	       */

	      if (owners_status == STM_WAITING_1)
		{
		  stm_set_status (exp_owner, STM_WAITING_1, STM_ABORTED);
		  MB();
		  owners_status = exp_owner -> status;
		}
	      
	      /*
	       * If it is in a deep sleep then discover the current
	       * version of the location it's sleeping on.  Then
	       * either:
	       *
	       *  (a) we want to sleep waiting for the same value to
	       *      change: register ourselves with the current owner,
	       *
	       *  (b) we want to update from that value: wake the sleeper
	       *      and try to take ownership of the location,
	       *
	       *  (c) we want to sleep or update from some other value:
	       *      we abort and leave the sleeper.	      
	       */

	      if (owners_status == STM_WAITING_2)
		{
		  int current_version;
		  int j;

		  current_version = -1;
		  for (j = 0; j < exp_owner -> length; j ++) {
		    if (OREC_OF(exp_owner -> entries[j].addr) == orec) {
		      current_version = exp_owner -> entries[j].old_version;
		      break; /* From searching for current_version */
		    }
		  }
		  stm_assert (current_version != -1);

		  if (stmd_to_wait) 
		    {
		      if (entry -> old_version == current_version)
			{
			  /* (a) We join sleeper */
			  entry -> waiting_with = exp_owner;
			  break; /* Go to next entry */
			}
		      else
			{
			  /* (c) Validation failed */
			  result = FALSE;
			  MMReleaseObj (st, i + 2, exp_owner);
			  goto done_acquire;
			}
		    }
		  else
		    {
		      if (entry -> old_version == current_version)
			{
			  /* (b) Wake the sleeper */
			  wake_sleeper (st, exp_owner);
			  owners_status = exp_owner -> status;
			}
		      else
			{
			  /* (c) Validation failed */
			  result = FALSE;
			  MMReleaseObj (st, i + 2, exp_owner);
			  goto done_acquire;
			}
		    }
		}
	      
	      /*
	       * If the current owner has aborted then we can try 
	       * to release their ownership of these locations.
	       */

	      if (owners_status == STM_ABORTED)
		{
                  release_ownership (exp_owner);
		  MMReleaseObj (st, i+2, exp_owner);
		  goto retry; /* Retry this entry */
		}

	      /*
	       * The current owner is active or committed.  Decide 
	       * how to proceed on the basis of our configuration.
	       *
	       *  OPT_ABORT_SELF => we abort and hope that the current
	       *       owner will be out of the way if we retry.
	       */

	      if (OPT_ABORT_SELF) {
		result = FALSE;
		MMReleaseObj (st, i + 2, exp_owner);
		goto done_acquire;
	      } 

	      /*
	       *  OPT_DEFER_OTHER => we spin trying to gain ownership
	       *       of this location and hope that the current 
	       *       owner will be done soon.
	       */

	      stm_assert (OPT_DEFER_OTHER);
	      MMReleaseObj (st, i+2, exp_owner);
	      goto retry;
	    }
	}
      while (CASPtr ((addr_t) &(orec -> u.owner), 
		     exp_owner, 
		     new_owner) != exp_owner);

      stm_assert (stmd_to_wait || 
		  (exp_owner == stmd) ||
		  ((((word_t) exp_owner) & 1) == 1));

      stm_assert (stmd_to_wait || 
		  (exp_owner == stmd) || 
		  ((word_t) exp_owner) == entry -> old_version);

    }

 done_acquire:
  return result;
}



static bool_t merge_into_descriptor (STMDescriptor_t *stmd,
				     STMDescriptor_t *from,
				     OwnershipRecord_t *orec)
{
  bool_t succeeded;
  bool_t result;
  int i;
  int n;
  int to_i;
  int to_n;
  int proto_length;
  bool_t stmd_to_update;

  /* 
   * Is "stmd" proposing to make any updates?  If so then entries we
   * acquire through merging must take a new version number in order to
   * ensure consistency with any existing updates that use the same
   * ownership record.
   */

  proto_length = stmd -> length;
  stmd_to_update = FALSE;
  for (i = 0; i < proto_length; i ++)
    {
      if ((OREC_OF(stmd -> entries[i].addr) == orec) &&
	  (stmd -> entries[i].new_version != stmd -> entries[i].old_version))
	{
	  stmd_to_update = TRUE;
	  break;
	}
    }


  result = TRUE;
  succeeded = ((from -> status) == STM_COMMITTED);
  stm_assert (succeeded || from -> status == STM_ABORTED);

  to_n = stmd -> length;
  n = from -> length;
  for (i = 0; i < n; i ++)
    {
      STMEntry_t *from_entry;
      STMEntry_t *to_entry;
      int exp_version;

      from_entry = &(from -> entries[i]);
      exp_version = ((succeeded) ? 
		     from_entry -> new_version : 
		     from_entry -> old_version);


      if (OREC_OF (from_entry -> addr) == orec)
	{
	  /* from_entry is owned by the orec we're interested in */
	  for (to_i = 0; to_i < to_n; to_i++)
	    {
	      to_entry = &(stmd -> entries[to_i]);

	      if (OREC_OF (to_entry -> addr) == orec)
		{
		  /* to_entry is owned by the same orec, check consistency */
		  if (to_entry -> old_version != exp_version)
		    {
		      result = FALSE;
		      goto done_merge_into_descriptor;
		    }
		  else
		    {
		      if (to_entry -> addr == from_entry -> addr)
			{
			  break; /* from searching for to_i */
			}
		    }
		}
	    }

	  /* If we did not find a match then we need a new entry */
	  if (to_i == to_n)
	    {
	      if (proto_length == MAX_LOCATIONS)
		{
		  stm_panic (("Out of all %d locations\n", MAX_LOCATIONS));
		}
	      to_entry = &(stmd -> entries[proto_length]);
	      to_entry -> addr = from_entry -> addr;
	      if (succeeded)
		{
		  to_entry -> old_val = from_entry -> new_val;
		  to_entry -> old_version = from_entry -> new_version;
		}
	      else
		{
		  to_entry -> old_val = from_entry -> old_val;
		  to_entry -> old_version = from_entry -> old_version;
		}    

	      to_entry -> new_val = to_entry -> old_val;
	      if (stmd_to_update) {
		to_entry -> new_version = to_entry -> old_version + 2;
	      } else {
		to_entry -> new_version = to_entry -> old_version;
	      }

	      to_entry -> waiting_with = NULL;
	      to_entry -> does_own = FALSE;

	      proto_length ++;
	    }
	}
    }

done_merge_into_descriptor:
  MB();
  stmd -> length = proto_length;

  return result;
}


static bool_t acquire_ownership_nb (STM_ENV *env,
				    STMState_t *st,
				    STMDescriptor_t *stmd, 
				    bool_t stmd_to_wait)
{
  int i, n;
  bool_t result;
  void *to_release;

  result = TRUE;
  n = stmd -> length;

  i = 0;
  while (i < n)
    {
      OwnershipRecord_t *orec;
      STMEntry_t        *entry;
      int                saw;
      STMDescriptor_t *exp_owner, *new_owner;
      int exp_count, new_count;
      union STMVersionInfo_t vi;
      void *to_release;

      entry = &(stmd -> entries[i]);
      orec = OREC_OF (entry -> addr);

      do
	{
retry:
	  vi = orec -> u;

	  exp_owner = vi.owner;
	  exp_count = orec -> updaters;

	  new_owner = stmd;
	  new_count = exp_count + 1;

	  if (VI_IS_VERSION (vi) && vi.version != entry -> old_version)
	    {
	      /* This is not the version we are looking for */
	      result = FALSE;
	      goto done_acquire;
	    }

	  if ((!VI_IS_VERSION(vi)) && exp_owner != stmd)
	    {
	      /* This orec is already owned */
	      STMResult_t owners_status;
	      void *og;

	      og = MMGuardObj (st, i + 2, exp_owner);
	      stm_assert (og == NULL);
	      if (orec -> u.owner != exp_owner) {
		MMReleaseObj (st, i+2, exp_owner);
		goto retry;
	      }

	      owners_status = exp_owner -> status;

	      stm_set_status (exp_owner, STM_ACTIVE, STM_ABORTED);

	      stm_assert (exp_owner -> status != STM_ACTIVE);
	      
	      /* 
	       * Try to wake it if it is in light sleep -- either 
	       * we need to own the location (so must get it to abort)
	       * or we want to become a registered sleeper with it
	       * (so we want it in STM_WAITING_2).
	       */

	    retry_waking_sleeper:
	      MB();
	      owners_status = exp_owner -> status;
	      if (owners_status == STM_WAITING_1)
		{
		  stm_set_status (exp_owner, 
				  STM_WAITING_1, 
				  STM_ABORTED);
		  owners_status = exp_owner -> status;
		}

	      stm_assert (exp_owner -> status != STM_ACTIVE);
	      stm_assert (exp_owner -> status != STM_WAITING_1);
	      
	      /*
	       * If it is in a deep sleep then discover the current
	       * version of the location it's sleeping on.  Then
	       * either:
	       *
	       *  (a) we want to sleep waiting for the same value to
	       *      change: register ourselves with the current owner,
	       *
	       *  (b) we want to update from that value: wake the sleeper
	       *      and try to take ownership of the location,
	       *
	       *  (c) we want to sleep or update from some other value:
	       *      we abort and leave the sleeper.	      
	       */

	      if (owners_status == STM_WAITING_2)
		{
		  int current_version;
		  int j;

		  current_version = -1;
		  for (j = 0; j < exp_owner -> length; j ++) {
		    if (OREC_OF(exp_owner -> entries[j].addr) == orec) {
		      current_version = exp_owner -> entries[j].old_version;
		      break; /* From searching for current_version */
		    }
		  }
		  stm_assert (current_version != -1);

		  if (stmd_to_wait) 
		    {
		      if (entry -> old_version == current_version)
			{
			  /* (a) We join sleeper */
			  entry -> waiting_with = exp_owner;
			  break; /* Go to next entry */
			}
		      else
			{
			  /* (c) Validation failed */
			  result = FALSE;
			  MMReleaseObj (st, i + 2, exp_owner);
			  goto done_acquire;
			}
		    }
		  else
		    {
		      if (entry -> old_version == current_version)
			{
			  /* (b) Wake the sleeper */
			  wake_sleeper (st, exp_owner);
			  owners_status = exp_owner -> status;
			}
		      else
			{
			  /* (c) Validation failed */
			  result = FALSE;
			  MMReleaseObj (st, i + 2, exp_owner);
			  goto done_acquire;
			}
		    }
		}
	      
	      MB();
	      owners_status = exp_owner -> status;
	      stm_assert (exp_owner -> status != STM_ACTIVE);
	      stm_assert (exp_owner -> status != STM_WAITING_1);
	      stm_assert (exp_owner -> status != STM_WAITING_2);

	      if (owners_status != STM_ABORTED &&
		  owners_status != STM_COMMITTED)
		{
		  stm_panic (("URK!  Saw status %d\n", owners_status));
		}

	      if (stmd_to_wait) 
		{
		  /* 
		   * If we're going to block anyway then wait for the
		   * current owner to get out of the way.  
		   *
		   * XXX Not sure what the 'right' semantics here are
		   * but this approach avoids anyone sleeping with
		   * updates merged into their descriptor -- i.e. 
		   * STM_WAITING_ people are always read-only.
		   */

		  while (orec -> u.owner == exp_owner) {
		    MB();
		  }

		  MMReleaseObj (st, i+2, exp_owner);
		  goto retry;
		}

	      /*
	       * Merge previous owner's view into ours.
	       */
	      MB();
	      if (!merge_into_descriptor (stmd, exp_owner, orec)) {
		/* The one we're merging into will doom us to fail. */
		result = FALSE;
		MMReleaseObj (st, i + 2, exp_owner);
		goto done_acquire;
	      }

	      n = stmd -> length;

	      if (CASPtrInt ((long long *) &(orec -> u.owner), 
			      exp_owner, new_owner,
			      exp_count, new_count))
		{
#ifdef DEBUG
		  stm_assert (new_owner -> entries[0].addr != DEAD_VALUE);
		  stm_assert (new_owner -> entries[1].addr != DEAD_VALUE);
#endif
		  MB();
		  entry -> does_own = TRUE;
		  MMReleaseObj (st, i+2, exp_owner);
		  break; /* from the while loop */
		}
	      else
		{
		  result = FALSE;
		  MMReleaseObj (st, i + 2, exp_owner);
		  goto done_acquire;
		}
	    }
	  else
	    {
	      if (CASPtrInt ((long long *) &(orec -> u.owner), 
			     exp_owner, new_owner,
			     exp_count, new_count))
		{
		  MB();
		  entry -> does_own = TRUE;
		  break; /* from the while loop */
		}
	    }
	}
      while (TRUE);
      i ++;
    }

 done_acquire:
  return result;
}

static void sort_descriptor (STMDescriptor_t *stmd)
{
  int i, j, n;

  n = stmd -> length;
  for (i = 0; i < n - 1; i ++)
    {
      for (j = i + 1; j < n; j ++)
	{
	  if (OREC_OF(stmd -> entries[i].addr) > OREC_OF(stmd -> entries[j].addr))
	    {
	      STMEntry_t temp = stmd -> entries[j];
	      stmd -> entries[j] = stmd -> entries[i];
	      stmd -> entries[i] = temp;
	    }
	}
    }

}

static void check_consistency (STMDescriptor_t *stmd)
{
  int i,n;
  n = stmd -> length;
  for (i = 0; i< n-1; i ++)
    {
      if (OREC_OF(stmd->entries[i].addr) == OREC_OF(stmd -> entries[i+1].addr))
	{
	  stm_assert ((stmd -> entries[i].old_version) ==
		      (stmd -> entries[i+=1].old_version));
	  stm_assert ((stmd -> entries[i].new_version) == 
		      (stmd -> entries[i+1].new_version));
	}
    }
}


static void check_no_writes (STMDescriptor_t *stmd)
{
  int i, n;
  
  n = stmd -> length;
  for (i = 0; i < n ; i++)
    {
      STMEntry_t *entry;
      entry = &(stmd -> entries[i]);
      if (entry -> new_val != entry -> old_val) 
	{
	  stm_panic (("STMWait has updates"));
	}
    }
}


/*......................................................................*/

/* 
 * A transaction calls stm_sleep_lightly when it has 
 * entered STM_WAITING_1.  This provides it with an opportunity
 * to spin brielfy before blocking in a 'heavyweight' sleep.
 * The thread will be woken by having its status updated to
 * STM_ABORTED.
 */

static void stm_sleep_lightly (STMState_t *st, STMDescriptor_t *stmd)
{
  /* Nothing */
}

/*
 * A transaction calls stm_sleep_heavily when it has entered
 * STM_WAITING_2.  The thread is to sleep by blocking on its condition
 * varaible and will be woken by that cv being notifid.
 *
 * Before it can sleep it must register itself with any transactions
 * that own locations that it is interested in.  Once woken, it must
 * wake any other transactions registered with it.
 *
 * This method is called (and returns) holding stmd -> mutex.  
 */

static void stm_sleep_heavily (STM_ENV *env, STMState_t *st, STMDescriptor_t *stmd)
{
  bool_t woken_before_sleeping;
  int i;
  int n;

  woken_before_sleeping = FALSE;

  /* 
   * Acquire mutexes on all transactions to register with.  Ensure
   * that they are in STM_WAITING_2.
   */
  
  n = stmd -> length;
  for (i = 0; i < n; i ++)
    {
      STMDescriptor_t *other;
      other = (stmd -> entries[i].waiting_with);
      if (other != NULL)
	{
	  STMMutexLock (&other -> mutex);
	  if (other -> status != STM_WAITING_2)
	    {
	      woken_before_sleeping = TRUE;
	    }
	}
    }
			
  if (!woken_before_sleeping)
    {
      /*
       * We now hold all the locks: register with anyone owning
       * a location that we're interested in.
       */

      for (i = 0; i < n; i ++)
	{
	  STMDescriptor_t *other;
	  other = (stmd -> entries[i].waiting_with);
	  if (other != NULL) {
	    woken_before_sleeping |= add_to_registered_sleepers (st, other);
	  }
	}
    }

  /*
   * Release all the locks except for our own.  Anyone trying to wake
   * us will now block waiting to acquire that mutex.
   */

  for (i = 0; i < n; i ++)
    {
      STMDescriptor_t *other;
      other = (stmd -> entries[i].waiting_with);
      if (other != NULL) {
	STMMutexUnlock (&other -> mutex);
      }
    }

  if (woken_before_sleeping)
    {
      /*
       * One of the other transaction we were supposed to wait for
       * got woken up before we registered with it.  We abort
       * and retry.
       */
      
      stm_set_status (stmd, STM_WAITING_2, STM_ABORTED);
    }
  else
    {
      /*
       * Now we block
       */

      stm_assert (st -> current_transaction == stmd);
      STMCondInit (&(stmd -> cond));
      STMCondWait (&stmd -> cond, &stmd -> mutex);
      wake_registered_sleepers (st, stmd);
    }

  /*  stm_assert ((stmd == NULL) || (stmd -> status == STM_ABORTED));*/
}

/*......................................................................*/

void STMVolatileWriteValue (STM_ENV *env, addr_t addr, word_t val)
{
  bool_t did_start;

  do
    {
      did_start = STMStartTransaction (env);
      if (!did_start) stm_panic (("Transaction didn't start\n"));
      STMWriteValue (env, addr, val);
    }
  while (!STMCommitTransaction (env));
}

/*......................................................................*/

word_t STMVolatileReadValue (STM_ENV *env, addr_t addr)
{
  word_t result;
  bool_t did_start;

  do
    {
      did_start = STMStartTransaction (env);
      if (!did_start) stm_panic (("Transaction didn't start\n"));
      result = STMReadValue (env, addr);
    }
  while (!STMCommitTransaction (env));

  return result;	 
}

/*......................................................................*/

int STMStartTransaction (STM_ENV *env) 
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  bool_t           result;

  st = GET_ST (env);
  stm_assert (st -> current_transaction == NULL);

  START_TIMING (TIMING_ALLOCATION);
  st -> tn ++;
  if (OPT_GC_HEAP_AVAILABLE && need_gc) {
    stmd = NULL;
    need_gc = FALSE;
    fprintf (stderr, "forcing gc\n");
  } else {
    stmd = MMAllocateBytes (st, descriptor_bytes ());
  }
  END_TIMING (st, TIMING_ALLOCATION);

  if (stmd != NULL)
    {
      result = TRUE;
      stmd -> status = STM_ACTIVE;
      stmd -> length = 0;
      stmd -> sleepers = NULL;
      stmd -> cd = TRUE;
      st -> current_transaction = stmd;
    }
  else
    {
      result = FALSE;
    }

  return result;
}

/*......................................................................*/

void STMWaitB (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  int              i;
  int              n;
  bool_t           waiting;

  MB();

  st = GET_ST (env);
  START_TIMING (TIMING_WAIT);
  stmd = st -> current_transaction;

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      /* Nothing: should retry immediately */
    }
  else
    {
      if (stmd -> status == STM_ACTIVE)
	{
	  n = stmd -> length;
	  
	  /* Sort entries to avoid deadlock */
	  sort_descriptor (stmd);
	  
	  /* Check no updates */
	  check_no_writes (stmd);
	  
	  /* Acquire ownership */
	  if (acquire_ownership (env, st, stmd, TRUE)) {
	    stm_set_status (stmd, STM_ACTIVE, STM_WAITING_1);
	  } else {
	    stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
	  }
	  
	  /* Wait if we need to */
	  if (stmd -> status == STM_WAITING_1)
	    {
	      stm_sleep_lightly (st, stmd);
	      
	      if (stmd -> status == STM_WAITING_1)
		{
		  STMMutexInit (&(stmd -> mutex));
		  STMCondInit (&(stmd -> cond));
		  STMMutexLock (&stmd -> mutex);
		  stm_set_status (stmd, STM_WAITING_1, STM_WAITING_2);
		  if (stmd -> status == STM_WAITING_2)
		    {
		      TEMP_EXIT_TIMING (TIMING_WAIT);
		      {
			START_TIMING (TIMING_ASLEEP);
			stm_sleep_heavily (env, st, stmd);
			END_TIMING (st, TIMING_ASLEEP);
		      }
		      TEMP_ENTER_TIMING (TIMING_WAIT);
		    }
		  STMMutexUnlock (&stmd -> mutex);
		  STMMutexDestroy (&stmd -> mutex);
		  STMCondDestroy (&stmd -> cond);
		}
	    }
	  
	  /* Release ownership and continue */
	  for (i = 0; i < stmd -> length ; i ++) 
	    {
	      STMEntry_t *entry = &(stmd -> entries[i]);
	      STMDescriptor_t *waiting_with = entry -> waiting_with;
	      if (waiting_with != NULL) {
		MMReleaseObj (st, i + 2, waiting_with);
	      }
	    }
	  release_ownership (stmd);
	}
    }

  END_TIMING (st, TIMING_WAIT);

  if (st -> current_transaction != NULL)
    {
       MMDeallocateObj (st, stmd);
    }

  st -> current_transaction = NULL;
}


void STMWaitNB (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  int              i;
  int              n;
  bool_t           waiting;

  MB();

  st = GET_ST (env);
  START_TIMING (TIMING_WAIT);
  stmd = st -> current_transaction;

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      /* Nothing: should retry immediately */
    }
  else
    {
      if (stmd -> status == STM_ACTIVE)
	{
	  void *og;
	  og = MMGuardObj (st, 1, stmd);
	  stm_assert (og == NULL);
	  stm_assert (stmd -> length > 0);
	  n = stmd -> length;
	  
	  /* Sort entries to avoid deadlock */
	  sort_descriptor (stmd);
	  
	  /* Check no updates */
	  check_no_writes (stmd);
	  
	  /* Acquire ownership */
	  if (acquire_ownership_nb (env, st, stmd, TRUE)) {
	    stm_set_status (stmd, STM_ACTIVE, STM_WAITING_1);
	  } else {
	    stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
	  }
	  
	  check_no_writes (stmd);

	  /* Wait if we need to */
	  if (stmd -> status == STM_WAITING_1)
	    {
	      stm_sleep_lightly (st, stmd);
	      
	      if (stmd -> status == STM_WAITING_1)
		{
		  STMMutexInit (&(stmd -> mutex));
		  STMCondInit (&(stmd -> cond));
		  STMMutexLock (&stmd -> mutex);
		  stm_set_status (stmd, STM_WAITING_1, STM_WAITING_2);
		  if (stmd -> status == STM_WAITING_2)
		    {
		      TEMP_EXIT_TIMING (TIMING_WAIT);
		      stm_sleep_heavily (env, st, stmd);
		      TEMP_ENTER_TIMING (TIMING_WAIT);
		    }
		  STMMutexUnlock (&stmd -> mutex);
		}
	    }

	  stm_assert (stmd -> status == STM_ABORTED);

	  /* Release ownership and continue */
	  for (i = 0; i < stmd -> length ; i ++) 
	    {
	      STMEntry_t *entry = &(stmd -> entries[i]);
	      STMDescriptor_t *waiting_with = entry -> waiting_with;
	      if (waiting_with != NULL) {
		MMReleaseObj (st, i + 2, waiting_with);
	      }
	    }

	  release_ownership_nb (st, stmd);

	  if (can_deallocate (stmd)) {
	    MMReleaseObj (st, 1, stmd);
	    stm_assert (stmd -> cd == FALSE);
	    MMDeallocateObj (st, stmd);
	  } else {
	    MMReleaseObj (st, 1, stmd);
	  }
	}
      else
	{
	  MMDeallocateObj (st, stmd);
	}
    }

  END_TIMING (st, TIMING_WAIT);

  st -> current_transaction = NULL;
}

/*......................................................................*/

static int STMCommitTransactionB (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  bool_t           succeeded;

  MB();

  st = GET_ST (env);
  stmd = st -> current_transaction;
  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      return FALSE;
    }
  else
    {
      if (OPT_DEBUG_MSG) fprintf (stderr, "%p trying to commit", stmd);
      
      /*
       * Check whether we have already aborted e.g. during a transactional
       * read or transactional write operaion.
       */
      
      succeeded = FALSE;
      START_TIMING (TIMING_COMMIT);
      if (stmd -> status == STM_ACTIVE)
	{
	  int i, n;
	  n = stmd -> length;
	  
#ifdef DO_STM_DEFER_OTHER
	  /* Sort entries to avoid deadlock */
	  sort_descriptor (stmd);
	  MB();
#endif
	  
	  /* take owmership of each of the locations involved */
	  if (acquire_ownership (env, st, stmd, FALSE)) {
	    stm_set_status (stmd, STM_ACTIVE, STM_COMMITTED);
	  } else {
	    stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
	  }
	  
	  /* make updates if necessary */
	  succeeded = ((stmd -> status) == STM_COMMITTED);
	  if (succeeded) 
	    {
	      MB();
	      for (i = 0; i < n ; i++)
		{
		  STMEntry_t        *entry;
		  entry = &(stmd -> entries[i]);
		  stm_assert (OREC_OF (entry -> addr) -> u.owner == stmd);
		  *(entry -> addr) = (void *) (entry -> new_val);
		  GC_WRITE_BARRIER (entry -> addr, entry -> new_val);
		  stm_assert (OREC_OF (entry -> addr) -> u.owner == stmd);
		}
	      MB();
	    }
	  
	  /* Release ownership and continue */
	  release_ownership (stmd);
	}
      
      MB();
      st -> current_transaction = NULL;
      MMDeallocateObj (st, stmd);
      END_TIMING (st, TIMING_COMMIT);
    }

  return succeeded;
}

static int STMCommitTransactionNB (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  bool_t           succeeded;
  void *og;

  MB();

  st = GET_ST (env);
  stmd = st -> current_transaction;
  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      return FALSE;
    }
  else
    {
      if (OPT_DEBUG_MSG) fprintf (stderr, "%p trying to commit", stmd);
      
      /*
       * Check whether we have already aborted e.g. during a transactional
       * read or transactional write operaion.
       */

      og = MMGuardObj (st, 1, stmd);
      stm_assert (og == NULL);
      stm_assert (stmd -> length > 0);
      succeeded = FALSE;
      START_TIMING (TIMING_COMMIT);
      if (stmd -> status == STM_ACTIVE)
	{
	  int i, n;
	  n = stmd -> length;

	  /* take owmership of each of the locations involved */
	  if (acquire_ownership_nb (env, st, stmd, FALSE)) {
	    stm_set_status (stmd, STM_ACTIVE, STM_COMMITTED);
	  } else {
	    stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
	  }

	  if (stmd -> status == STM_COMMITTED)
	    {
	      if (do_debug) check_consistency (stmd);
	    }

	  /* Release ownership and continue */
	  release_ownership_nb (st, stmd);
	}
      
      st -> current_transaction = NULL;
      END_TIMING (st, TIMING_COMMIT);
      
      if ((stmd -> status) == STM_COMMITTED) {
	succeeded = TRUE;
      }

      if (can_deallocate (stmd)) {
	MMReleaseObj (st, 1, stmd);
	stm_assert (stmd -> cd == FALSE);
	MMDeallocateObj (st, stmd);
      } else {
	MMReleaseObj (st, 1, stmd);
      }
      
      return succeeded;
    }
}


int STMCommitTransaction (STM_ENV *env)
{
  int result;
#ifdef DO_STM_ABORT_OTHER
  result = STMCommitTransactionNB (env);
#else
  result = STMCommitTransactionB (env);
#endif
  if (result) {
    COUNT(GET_ST(env), COUNT_COMMIT_SUCCEEDS); 
  } else {
    COUNT(GET_ST(env), COUNT_COMMIT_FAILS); 
  }
  return result;
}

void STMWait (STM_ENV *env)
{
#ifdef DO_STM_ABORT_OTHER
  STMWaitNB (env);
#else
  STMWaitB (env);
#endif
  COUNT(GET_ST(env), COUNT_WAITS);
}

/*......................................................................*/

void STMAbortTransaction (STM_ENV *env)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;

  st = GET_ST (env);
  stmd = st -> current_transaction;
  COUNT(GET_ST(env), COUNT_ABORTS);

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd != NULL) 
    {
      if (OPT_DEBUG_MSG) fprintf (stderr, "%p aborting", stmd);
      
      /*
       * To abort a transaction we just update its status field:
       * descriptors are only made public during a commit operation.
       */
      
      START_TIMING (TIMING_ABORT);
      stm_set_status (stmd, STM_ACTIVE, STM_ABORTED);
      stm_assert (stmd -> status == STM_ABORTED);
      st -> current_transaction = NULL;
      stm_assert (stmd -> cd == TRUE);
      MMDeallocateObj (st, stmd);
      END_TIMING (st, TIMING_ABORT);
    }
}

/*......................................................................*/

word_t STMReadValue (STM_ENV *env, addr_t addr)
{
  union STMVersionInfo_t vi_before, vi_after;
  STMState_t      *st;
  STMDescriptor_t *stmd;
  word_t           result;
  OwnershipRecord_t *orec;
  int              version;
  int              wanted;
  int              i;
  int              n;
int ocd;
void *oa1;
void *oa2;
 int ost;

  st = GET_ST (env);
  START_TIMING (TIMING_READ);
  stmd = st -> current_transaction;
  wanted = -1;

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd == NULL) 
    {
      /* 
       * We don't want to polute the space of the return values, so
       * give a plausible looking result and expose the abort when
       * the transaction tries to commit.
       */

      result = * ((word_t *) addr);
    }
 else
   {
     /* Have we read it already? */
     n = stmd -> length;
     for (i = 0; i < n; i ++)
       {
	 if (OREC_OF(stmd -> entries[i].addr) == OREC_OF (addr))
	   {
	     wanted = stmd -> entries[i].old_version;
	     if (stmd -> entries [i].addr == addr)
	       {
		 result = stmd -> entries[i].new_val;
		 goto done_read;
	       }
	   }
       }
     
     /* Not yet read it: fetch it from memory */
     orec = OREC_OF(addr);
MB();
     do
       {

       retry_read_location:
MB();
	 vi_before = orec -> u;
	 MB();
MB();
	 if (VI_IS_VERSION (vi_before)) {
	   /* Ordinary read of version vi_before */
	   MB();
MB();
	 result = (word_t) *addr;
MB();
	   version = vi_before.version;
	   MB();
	 } else {
	   /* Read from an owned location */
	   STMDescriptor_t *owner = vi_before.owner;
	   void *og;
	   bool_t committed;
	   int ost;

	   og = MMGuardObj (st, 0, owner);
	   stm_assert (og == NULL);
	   MB();
	   if (orec -> u.owner != owner) {
	   MB();
	     MMReleaseObj (st, 0, owner);
	     goto retry_read_location;
	   }

MB();
           result = (word_t) *addr; 
MB();

	   MB();
	   ocd = owner -> cd;
	   ost = owner -> status;
           oa1 = (void *) (owner -> entries[0].addr);
           oa2 = (void *) (owner -> entries[1].addr);

	   MB();

	   version = -1;
	   committed = (owner -> status == STM_COMMITTED);
	   for (i = 0; i < owner -> length; i ++)
	     {
	       if (OREC_OF (owner -> entries[i].addr) == orec)
		 {
		   STMEntry_t *entry;
		   entry = &(owner -> entries[i]);

		   /* Found correct ownership record */
		   version = committed ? entry -> new_version : entry -> old_version;
		   if (owner -> entries[i].addr == addr)
		     {
		       /* Found exactly matching address */
		       result = committed ? entry -> new_val : entry -> old_val;
		       break;
		     }
		 }
	     }
MB();

	   stm_assert (version != -1);
	   stm_assert ((version & 1) == 1);
MB();

	   MMReleaseObj (st, 0, owner);
	 } 
	 MB();
MB();
	 vi_after = orec -> u;
       }
     while (vi_before.version != vi_after.version);

MB();
     /* Record value and version in descriptor */
     i = (stmd -> length) ++;
     if (i == MAX_LOCATIONS) {
       stm_panic (("Exceeded %d locations", MAX_LOCATIONS));
     }

     stm_assert ((version & 1) == 1);
     if (wanted != -1 && version != wanted) 
       {
	 /* 
	  * We are now doomed to fail: we would have two entries
	  * with conflicting old version numbers.  
	  */
	 stm_assert (stmd -> cd == TRUE);
	 MMDeallocateObj (st, stmd);
	 st -> current_transaction = NULL;
       }
     else
       {
	 stmd -> entries[i].addr = addr;
	 stmd -> entries[i].old_val = result;
	 stmd -> entries[i].new_val = result;
	 stmd -> entries[i].old_version = version;
	 stmd -> entries[i].new_version = version;
	 stmd -> entries[i].waiting_with = NULL;
	 stmd -> entries[i].does_own = FALSE;
       }
   }
     
 done_read:
  END_TIMING (st, TIMING_READ);
  return result;
}

/*......................................................................*/

void STMWriteValue (STM_ENV *env, addr_t addr, word_t val)
{
  STMState_t      *st;
  STMDescriptor_t *stmd;
  int              i;
  int old_version;
  int new_version;

  STMReadValue (env, addr);

  st = GET_ST (env);
  stmd = st -> current_transaction;

  /* stmd == NULL  <=>  GC occurred since transaction started */
  if (stmd != NULL) 
    {
      /* Perform a read as a hack to make sure its in our descriptor */
      
      START_TIMING (TIMING_WRITE);
      /* Update it in the descriptor */
      for (i = 0; i < stmd -> length; i ++)
	{
	  if (stmd -> entries[i].addr == addr)
	    {
	      old_version = stmd -> entries[i].old_version;
	      new_version = old_version + 2;
	      stmd -> entries[i].new_val = val;
	      stmd -> entries[i].new_version = new_version;
	      stm_assert (((stmd -> entries[i].new_version) & 1) == 1);
	      stm_assert (((stmd -> entries[i].old_version) & 1) == 1);
	      goto done_write;
	    }
	}

      stm_panic (("Write at %p not find amongst %d\n", addr, stmd -> length));
      
    done_write:
      for (i = 0; i < stmd -> length; i++)
	{
	  if (OREC_OF(stmd -> entries[i].addr) == OREC_OF(addr))
	    {
	      //	      stm_assert (stmd -> entries[i].old_version == old_version);
	      stmd -> entries[i].new_version = new_version;
	    }
	}
      END_TIMING (st, TIMING_WRITE);
    }
  MB();

  return;
}

/***********************************************************************/

/* Simple spin-lock to protect tctr, free_state, first_state*/

static volatile int state_management_lock = 0;

/* Used to assign unique idx values to STMState_t structs */
static int tctr = 0;

/* Head of a list of free state structures (st -> env == NULL) */
static STMState_t *free_state = NULL;

/* Head of a list of active state structures (st -> env != NULL) */
static STMState_t *first_state = NULL;

static volatile int active_threads = 0;

static void lock_state_management (void) {
  while (CASInt (&state_management_lock, 0, 1) != 0) {
    /* Nothing */
  }
}

static void unlock_state_management (void) {
  stm_assert (state_management_lock == 1);
  state_management_lock = 0;
}

void STMAddThread (STM_ENV *env) {
  STMState_t *st;

  /* Obtain state structure */
  st = NULL;
  lock_state_management ();
  if (free_state != NULL) {
    /* Re-use free state */
    st = free_state;
    free_state = st -> next_state;

    st -> next_state = first_state;
    first_state = st;

  } else {
    /* Allocate new structure and idx value */
    st = mallocBytes (sizeof (STMState_t));
    stm_assert(st != NULL);
    st -> idx = tctr;
    st -> tn = 0;
    tctr ++;

    if (st -> idx >= MAX_THREADS) {
      stm_panic (("Cannot handle more than %d threads", MAX_THREADS));
    }

    /* Initialize structure */
    st -> next_state = first_state;
    first_state = st;

    /* Inform memory management */
    MMInitState (st);
  }
  active_threads ++;
  unlock_state_management ();

  st -> current_transaction = NULL;

  /* Assign structure to env */
  SET_ST(env, st);
  st -> env = env;
  stm_assert (GET_ST (env) == st);

#ifdef DO_TIMINGS
  {
    int i;
    for (i = 0; i < NUM_TIMINGS; i ++)
      {
	st -> timings[i] = 0;
      }
    st -> init_time = RDTICK ();
  }
#endif

#ifdef DO_COUNTS
  {
    int i;
    for (i = 0; i < NUM_COUNTS; i ++)
      {
	st -> counts[i] = 0;
      }
  }
#endif

#ifdef DO_ALLOCATION_COUNT
  st -> allocation_count = 0;
#endif

  st -> object_cache_1_size = 0;
  st -> object_cache_2_size = 0;
}

/*......................................................................*/

static void dump_status (STMState_t *st)
{

#ifdef DO_TIMINGS
  {
    int i;
    tick_t tot = RDTICK() - (st -> init_time);
    
    fprintf(stderr, "%p ", st);
    for (i = 0; i < NUM_TIMINGS; i ++)
      {
	fprintf (stderr, "%2.2f ", ((float) (st -> timings[i]) / tot) * 100);
      }
  }
#endif

#ifdef DO_COUNTS
  {
    int i;
    fprintf(stderr,"(");
    for (i = 0;i < NUM_COUNTS; i ++)
      {
	if (i != 0) fprintf (stderr, " ");
	fprintf (stderr, "%d", st -> counts[i]);
      }
    fprintf(stderr,") ");
  }
#endif

#ifdef DO_ALLOCATION_COUNT
  {
    float n = st -> allocation_count;
    float n2 = (st -> object_cache_1_size) + (st -> object_cache_2_size);
    char *u = "bytes";
    char *u2 = "bytes";
    if (n > 1024) {
      n /= 1024;
      u = "Kbytes";
    }
    if (n > 1024) {
      n /= 1024;
      u = "Mbytes";
    }
    if (n > 1024) {
      n /= 1024;
      u = "Gbytes";
    }
    if (n2 > 1024) {
      n2 /= 1024;
      u2 = "Kbytes";
    }
    if (n2 > 1024) {
      n2 /= 1024;
      u2 = "Mbytes";
    }
    if (n2 > 1024) {
      n2 /= 1024;
      u2 = "Gbytes";
    }

    fprintf (stderr, "(Allocated %2.2f %s,  Keeping %2.2f %s)", n, u, n2, u2);
  }
#endif

#if (defined(DO_ALLOCATION_COUNT) || defined (DO_TIMINGS))
  fprintf (stderr, "\n");
#endif
}

void snapshot (STM_ENV *env)
{
  fprintf (stderr, "\n");
  dump_status (GET_ST(env));
}

void STMRemoveThread (STM_ENV *env) {
  STMState_t *st;
  STMState_t **tmp;

  st = GET_ST(env);
  stm_assert (st != NULL);
  lock_state_management ();
  active_threads --;

  dump_status(st);


#ifdef DO_DUMP_ORECS
  {
    if (active_threads == 0)
      {
	int i;
	fprintf (stderr, "Versions: ");
	for (i = 0; i < STM_OWNERSHIP_RECORDS; i ++)
	  {
	    fprintf (stderr, "%d ", orecs[i].u.version);
	  }
	fprintf (stderr, "\n");
      }
  }
#endif

  /* Remove from list of active structures */
  tmp = &first_state;
  while (*tmp != st) {
    tmp = &((*tmp) -> next_state);
    stm_assert (tmp != NULL);
  }
  *tmp = st -> next_state;

  /* Add to list of free structures */
  st -> next_state = free_state;
  free_state = st;
  unlock_state_management();

  /* Remove ownership from env */
  SET_ST (env, NULL);
  st -> env = NULL;
}

/*......................................................................*/

/* This is called before the GC attempts to suspend mutators */
void STMPreGCHook (void)
{
  if (OPT_NEED_PRE_GC_HOOK) {
    STMState_t *st;
    
    st = first_state;
    while (st != NULL) {
      STMDescriptor_t *stmd;
      stmd = st -> current_transaction;

      if (stmd != NULL)
	{
	  st -> current_transaction = NULL;
	}

      st = st -> next_state;
    }
  }
}

void STMPostGCHook (void)
{
  if (OPT_NEED_POST_GC_HOOK) {
    STMState_t *st;

    need_gc = FALSE;

    /* 
     * Iterate over active states 
     */

    st = first_state;
    while (st != NULL) {
      MMDoneGC (st);
      st = st -> next_state;
    }

    /* 
     * Iterate over free states -- e.g. PTB may still have values
     * in a handoff entry.
     */

    st = free_state;
    while (st != NULL) {
      MMDoneGC (st);
      st = st -> next_state;
    }
  }
}

/*......................................................................*/

void STMInit (void) 
{
  int i;

  /* 
   * Run-time sanity checks about properties we assume about the
   * layout of data structures.
   */

  stm_assert (sizeof (void *) == sizeof (uintptr_t));

#ifdef DO_SECONDARY_PTB
  /*
   * 2. The PTB handoff array should be 8-byte aligned (assuming,
   *    in turn, that that's the width a CASPtrInt acts on and that
   *    the hardware requires the alignment).
   */
  stm_assert ((((uintptr_t) &(handoff[0])) & 7) == 0);
#endif

  shared_oc_1 = mallocBytes (sizeof (OCBalanceInfo_t));
  shared_oc_2 = mallocBytes (sizeof (OCBalanceInfo_t));
  shared_oc_1 -> shared_oc = NULL;
  shared_oc_2 -> shared_oc = NULL;
  shared_oc_1 -> size = 0;
  shared_oc_2 -> size = 0;

#ifdef DO_BALANCE_OBJECT_CACHES
  /* 
   * 3. The object cache info should be 8-byte aligned as above
   */
  stm_assert ((((uintptr_t) (shared_oc_1)) & 7) == 0);
  stm_assert ((((uintptr_t) (shared_oc_2)) & 7) == 0);
#endif

  /*
   * Initialization
   */

  MMInitScheme ();
  orecs = mallocBytes (STM_OWNERSHIP_RECORDS * sizeof (OwnershipRecord_t));
  stm_assert ((((int)orecs) & 7) == 0);
  for (i = 0; i < STM_OWNERSHIP_RECORDS; i ++)
    {
      orecs[i].u.version = 1;
#ifdef DO_STM_ABORT_OTHER
      orecs[i].updaters = 0;
#endif
    }
}

void STMDone (void)
{
  /* To do */
}

/*......................................................................*/

long long STMTestRDTick(void)
{
  return RDTICK();
}

int STMTestCASInt (int volatile *a, int o, int n)
{
  return CASInt (a, o, n);
}

